/*
 * Copyright (c) 2004 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corporation. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC SOFTWARE MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY
 * OF THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. PROGRESS SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 *
 * @author Rajiv Kanaujia
 *
 * Revision History
 * Date         Author          Desc
 * ----         ------          ----
 * 2002May14    Rajiv           Defined the interface
 *
 */
package com.sonicsw.security.pass.mf;

import java.util.HashMap;

import com.sonicsw.security.pass.client.IPasswordUser;


/**
 * The Management SPI is used to retrieve the user and group from any 
 * third party security store. Implementing this SPI results in viewing the 
 * read-only users and groups present on the external security store.
 * <p>
 * The management SPI can be categorized into two:
 * <p>
 * <b>Core Interfaces</b>
 * <p>
 * <code>com.sonicsw.security.pass.mf.IManagement</code>
 * <p>
 * <code>com.sonicsw.security.pass.client.IPasswordUser</code>
 * <p>
 * <code>com.sonicsw.security.pass.mf.IGroup</code>
 * <p>
 * <p>
 * <b>Additional Interfaces</b>
 * <p>
 * <code>com.sonicsw.security.pass.mf.IEvent</code>
 * <p>
 * <code>com.sonicsw.security.pass.mf.IEventDisconnected</code>
 * <p>
 * <code>com.sonicsw.security.pass.mf.IEventGroupsAdded</code>
 * <p>
 * <code>com.sonicsw.security.pass.mf.IEventGroupsDeleted</code>
 * <p>
 * <code>com.sonicsw.security.pass.mf.IEventGroupsModified</code>
 * <p>
 * <code>com.sonicsw.security.pass.mf.IEventUsersAdded</code>
 * <p>
 * <code>com.sonicsw.security.pass.mf.IEventUsersDeleted</code>
 * <p>
 * <code>com.sonicsw.security.pass.mf.IEventUsersModified</code>
 * <p>
 * <code>com.sonicsw.security.pass.mf.INotificationListener</code>
 * <p>
 * 
 * The basic functionality is achieved by implementing the core interfaces.
 * <p>
 * <b>Configuring Management SPI & using it in external Authentication Domain</b>
 * <p>
 * To use this functionality, users have to create a Management SPI configuration, 
 * which contains the Management SPI name, its classname and its classpath 
 * using Sonic Management Console (SMC).
 * <p>
 * While configuring an external Authentication Domain, users have to select 
 * a preconfigured Management SPI name and provide attributes which are to be 
 * added as name-value pairs using the Attribute Setting group box. 
 * These attributes are passed to the SPI implementation as a HashMap.
 * <p>
 * @author Rajiv Kanaujia
 * <p>
 */
public interface IManagement
{
    /**
     * This method is used to connect to an external store. This method can be 
     * called more than once. Examples are:
     * <p>
     * connecting after the disconnect() method was called
     * <p>
     * or
     * <p>
     * the IManagement object was disconnected because the connection to 
     * external source failed 
     * <p>
     * or
     * IManagement received an IEvenetDisconnected event
     * <p>
     *
     * @param configuration This is a name and value pair of configuration
     * attributes that is required by the implementation and configured via SMC.
     * <p>
     * @throws InvalidConfigurationException thrown if the connection failed because of bad connection
     * @throws ConnectionException thrown if connect() fails.
     */
    public void connect(HashMap configuration) throws InvalidConfigurationException, ConnectionException;


    /**
     * This is used to disconnect the connection to the external data store
     */
    public void disconnect();

    
    /**
     * This method is used to retrieve a group that matches the id.
     * <p>
     * @param id id of a group
     * <p>
     * @return the group matching the id.
     * <p>
     * @throws ConnectionException
     */
    public IGroup getGroup(String id)  throws ConnectionException;


    /**
     * This method is used to retrieve a list of groups that exist in the 
     * external security store.
     * <p>
     * @return list of groups
     * <p>
     * @throws ConnectionException
     */
    public IGroup[] getGroups() throws ConnectionException;


    /**
     * This method is returns a user that matches the id
     * <p>
     * @param id id of a user
     * <p>
     * @return the users credentials
     * <p>
     * @throws ConnectionException
     */
    public IPasswordUser getUser(String id) throws ConnectionException;


    /**
     * This method returns the list of uses existing in the external domain.
     * <p>
     * @return list of users
     * <p>
     * @throws ConnectionException
     */
    public IPasswordUser[] getUsers() throws ConnectionException;


    /**
     * This method is used to register event notifications listeners. It is 
     * called before <code>connect(HashMap)</code> method is called.
     * If <code>disconnect()</code> is called for any reason, then the next 
     * <code>connect(HashMap)</code> call must restore listener - no additional 
     * <code>setNotificationListener</code> call is needed. Setting of a single 
     * listener must be implemented - a second attempt to call 
     * <code>setNotificationListener</code> may fail.
     * <p>
     * If this method is not be implemented, users should return 
     * <code>false</code>. Corollary, <code>true</code> is to be returned if 
     * <code>setNotificationListener</code> is implemented.
     * <p>
     * @param listener the listener
     * <p>
     * @return true if setNotificationListener is implemented.
     * <p>
     * @see #connect(HashMap)
     * @see #disconnect()
     */
    public boolean setNotificationListener(INotificationListener listener);


    /**
     * This method removes/unregisters the event notifications listeners.
     */
    public void deleteNotificationListener();

}