/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */

package modelobjects.framework;

/**
 *  This class is a customizable ViewValueConverter for converting Boolean
 *  model values to Strings and back again.
 *  The customization allows specification of what String to user for
 *  the value 'true', what value for 'false', and whether matching of
 *  Strings entered by the user will be case sensitive.
 */
public class BooleanToStringConverter implements ViewValueConverter
{
  /**
   *  Construct a BooleanToStringConverter with the default strings
   *  "true" and "false" and not case-sensitive.
   */
  public BooleanToStringConverter()
  {
    this("false", "true", false);
  }

  /**
   *  Construct a BooleanToStringConverter with the specified values for
   *  'true' and 'false' and not case-sensitive.
   *  @param falseString  the string value to use for 'false'
   *  @param trueString   the string value to use for 'true'
   */
  public BooleanToStringConverter(String falseString, String trueString)
  {
    this(falseString, trueString, false);
  }

  /**
   *  Construct a BooleanToStringConverter with the specified values for
   *  'true' and 'false' and the specified value for case-sensitive.
   *  @param falseString  the string value to use for 'false'
   *  @param trueString   the string value to use for 'true'
   *  @param caseSensitive  whether string comparisons are case-sensitive
   */
  public BooleanToStringConverter(String falseString, String trueString,
                                  boolean caseSensitive)
  {
    this.falseString   = falseString;
    this.trueString    = trueString;
    this.caseSensitive = caseSensitive;
  }

  /**
   *  Return the legal string values
   */
  @Override
public String[] getViewStringValues()
  {
    return(new String[] { falseString, trueString });
  }

  /**
   *  Return the model class (Boolean)
   */
  @Override
public Class getModelClass()
  {
    return(java.lang.Boolean.class);
  }

  /**
   *  Return the view class (String)
   */
  @Override
public Class getViewClass()
  {
    return(java.lang.String.class);
  }

  /**
   *  Return the view string to display corresponding to the value 'true'
   */
  public String getTrueString()
  {
    return(trueString);
  }

  /**
   *  Assign the view string to display corresponding to the value 'true'
   */
  public void setTrueString(String val)
  {
    trueString = val;
  }

  /**
   *  Return the view string to display corresponding to the value 'false'
   */
  public String getFalseString()
  {
    return(falseString);
  }

  /**
   *  Assign the view string to display corresponding to the value 'false'
   */
  public void setFalseString(String val)
  {
    falseString = val;
  }

  /**
   *  Return whether string comparisons are case-sensitive.
   */
  public boolean isCaseSensitive()
  {
    return(caseSensitive);
  }

  /**
   *  Assign whether string comparisons are case-sensitive.
   */
  public void setCaseSensitive(boolean val)
  {
    caseSensitive = val;
  }

  /**
   *  Convert a String view value to a Boolean model value according to
   *  the Strings designated to mean 'true' and 'false' and whether the
   *  comparison is case-sensitive.
   *  @exception ValueConversionException is thrown if the parameter does
   *    not match either the true-string or false-string.
   */
  @Override
public Object viewValueToModel(Object viewValue)
       throws ViewValueConversionException
  {
    if (viewValue instanceof String)
      {
        String val = (String)viewValue;

        if (caseSensitive)
          {
            if (val.equals(trueString))
            {
                return(Boolean.TRUE);
            }
            else if (val.equals(falseString))
            {
                return(Boolean.FALSE);
            }
          }
        else
          {
            if (val.equalsIgnoreCase(trueString))
            {
                return(Boolean.TRUE);
            }
            else if (val.equalsIgnoreCase(falseString))
            {
                return(Boolean.FALSE);
            }
          }
      }
    throw(new ViewValueConversionException("value not " + falseString +
                                           " or " + trueString));
  }

  /**
   *  Convert a Boolean model value to a String view value using the Strings
   *  designated to mean 'true' and 'false'.
   *  @exception ValueConversionException is thrown if the parameter is not
   *    an instance of Boolean.
   */
  @Override
public Object modelValueToView(Object modelValue)
       throws ViewValueConversionException
  {
    if (modelValue == null)
    {
        modelValue = Boolean.FALSE;
    }
    if (modelValue instanceof Boolean)
    {
        return(modelValue.equals(Boolean.TRUE) ? trueString : falseString);
    }
    throw new ViewValueConversionException("value not boolean");
  }

  /**
   *  Indicate that conversion from the view type (String) to the model
   *  type (Boolean) is supported by this ValueConverter.
   */
  @Override
public boolean isViewValueToModelSupported()
  {
    return(true);
  }

  ///
  ///  Representation
  ///

  private String  falseString;
  private String  trueString;
  private boolean caseSensitive;

}
