/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */

package modelobjects.framework;

import java.awt.Component;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyVetoException;
import java.beans.VetoableChangeListener;
import java.beans.VetoableChangeSupport;

import modelobjects.framework.model.ModelAspectId;
import modelobjects.framework.model.ModelDescriptor;
import modelobjects.framework.model.ModelObjectAdapter;
import modelobjects.framework.model.NoSuchAspectException;

/**
 *  An EditForm encapsulates a ModelEditMediator and the actual GUI
 *  implementation it's attached to.  It is assumed that the GUI is not
 *  shared by multiple ModelEditMediators.
 *
 *  An EditForm's users are encouraged to minimize their interactions
 *  with the EditForm's ModelEditMediator, and interact instead with
 *  the EditForm whenever possible.
 */
public class EditForm
  implements ModelEditListener, ModelEditFailureListener, PropertyChangeListener
{
  /**
   *  Construct an EditForm for the specified ModelDescriptor,
   *  form-component, and ModelEditMediator.
   */
  public EditForm(ModelDescriptor modelTypeDesc,
                  Component formComponent,
                  ModelEditMediator modelEditMediator)
  {
    this.modelTypeDesc     = modelTypeDesc;
    this.formComponent     = formComponent;
    this.modelEditMediator = modelEditMediator;

    propertyChangeSupport = new PropertyChangeSupport(this);
    vetoableChangeSupport = new VetoableChangeSupport(this);
    modelEditEventSupport = new ModelEditEventSupport(this);
    modelEditFailureSupport = new ModelEditFailureSupport(this);

    modelEditMediator.addPropertyChangeListener(this);
    modelEditMediator.addModelEditListener(this);
    modelEditMediator.addModelEditFailureListener(this);

    try {
      editModelObject(null);
    }
    catch (Exception e) {
      // that shouldn't happen!
    }
  }

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  basic property accessor methods
  ///
  /////////////////////////////////////////////////////////////////////////////

  /**
   *  Return this EditForm's ModelDescriptor.
   */
  public ModelDescriptor getModelDescriptor()
  {
    return(modelTypeDesc);
  }

  /**
   *  Return this EditForm's top-level form Component.
   */
  public Component getFormComponent()
  {
    return(formComponent);
  }

  /**
   *  Return this EditForm's ModelEditMediator.
   */
  public ModelEditMediator getModelEditMediator()
  {
    return(modelEditMediator);
  }

  /**
   *  Return the edit-type metadata-key from this EditForm's ModelDescriptor.
   */
  public Object getEditTypeKey()
  {
    return(modelTypeDesc.getEditTypeKey());
  }

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  Pass-through and relay methods that allow EditForm users to interact
  ///  just with the EditForm and not directly with the ModelEditMediator.
  ///
  /////////////////////////////////////////////////////////////////////////////

  /**
   *  Set the model to be viewed or edited as a non-new object.
   *
   *  Note: This method must not be synchronized because it fires events
   *  to VetoableChangeListeners and PropertyChangeListeners for the
   *  property named by ModelEditMediator.MODEL_OBJECT_PROPERTY.
   *
   *  @param model a ModelObjectAdapter for the application model object,
   *    which should be edited as an existing (i.e. non-new) object.
   *
   *  @throws PropertyVetoException if thrown by any of the registered
   *    VetoableChangeListeners
   *  @throws ViewValueConversionException if thrown by any of the
   *    ViewAspectAdapters
   *  @throws NoSuchAspectException if the model aspect name specified for
   *    a ViewAspectAdapter does not match a ModelAspectAdapter in the model
   */
  public final void editModelObject(ModelObjectAdapter modelObjectAdapter)
    throws PropertyVetoException, ViewValueConversionException,
           NoSuchAspectException
  {
    modelEditMediator.editModelObject(modelObjectAdapter);
  }

  /**
   *  Set the model to be viewed or edited as a new object.
   *
   *  Note: This method must not be synchronized because it fires events
   *  to VetoableChangeListeners and PropertyChangeListeners for the
   *  property named by ModelEditMediator.MODEL_OBJECT_PROPERTY.
   *
   *  @param model a ModelObjectAdapter for the application model object,
   *    which should be edited for for initialization as a new  object
   *
   *  @throws PropertyVetoException if thrown by any of the registered
   *    VetoableChangeListeners
   *  @throws ViewValueConversionException if thrown by any of the
   *    ViewAspectAdapters
   *  @throws NoSuchAspectException if the model aspect name specified for
   *    a ViewAspectAdapter does not match a ModelAspectAdapter in the model
   */
  public void editModelAsNewObject(ModelObjectAdapter modelObjectAdapter)
    throws PropertyVetoException, ViewValueConversionException,
           NoSuchAspectException
  {
    modelEditMediator.editModelAsNewObject(modelObjectAdapter);
  }

  /**
   *  Discard edit changes so that there are no pending changes.
   *  In the case of a non-new object, this is the same as reset,
   *  but in the case of a new object, the new object is discarded
   *  and replaced by null as the model.
   */
  public synchronized void discardEdit()
  {
    modelEditMediator.discardEdit();
  }

  /**
   *  Return the ModelObjectAdapter being edited.
   */
  public ModelObjectAdapter getModel()
  {
    return(modelEditMediator.getModel());
  }

  /**
   *  Return whether the current model is being edited as a new object.
   */
  public boolean getEditModelAsNewObject()
  {
    return(modelEditMediator.getEditModelAsNewObject());
  }

  /**
   *  Return whether there are any unapplied changes in this EditForm.
   */
  public boolean hasUnappliedChanges()
  {
    return(modelEditMediator.hasUnappliedChanges());
  }

  /**
   *  Recompute editability on all ViewAspectAdapters.  This is not needed
   *  in most cases, but is necessary in cases where the editability of a
   *  field depends on view-values or other state that can change dynamically.
   */
  public void recomputeEditability()
    throws NoSuchAspectException
  {
    modelEditMediator.recomputeEditability();
  }

  /**
   *  Return the current model-aspect-value for the specified ModelAspectId.
   *  The value returned is provided directly by the ViewValueConverter, so
   *  it represents what's on the screen rather than what's stored in the
   *  model object being edited.
   */
  public Object getCurrentModelAspectValue(ModelAspectId modelAspectId)
    throws ViewValueConversionException, NoSuchAspectException
  {
    return(modelEditMediator.getCurrentModelAspectValue(modelAspectId));
  }

  /**
   *  Return the current view-aspect-value for the specified ModelAspectId.
   *  The value returned is provided directly by the ViewValueConverter, so
   *  it represents what's on the screen rather than what's stored in the
   *  model object being edited.
   */
  public Object getCurrentViewAspectValue(ModelAspectId modelAspectId)
    throws ViewValueConversionException, NoSuchAspectException
  {
    return(modelEditMediator.getCurrentViewAspectValue(modelAspectId));
  }

  /**
   *  Forward PropertyChangeEvents from the ModelEditMediator to this
   *  EditForm's PropertyChangeListeners, particularly so that they can
   *  handle changes to the ModelEditMediator.HAS_UNAPPLIED_CHANGES_PROPERTY
   *  property.
   */
  @Override
public void propertyChange(PropertyChangeEvent propChangeEvent)
  {
    propertyChangeSupport.firePropertyChange(propChangeEvent);
  }

  /**
   *  Forward VetoableChangeEvents from the ModelEditMediator to this
   *  EditForm's PropertyChangeListeners, particularly so that they can
   *  handle changes to the ModelEditMediator.MODEL_OBJECT_PROPERTY
   *  property to handle unapplied changes.
   */
  public void vetoableChange(PropertyChangeEvent vetoableChangeEvent)
    throws PropertyVetoException
  {
    vetoableChangeSupport.fireVetoableChange(vetoableChangeEvent);
  }

  /**
   *  Forward ModelEditEvents from the ModelEditMediator to this EditForm's
   *  ModelEditListeners.
   */
  @Override
public void modelEdited(ModelEditEvent event)
  {
    modelEditEventSupport.fireModelEdited(event);
  }

  /**
   *  Forward ModelEditEvents from the ModelEditMediator to this EditForm's
   *  ModelEditListeners.
   */
  @Override
public void modelCreated(ModelEditEvent event)
  {
    modelEditEventSupport.fireModelCreated(event);
  }

  /**
   *  Forward ModelEditEvents from the ModelEditMediator to this EditForm's
   *  ModelEditListeners.
   */
  @Override
public void modelDeleted(ModelEditEvent event)
  {
    modelEditEventSupport.fireModelDeleted(event);
  }

  /**
   *  Forward ModelEditFailureEvents from the ModelEditMediator to
   *  this EditForm's ModelEditFailureListeners.
   */
  @Override
public void modelEditFailed(ModelEditFailureEvent event)
  {
    modelEditFailureSupport.fireModelEditFailure(event);
  }


  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  Listener management methods
  ///
  /////////////////////////////////////////////////////////////////////////////

  /**
   *  Register a PropertyChangeListener.
   *
   *  @param listener a PropertyChangeListener
   */
  public void addPropertyChangeListener(PropertyChangeListener listener)
  {
    propertyChangeSupport.addPropertyChangeListener(listener);
  }

  /**
   *  Unregister a PropertyChangeListener.
   *
   *  @param listener a PropertyChangeListener
   */
  public void removePropertyChangeListener(PropertyChangeListener listener)
  {
    propertyChangeSupport.addPropertyChangeListener(listener);
  }

  /**
   *  Register a VetoableChangeListener.
   *
   *  @param listener a VetoableChangeListener
   */
  public void addVetoableChangeListener(VetoableChangeListener listener)
  {
    vetoableChangeSupport.addVetoableChangeListener(listener);
  }

  /**
   *  Unregister a VetoableChangeListener.
   *
   *  @param listener a VetoableChangeListener
   */
  public void removeVetoableChangeListener(VetoableChangeListener listener)
  {
    vetoableChangeSupport.removeVetoableChangeListener(listener);
  }

  /**
   *  Register a ModelEditListener.
   */
  public void addModelEditListener(ModelEditListener listener)
  {
    modelEditEventSupport.addModelEditListener(listener);
  }

  /**
   *  Unregister a ModelEditListener.
   */
  public void removeModelEditListener(ModelEditListener listener)
  {
    modelEditEventSupport.removeModelEditListener(listener);
  }

  /**
   *  Register a ModelEditFailureListener.
   */
  public void addModelEditFailureListener(ModelEditFailureListener listener)
  {
    modelEditFailureSupport.addModelEditFailureListener(listener);
  }

  /**
   *  Unregister a ModelEditFailureListener.
   */
  public void removeModelEditFailureListener(ModelEditFailureListener listener)
  {
    modelEditFailureSupport.removeModelEditFailureListener(listener);
  }


  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  private representation
  ///
  /////////////////////////////////////////////////////////////////////////////

  /** the ModelDescriptor for the kind of object editable in this EditForm */
  private ModelDescriptor       modelTypeDesc;

  /** the top-level GUI Component (usually a Container) for this EditForm */
  private Component             formComponent;

  /** the ModelEditMediator used to manage editing in this EditForm */
  private ModelEditMediator     modelEditMediator;

  /** support for PropertyChangeListeners */
  private PropertyChangeSupport propertyChangeSupport;

  /** support for VetoableChangeListeners */
  private VetoableChangeSupport vetoableChangeSupport;

  /** support for ModelEditListeners */
  private ModelEditEventSupport modelEditEventSupport;

  /** support for ModelEditFailureListeners */
  private ModelEditFailureSupport modelEditFailureSupport;

}
