/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */

package modelobjects.framework;

import java.awt.event.ActionEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyVetoException;
import java.beans.VetoableChangeListener;
import java.beans.VetoableChangeSupport;
import java.util.HashMap;

import javax.swing.AbstractAction;
import javax.swing.Action;

import modelobjects.framework.model.ModelAspectId;
import modelobjects.framework.model.ModelObjectAdapter;
import modelobjects.framework.model.NoSuchAspectException;
import modelobjects.util.swing.RelayAction;

/**
 *  An EditFormManager manages multiple alternative EditForms and presents
 *  them nearly as if there was only one.
 */
public class EditFormManager
  implements PropertyChangeListener, VetoableChangeListener,
             ModelEditListener, ModelEditFailureListener
{
  /**
   *  Construct an EditFormManager using the specified ModelObjectFactory,
   *  which also specifies the Class of model-objects that can be edited.
   */
  public EditFormManager(ModelObjectFactory modelObjectFactory)
  {
    this.modelObjectFactory = modelObjectFactory;
    this.modelObjectType    = modelObjectFactory.getModelObjectType();

    propertyChangeSupport   = new PropertyChangeSupport(this);
    vetoableChangeSupport   = new VetoableChangeSupport(this);
    modelEditEventSupport   = new ModelEditEventSupport(this);
    modelEditFailureSupport = new ModelEditFailureSupport(this);

    editFormMap             = new HashMap();

    applyAction             = new RelayAction();
    resetAction             = new RelayAction();
    undoAction              = new RelayAction();
    redoAction              = new RelayAction();
    newInstanceAction       = new NewInstanceAction();
  }

  /**
   *  Add an EditForm to the collection of managed EditForms.
   */
  public void addEditForm(EditForm editForm)
  {
    if (editForm.getModelDescriptor().getModelType() != modelObjectType) {
      String message = "EditForm's model type does not match EditFormManager's";
      throw(new IllegalArgumentException(message));
    }
    editFormMap.put(editForm.getEditTypeKey(), editForm);

    // make the first registered form the active one
    if (activeEditForm == null)
    {
        activateEditForm(editForm);
    }
  }

  /**
   *  Remove an EditForm from the collection of managed EditForms.
   */
  public void removeEditForm(EditForm editForm)
  {
    if (editForm == activeEditForm)
    {
        activateEditForm(null);
    }

    editFormMap.remove(editForm.getEditTypeKey());
  }

  /**
   *  Return the edit-type metadata key appropriate for the specified model
   *  object.  The default implementation just returns the model-object-class
   *  provided to the EditFormManager's constructor.
   *
   *  It is necessary to override this method as appropriate if the
   *  EditFormManager is used to manage multiple forms with different
   *  edit-type keys, or if the model-object Class is not the correct
   *  edit-type key.
   */
  public Object getEditTypeKey(Object modelObject)
  {
    return(modelObjectType);
  }

  /**
   *  Report an error thrown during normal edit processing.
   *  The default implementation just call printStackTrace().
   *  Applications should override this method to report errors
   *  appropriately.
   */
  protected void reportError(Exception error)
  {
    error.printStackTrace();
  }

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  basic property accessor methods
  ///
  /////////////////////////////////////////////////////////////////////////////

  /**
   *  Return the ModelObjectFactory used to create new instances for editing.
   */
  public ModelObjectFactory getModelObjectFactory()
  {
    return(modelObjectFactory);
  }

  /**
   *  Return this EditForm's model-object-type
   */
  public Class getModelObjectType()
  {
    return(modelObjectType);
  }

  /**
   *  Get the currently active EditForm, if any.
   */
  public EditForm getActiveEditForm()
  {
    return(activeEditForm);
  }

  /**
   *  Get the managed EditForm corresponding to the specified edit-type
   *  metadata key.
   */
  public EditForm getEditForm(Object editTypeKey)
  {
    return((EditForm)editFormMap.get(editTypeKey));
  }

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  Pass-through and relay methods that allow EditForm users to interact
  ///  just with the EditForm and not directly with the ModelEditMediator.
  ///
  /////////////////////////////////////////////////////////////////////////////

  /**
   *  Set the model to be viewed or edited as a non-new object, and return
   *  the EditForm it's being edited in.
   *
   *  Note: This method must not be synchronized because it fires events
   *  to VetoableChangeListeners and PropertyChangeListeners for the
   *  property named by ModelEditMediator.MODEL_OBJECT_PROPERTY.
   *
   *  @param modelObject the application model (not a ModelObjectAdapter)
   *    that should be edited as an existing (i.e. non-new) object.
   *  @param editTypeKey the class metadata object that identifies the
   *    ModelDescriptor for the object and the appropriate EditForm to use.
   *
   *  @throws PropertyVetoException if thrown by any of the registered
   *    VetoableChangeListeners
   *  @throws ViewValueConversionException if thrown by any of the
   *    ViewAspectAdapters
   *  @throws NoSuchAspectException if the model aspect name specified for
   *    a ViewAspectAdapter does not match a ModelAspectAdapter in the model
   */
  public EditForm editModelObject(Object modelObject, Object editTypeKey)
    throws PropertyVetoException, ViewValueConversionException,
           NoSuchAspectException
  {
    EditForm editForm = getEditForm(editTypeKey);
    if (editForm == null) {
      String msg = "no EditForm registered with edit-type key " + editTypeKey;
      throw(new IllegalArgumentException(msg));
    }

    activateEditForm(editForm);

    ModelObjectAdapter modelObjectAdapter = (modelObject == null) ? null :
      editForm.getModelDescriptor().makeModelObjectAdapter(modelObject);
    editForm.editModelObject(modelObjectAdapter);
    return(editForm);
  }

  /**
   *  Set the model to be viewed or edited as a new object, and return the
   *  EditForm it's being edited in.
   *
   *  Note: This method must not be synchronized because it fires events
   *  to VetoableChangeListeners and PropertyChangeListeners for the
   *  property named by ModelEditMediator.MODEL_OBJECT_PROPERTY.
   *
   *  @param modelObject the application model (not a ModelObjectAdapter)
   *    that should be edited for initialization as a new object.
   *  @param editTypeKey the class metadata object that identifies the
   *    ModelDescriptor for the object and the appropriate EditForm to use.
   *
   *  @throws PropertyVetoException if thrown by any of the registered
   *    VetoableChangeListeners
   *  @throws ViewValueConversionException if thrown by any of the
   *    ViewAspectAdapters
   *  @throws NoSuchAspectException if the model aspect name specified for
   *    a ViewAspectAdapter does not match a ModelAspectAdapter in the model
   */
  public EditForm editModelAsNewObject(Object modelObject, Object editTypeKey)
    throws PropertyVetoException, ViewValueConversionException,
           NoSuchAspectException
  {
    EditForm editForm = getEditForm(editTypeKey);
    if (editForm == null) {
      String msg = "no EditForm registered with edit-type key " + editTypeKey;
      throw(new IllegalArgumentException(msg));
    }

    activateEditForm(editForm);

    ModelObjectAdapter modelObjectAdapter = (modelObject == null) ? null :
      editForm.getModelDescriptor().makeModelObjectAdapter(modelObject);
    editForm.editModelAsNewObject(modelObjectAdapter);
    return(editForm);
  }

  /**
   *  Create and start editing a new instance of this EditFormManager's
   *  model class.  This method makes use of the ModelObjectFactory to
   *  create the new instance, and the editTypeKey(Object) method to get
   *  the edit-type-key for selecting the appropriate EditForm.
   *
   *  @throws InstantiationException - if the model Class represents an
   *    abstract class, an interface, an array class, a primitive type,
   *    or void; or if the instantiation fails for some other reason.
   *  @throws IllegalAccessException - if the model class or initializer is
   *    not accessible.
   *  @throws ExceptionInInitializerError - if the instance initialization
   *    provoked by this method fails.
   *  @throws PropertyVetoException if thrown by any of the registered
   *    VetoableChangeListeners
   *  @throws ViewValueConversionException if thrown by any of the
   *    ViewAspectAdapters
   *  @throws NoSuchAspectException if the model aspect name specified for
   *    a ViewAspectAdapter does not match a ModelAspectAdapter in the model
   */
  public EditForm createAndEditNewInstance()
    throws InstantiationException, IllegalAccessException,
           ExceptionInInitializerError, PropertyVetoException,
           ViewValueConversionException, NoSuchAspectException
  {
    Object newInstance = modelObjectFactory.createNewInstance();

    if (newInstance != null) {
      Object editTypeKey = getEditTypeKey(newInstance);
      return(editModelAsNewObject(newInstance, editTypeKey));
    }
    else {
      // factory did not return a new object, so don't change anything
      return(activeEditForm);
    }
  }

  /**
   *  Discard edit changes so that there are no pending changes.
   *  In the case of a non-new object, this is the same as reset,
   *  but in the case of a new object, the new object is discarded
   *  and replaced by null as the model.
   */
  public synchronized void discardEdit()
  {
    if (activeEditForm != null)
    {
        activeEditForm.discardEdit();
    }
  }

  /**
   *  Return the ModelObjectAdapter of the model being edited, or null if
   *  there is no active form or no model being edited.
   */
  public ModelObjectAdapter getModelObjectAdapter()
  {
    if (activeEditForm == null)
    {
        return(null);
    }
    else
    {
        return(activeEditForm.getModel());
    }
  }

  /**
   *  Return whether the current model is being edited as a new object.
   */
  public boolean getEditModelAsNewObject()
  {
    if (activeEditForm == null)
    {
        return(false);
    }
    else
    {
        return(activeEditForm.getEditModelAsNewObject());
    }
  }

  /**
   *  Return whether there are any unapplied changes in this EditForm.
   */
  public boolean hasUnappliedChanges()
  {
    if (activeEditForm == null)
    {
        return(false);
    }
    else
    {
        return(activeEditForm.hasUnappliedChanges());
    }
  }

  /**
   *  Recompute editability on all ViewAspectAdapters.  This is not needed
   *  in most cases, but is necessary in cases where the editability of a
   *  field depends on view-values or other state that can change dynamically.
   */
  public void recomputeEditability()
    throws NoSuchAspectException
  {
    if (activeEditForm != null)
    {
        activeEditForm.recomputeEditability();
    }
  }

  /**
   *  Return the current model-aspect-value for the specified ModelAspectId.
   *  The value returned is provided directly by the ViewValueConverter, so
   *  it represents what's on the screen rather than what's stored in the
   *  model object being edited.
   */
  public Object getCurrentModelAspectValue(ModelAspectId modelAspectId)
    throws ViewValueConversionException, NoSuchAspectException
  {
    if (activeEditForm == null)
    {
        return(null);
    }
    else
    {
        return(activeEditForm.getCurrentModelAspectValue(modelAspectId));
    }
  }

  /**
   *  Return the current view-aspect-value for the specified ModelAspectId.
   *  The value returned is provided directly by the ViewValueConverter, so
   *  it represents what's on the screen rather than what's stored in the
   *  model object being edited.
   */
  public Object getCurrentViewAspectValue(ModelAspectId modelAspectId)
    throws ViewValueConversionException, NoSuchAspectException
  {
    if (activeEditForm == null)
    {
        return(null);
    }
    else
    {
        return(activeEditForm.getCurrentViewAspectValue(modelAspectId));
    }
  }


  /**
   *  Forward PropertyChangeEvents from the ModelEditMediator to this
   *  EditForm's PropertyChangeListeners, particularly so that they can
   *  handle changes to the ModelEditMediator.HAS_UNAPPLIED_CHANGES_PROPERTY
   *  property.
   */
  @Override
public void propertyChange(PropertyChangeEvent propertyChangeEvent)
  {
    String propName = propertyChangeEvent.getPropertyName();

    if (propName.equals(ModelEditMediator.HAS_UNAPPLIED_CHANGES_PROPERTY)) {
      newInstanceAction.setEnabled((activeEditForm != null) &&
                                   !activeEditForm.hasUnappliedChanges());
    }

    propertyChangeSupport.firePropertyChange
      (propName,
       propertyChangeEvent.getOldValue(),
       propertyChangeEvent.getNewValue());
  }

  /**
   *  Forward VetoableChangeEvents from the ModelEditMediator to this
   *  EditForm's PropertyChangeListeners, particularly so that they can
   *  handle changes to the ModelEditMediator.MODEL_OBJECT_PROPERTY
   *  property to handle unapplied changes.
   */
  @Override
public void vetoableChange(PropertyChangeEvent vetoableChangeEvent)
    throws PropertyVetoException
  {
    vetoableChangeSupport.fireVetoableChange
      (vetoableChangeEvent.getPropertyName(),
       vetoableChangeEvent.getOldValue(),
       vetoableChangeEvent.getNewValue());
  }

  /**
   *  Forward ModelEditEvents from the ModelEditMediator to this EditForm's
   *  ModelEditListeners.
   */
  @Override
public void modelEdited(ModelEditEvent event)
  {
    modelEditEventSupport.fireModelEdited(event);
  }

  /**
   *  Forward ModelEditEvents from the ModelEditMediator to this EditForm's
   *  ModelEditListeners.
   */
  @Override
public void modelCreated(ModelEditEvent event)
  {
    modelEditEventSupport.fireModelCreated(event);
  }

  /**
   *  Forward ModelEditEvents from the ModelEditMediator to this EditForm's
   *  ModelEditListeners.
   */
  @Override
public void modelDeleted(ModelEditEvent event)
  {
    modelEditEventSupport.fireModelDeleted(event);
  }

  /**
   *  Forward ModelEditFailureEvents from the ModelEditMediator to
   *  this EditForm's ModelEditFailureListeners.
   */
  @Override
public void modelEditFailed(ModelEditFailureEvent event)
  {
    modelEditFailureSupport.fireModelEditFailure(event);
  }

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  Action accessor methods
  ///
  /////////////////////////////////////////////////////////////////////////////

  /**
   *  Return the Action used to initiate apply behavior.
   */
  public Action getApplyAction()
  {
    return(applyAction);
  }

  /**
   *  Return the Action used to initiate reset behavior.
   */
  public Action getResetAction()
  {
    return(resetAction);
  }

  /**
   *  Return the Action used to initiate undo behavior.
   */
  public Action getUndoAction()
  {
    return(undoAction);
  }

  /**
   *  Return the Action used to initiate redo behavior.
   */
  public Action getRedoAction()
  {
    return(redoAction);
  }

  /**
   *  Return the Action used to create and edit new model instances.
   */
  public Action getNewInstanceAction()
  {
    return(newInstanceAction);
  }

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  Listener management methods
  ///
  /////////////////////////////////////////////////////////////////////////////

  /**
   *  Register a PropertyChangeListener.
   *
   *  @param listener a PropertyChangeListener
   */
  public void addPropertyChangeListener(PropertyChangeListener listener)
  {
    propertyChangeSupport.addPropertyChangeListener(listener);
  }

  /**
   *  Unregister a PropertyChangeListener.
   *
   *  @param listener a PropertyChangeListener
   */
  public void removePropertyChangeListener(PropertyChangeListener listener)
  {
    propertyChangeSupport.addPropertyChangeListener(listener);
  }

  /**
   *  Register a VetoableChangeListener.
   *
   *  @param listener a VetoableChangeListener
   */
  public void addVetoableChangeListener(VetoableChangeListener listener)
  {
    vetoableChangeSupport.addVetoableChangeListener(listener);
  }

  /**
   *  Unregister a VetoableChangeListener.
   *
   *  @param listener a VetoableChangeListener
   */
  public void removeVetoableChangeListener(VetoableChangeListener listener)
  {
    vetoableChangeSupport.removeVetoableChangeListener(listener);
  }

  /**
   *  Register a ModelEditListener.
   */
  public void addModelEditListener(ModelEditListener listener)
  {
    modelEditEventSupport.addModelEditListener(listener);
  }

  /**
   *  Unregister a ModelEditListener.
   */
  public void removeModelEditListener(ModelEditListener listener)
  {
    modelEditEventSupport.removeModelEditListener(listener);
  }

  /**
   *  Register a ModelEditFailureListener.
   */
  public void addModelEditFailureListener(ModelEditFailureListener listener)
  {
    modelEditFailureSupport.addModelEditFailureListener(listener);
  }

  /**
   *  Unregister a ModelEditFailureListener.
   */
  public void removeModelEditFailureListener(ModelEditFailureListener listener)
  {
    modelEditFailureSupport.removeModelEditFailureListener(listener);
  }


  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  private implementation
  ///
  /////////////////////////////////////////////////////////////////////////////

  private void activateEditForm(EditForm editForm)
  {
    if (editForm == activeEditForm)
    {
        return;
    }

    EditForm oldEditForm = activeEditForm;

    if (activeEditForm != null) {
      try {
        // if we're going to deactivate the form set a null model
        activeEditForm.editModelObject(null);
      }
      catch (Exception e) {
        // too bad, but we tried
      }

      activeEditForm.removePropertyChangeListener(this);
      activeEditForm.removeVetoableChangeListener(this);
      activeEditForm.removeModelEditListener(this);
      activeEditForm.removeModelEditFailureListener(this);
    }

    if (editForm != null) {
      editForm.addPropertyChangeListener(this);
      editForm.addVetoableChangeListener(this);
      editForm.addModelEditListener(this);
      editForm.addModelEditFailureListener(this);

      ModelEditMediator m = editForm.getModelEditMediator();
      applyAction.setTargetAction(m.getApplyAction());
      resetAction.setTargetAction(m.getResetAction());
      undoAction.setTargetAction(m.getUndoAction());
      redoAction.setTargetAction(m.getRedoAction());
    }
    else {
      applyAction.setTargetAction(null);
      resetAction.setTargetAction(null);
      undoAction.setTargetAction(null);
      redoAction.setTargetAction(null);
    }

    newInstanceAction.setEnabled(editForm != null);

    activeEditForm = editForm;

    propertyChangeSupport.firePropertyChange(ACTIVE_EDIT_FORM_PROPERTY,
                                             oldEditForm, activeEditForm);
  }

  class NewInstanceAction extends AbstractAction
  {
    public NewInstanceAction()
    {
      super("New");
    }
    
    private void putNameProperty() {
        if (modelObjectFactory.isCreationInteractive())
        {
            putValue(Action.NAME, "New...");
        }
    }

    @Override
    public void actionPerformed(ActionEvent event)
    {
      try {
        createAndEditNewInstance();
      }
      catch (Exception e) {
        reportError(e);
      }
    }
  }

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  private representation
  ///
  /////////////////////////////////////////////////////////////////////////////

  public  static final String   ACTIVE_EDIT_FORM_PROPERTY = "activeEditForm";

  /** the ModelObjectFactory used to create new instances for editing */
  private ModelObjectFactory    modelObjectFactory;

  /** the Class of object editable in this EditFormManager */
  private Class                 modelObjectType;

  /** the currently active EditForm */
  private EditForm              activeEditForm;

  /** a mapping from edit-type metadata key to EditForm */
  private HashMap               editFormMap;

  /** the RelayAction for initiating apply behavior */
  private RelayAction           applyAction;

  /** the RelayAction for initiating reset behavior */
  private RelayAction           resetAction;

  /** the RelayAction for initiating undo behavior */
  private RelayAction           undoAction;

  /** the RelayAction for initiating redo behavior */
  private RelayAction           redoAction;

  /** the Action used to create and edit new model instances. */
  private Action                newInstanceAction;

  /** support for PropertyChangeListeners */
  private PropertyChangeSupport propertyChangeSupport;

  /** support for VetoableChangeListeners */
  private VetoableChangeSupport vetoableChangeSupport;

  /** support for ModelEditListeners */
  private ModelEditEventSupport modelEditEventSupport;

  /** support for ModelEditFailureListeners */
  private ModelEditFailureSupport modelEditFailureSupport;

}
