package modelobjects.framework;

import java.text.DateFormat;
import java.text.MessageFormat;
import java.util.Date;

/**
 *  This class is a customizable ViewValueConverter for converting Date model
 *  values to Strings and back again.
 */
public class LongDateToStringConverter implements ViewValueConverter
{
    private static final int ITEM_DAYS    = 0;
    private static final int ITEM_HOURS   = 1;
    private static final int ITEM_MINUTES = 2;

    boolean    m_allowNull;
    boolean    m_uptime;
    DateFormat m_dateFormat;
    String     m_msgFormat;

    public LongDateToStringConverter()
    {
        this(true, DateFormat.getDateTimeInstance());
    }

    public LongDateToStringConverter(boolean allowNull, DateFormat dateFormat)
    {
        setNullAllowed(allowNull);
        setDateFormat(dateFormat);
    }

    /**
     * eg. "{0} days, {1} hours, {2} minutes";
     */
    public LongDateToStringConverter(boolean uptime, String msgFormat)
    {
        this(true, (DateFormat)null);
        setProperties(uptime, msgFormat);
    }

    private void setProperties(boolean uptime, String msgFormat) {
        setUptimeMode(uptime);
        setMsgFormat(msgFormat);
    }
    
    public final void setNullAllowed(boolean allowNull)
    {
        m_allowNull = allowNull;
    }

    public boolean isNullAllowed()
    {
        return m_allowNull;
    }

    public final void setDateFormat(DateFormat dateFormat)
    {
        m_dateFormat = dateFormat;
    }

    public DateFormat getDateFormat()
    {
        return m_dateFormat;
    }

    public void setMsgFormat(String msgFormat)
    {
        m_msgFormat = msgFormat;
    }

    public String getMsgFormat()
    {
        return m_msgFormat;
    }

    public void setUptimeMode(boolean uptime)
    {
        m_uptime = uptime;
    }

    public boolean isUptimeMode()
    {
        return m_uptime;
    }

    @Override
    public Object viewValueToModel(Object viewValue)
    throws ViewValueConversionException
    {
        Object res = null;

        if (viewValue instanceof String)
        {
            if (isNullAllowed() && (viewValue == null || viewValue.equals("")))
            {
                res = new Long(0);
            }
            else
            {
                String stringValue = (String)viewValue;

                res = new Long(0);
            }
        }
        else
        {
            throw new ViewValueConversionException("expected a string");
        }

        return res;
    }

    @Override
    public Object modelValueToView(Object modelValue)
    throws ViewValueConversionException
    {
        Object res = null;

        if (modelValue instanceof Long)
        {
            long longValue = ((Long)modelValue).longValue();

            if (longValue != 0)
            {
                if (isUptimeMode())
                {
                    Integer[] args = new Integer[3];
                    long  tmp  = longValue / 1000;
                    // We now have tmp as seconds
                    tmp /= 60;
                    // We now have tmp as minutes
                    args[ITEM_MINUTES] = new Integer((int)(tmp % 60));
                    tmp /= 60;
                    // We now have tmp as hours 
                    args[ITEM_HOURS] = new Integer((int) (tmp%24));
                    tmp /= 24;
                    // We now have tmp as days
                    args[ITEM_DAYS]    = new Integer((int)(tmp));

                    res = new MessageFormat(getMsgFormat()).format(args);
                }
                else
                {
                    res = getDateFormat().format(new Date(longValue));
                }
            }
        }
        else
        {
            throw new ViewValueConversionException(modelValue + " is not an acceptable type");
        }

        return res;
    }

    /**
     *  Returns an array of all the String values that may be used as view
     *  aspect values, or null if no such constraint applies.
     */
    @Override
    public String[] getViewStringValues()
    {
        return null;
    }

    /**
     *  Returns whether or not conversion from view values to model values
     *  is supported by the ViewValueConverter.  If not, ViewAspectAdapters
     *  using this ViewValueConverter cannot be edited.
     */
    @Override
    public boolean isViewValueToModelSupported()
    {
        return true;
    }

    /**
     *  Return the model type returned by viewValueToModel()
     */
    @Override
    public Class getModelClass()
    {
        return Long.class;
    }

    /**
     *  Return the view type returned by modelValueToView()
     */
    @Override
    public Class getViewClass()
    {
        return String.class;
    }

}
