/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */

package modelobjects.framework;

import java.awt.event.ActionEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyVetoException;
import java.beans.VetoableChangeListener;
import java.beans.VetoableChangeSupport;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.WeakHashMap;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.UndoableEditEvent;
import javax.swing.event.UndoableEditListener;
import javax.swing.undo.UndoManager;
import javax.swing.undo.UndoableEditSupport;

import modelobjects.framework.model.ModelAspectAdapter;
import modelobjects.framework.model.ModelAspectId;
import modelobjects.framework.model.ModelDescriptor;
import modelobjects.framework.model.ModelObjectAdapter;
import modelobjects.framework.model.ModelValidationException;
import modelobjects.framework.model.NoSuchAspectException;
import modelobjects.util.WrapperRuntimeException;
import modelobjects.util.undo.CannotDoException;

public class ModelEditMediator
  implements ChangeListener, UndoableEditListener
{
  /**
   *  Construct a ModelEditMediator
   */
  public ModelEditMediator()
  {
    applyAction = new ApplyAction();
    resetAction = new ResetAction();
    undoAction  = new UndoAction();
    redoAction  = new RedoAction();

    applyAction.setEnabled(false);
    resetAction.setEnabled(false);
    undoAction.setEnabled(false);
    redoAction.setEnabled(false);

    viewAspectAdapterMap  = new HashMap();
    propertyChangeSupport = new PropertyChangeSupport(this);
    vetoableChangeSupport = new VetoableChangeSupport(this);
    undoableEditSupport   = new UndoableEditSupport(this);
    modelEditEventSupport = new ModelEditEventSupport(this);
    modelEditFailureSupport = new ModelEditFailureSupport(this);

    aspectChanges         = new HashSet();
    allAspects            = new HashSet();

    savedUndoHistory      = new WeakHashMap();
    objectEdits           = new UndoManager();
    fieldEdits            = null;

    addUndoableEditListener(this);
  }

  /**
   *  Register a ViewAspectAdapter to be managed by the ModelEditroller.
   *  Note that there can only be one ViewAspectAdapter for a given
   *  ModelAspectId, and it is an error to add multiple ViewAspectAdapters
   *  that are associated with the same ModelAspectId.
   *
   *  @param viewAspectAdapter the ViewAspectAdapter to register
   */
  public synchronized void
    addViewAspectAdapter(ViewAspectAdapter viewAspectAdapter)
  {
    ModelAspectId modelAspectId = viewAspectAdapter.getModelAspectId();
    allAspects.add(modelAspectId);
    viewAspectAdapterMap.put(modelAspectId, viewAspectAdapter);
    viewAspectAdapter.addChangeListener(this);
    viewAspectAdapter.addUndoableEditListener(this);
  }

  /**
   *  Unregister a ViewAspectAdapter from the ModelEditroller
   *
   *  @param viewAspectAdapter the ViewAspectAdapter to unregister
   */
  public synchronized void
    removeViewAspectAdapter(ViewAspectAdapter viewAspectAdapter)
  {
    ModelAspectId modelAspectId = viewAspectAdapter.getModelAspectId();
    allAspects.remove(modelAspectId);
    viewAspectAdapterMap.remove(modelAspectId);
    viewAspectAdapter.removeChangeListener(this);
  }

  /**
   *  Get the registered ViewAspectAdapter with the specified model aspect id
   *
   *  @param modelAspectId the ModelAspectId of the ViewAspectAdapter
   */
  public ViewAspectAdapter getViewAspectAdapter(ModelAspectId modelAspectId)
  {
    return((ViewAspectAdapter)viewAspectAdapterMap.get(modelAspectId));
  }

  private ModelAspectAdapter getModelAspectAdapter(ModelAspectId aspectId)
  {
    ModelDescriptor      descriptor = modelObjectAdapter.getModelDescriptor();
    ModelAspectAdapter[] adapter    = descriptor.getModelAspectAdapters();

    for (int i = 0; i < adapter.length; i++)
    {
        if (adapter[i].getModelAspectId().equals(aspectId))
        {
            return adapter[i];
        }
    }

    return null;
  }

  /**
   *  Handle a ChangeEvent from one of the ViewAspectAdapters.
   *  The only thing we need to do at this point is to record that the
   *  aspect has changed, and to enable the apply and reset logic.
   *
   *  ChangeEvents fired while setting the model are discarded, since setting
   *  the state to reflect the unedited model does not represent an edit.
   */
  @Override
public void stateChanged(ChangeEvent event)
  {
    if (settingModel)
    {
        return;
    }

    // record the aspect change
    ViewAspectAdapter source = (ViewAspectAdapter)event.getSource();
    aspectChanges.add(source.getModelAspectId());

    // enable apply and reset behavior
    enableApplyAndReset(true);
  }

  /**
   *  Handle an UndoableEditEvent.
   */
  @Override
public void undoableEditHappened(UndoableEditEvent event)
  {
    //!! System.out.println("Got UndoableEdit " + event.getEdit());

    if (inUndoOrRedo)
    {
        return;
    }

    if (event.getSource() == this)
      {
        // discard all field edits on a setModel or Apply event
        discardFieldEdits();
        objectEdits.addEdit(event.getEdit());
      }
    else // edit is from one of the ViewAspectAdapters
      {
        if (fieldEdits == null)
          {
            //!! System.out.println("creating and adding fieldEdits");
            fieldEdits = new UndoManager();
          }
        fieldEdits.addEdit(event.getEdit());
      }

    undoAction.setEnabled(true);
  }

  /**
   *  Return the ModelObjectAdapter being edited.
   */
  public ModelObjectAdapter getModel()
  {
    return(modelObjectAdapter);
  }

  /**
   *  Return whether the current model is being edited as a new object.
   */
  public boolean getEditModelAsNewObject()
  {
    return(editModelAsNewObject);
  }

  /**
   *  Set the model to be viewed or edited as a non-new object.
   *
   *  Note: This method must not be synchronized because it fires events
   *  to VetoableChangeListeners and PropertyChangeListeners for the
   *  property named by ModelEditMediator.MODEL_OBJECT_PROPERTY.
   *
   *  @param model a ModelObjectAdapter for the application model object,
   *    which should be edited as an existing (i.e. non-new) object.
   *
   *  @throws PropertyVetoException if thrown by any of the registered
   *    VetoableChangeListeners
   *  @throws ViewValueConversionException if thrown by any of the
   *    ViewAspectAdapters
   *  @throws NoSuchAspectException if the model aspect name specified for
   *    a ViewAspectAdapter does not match a ModelAspectAdapter in the model
   */
  public void editModelObject(ModelObjectAdapter model)
    throws PropertyVetoException, ViewValueConversionException,
           NoSuchAspectException
  {
    editModel(model, false);
  }

  /**
   *  Set the model to be viewed or edited as a new object.
   *
   *  Note: This method must not be synchronized because it fires events
   *  to VetoableChangeListeners and PropertyChangeListeners for the
   *  property named by ModelEditMediator.MODEL_OBJECT_PROPERTY.
   *
   *  @param model a ModelObjectAdapter for the application model object,
   *    which should be edited for for initialization as a new  object
   *
   *  @throws PropertyVetoException if thrown by any of the registered
   *    VetoableChangeListeners
   *  @throws ViewValueConversionException if thrown by any of the
   *    ViewAspectAdapters
   *  @throws NoSuchAspectException if the model aspect name specified for
   *    a ViewAspectAdapter does not match a ModelAspectAdapter in the model
   */
  public void editModelAsNewObject(ModelObjectAdapter model)
    throws PropertyVetoException, ViewValueConversionException,
           NoSuchAspectException
  {
    editModel(model, true);
  }

  /**
   *  Notify VetoableChangeListeners about the proposed model change,
   *  change the instance state, and notify PropertyChangeListeners.
   *
   *  @param model the ModelObjectAdapter to be edited
   *  @param editAsNew indicates whether the model should be edited for
   *    for initialization as a new  object
   *
   *  @throws PropertyVetoException if thrown by any of the registered
   *    VetoableChangeListeners
   *  @throws ViewValueConversionException if thrown by any of the
   *    ViewAspectAdapters
   *  @throws NoSuchAspectException if the model aspect name specified for
   *    a ViewAspectAdapter does not match a ModelAspectAdapter in the model
   */
  private void editModel(ModelObjectAdapter newModel, boolean editAsNew)
    throws PropertyVetoException, ViewValueConversionException,
           NoSuchAspectException
  {
    ModelObjectAdapter oldModel = this.modelObjectAdapter;
    boolean oldModelWasNew = this.editModelAsNewObject;

    PropertyChangeEvent propChangeEvent =
      new PropertyChangeEvent(this, MODEL_OBJECT_PROPERTY,
                              oldModel, newModel);

    // notify VetoableChangeListeners
    vetoableChangeSupport.fireVetoableChange(propChangeEvent);

    // update the instance state and set the model in the ViewAspectAdapters
    setModelInternal(newModel, editAsNew);

    // either restore the object's previous undo-history or make a new one
    Object modelObject = (newModel == null) ? null : newModel.getModelObject();
    objectEdits = ((editAsNew || (modelObject == null)) ? null :
                   (UndoManager)savedUndoHistory.get(modelObject));
    if (objectEdits == null)
    {
        objectEdits = new UndoManager();
    }

    enableApplyAndReset(false);
    enableUndoAndRedo();

    // notify PropertyChangeListeners
    propertyChangeSupport.firePropertyChange(propChangeEvent);
  }

  /**
   *  Update the instance state for a new model, and update all the
   *  ViewAspectAdapters.
   *
   *  Deadlock Warning:  This method is synchronized and holds the
   *  ModelEditMediator lock while it updates the ViewAspectAdapters.
   *
   *  @param model the ModelObjectAdapter to be edited
   *  @param editAsNew indicates whether the model should be edited for
   *    for initialization as a new  object
   *
   *  @throws ViewValueConversionException if thrown by any of the
   *    ViewAspectAdapters
   *  @throws NoSuchAspectException if the model aspect name specified for
   *    a ViewAspectAdapter does not match a ModelAspectAdapter in the model
   */
  private synchronized void setModelInternal(ModelObjectAdapter model,
                                             boolean editAsNew)
    throws ViewValueConversionException, NoSuchAspectException
  {

    if(isValidating)
    {
        return;
    }
    try {

      settingModel = true;

      // set the instance state
      this.modelObjectAdapter = model;
      this.editModelAsNewObject = editAsNew;

      setViewFromModel(model);
    }
    finally {
      discardFieldEdits();
      enableApplyAndReset(false);
      settingModel = false;
    }
  }

  /**
   *  Set the values of all the ViewAspectAdapters from the model, and clear
   *  the set of changed aspects.
   */
  private synchronized void setViewFromModel(ModelObjectAdapter model)
    throws ViewValueConversionException, NoSuchAspectException
  {
    aspectChanges.clear();

    Iterator modelAspectIds = viewAspectAdapterMap.keySet().iterator();
    while (modelAspectIds.hasNext())
      {
        ModelAspectId modelAspectId = (ModelAspectId)modelAspectIds.next();
        ViewAspectAdapter viewAspectAdapter =
          (ViewAspectAdapter)viewAspectAdapterMap.get(modelAspectId);

        Object modelAspectValue =
          ((model == null) ? null : model.getAspectValue(modelAspectId));
        viewAspectAdapter.setModelAspectValue(modelAspectValue);

        // set editability according to whether the model aspect is readonly
        // and according to the ViewAspectAdapter's EditRule
        EditRule editRule = viewAspectAdapter.getEditRule();
        boolean editable =
          ((model != null) &&
           !model.isReadonlyAspect(modelAspectId) &&
           ((editRule == null) ||
            editRule.isEditableInEditor(viewAspectAdapter, this)));

        viewAspectAdapter.setEditable(editable);
      }
  }

  /**
   *  Recompute editability on all ViewAspectAdapters.  This is not needed
   *  in most cases, but is necessary in cases where the editability of a
   *  field depends on view-values or other state that can change dynamically.
   */
  public void recomputeEditability()
    throws NoSuchAspectException
  {
    ModelObjectAdapter model = getModel();
    Iterator modelAspectIds = viewAspectAdapterMap.keySet().iterator();
    while (modelAspectIds.hasNext())
      {
        ModelAspectId modelAspectId = (ModelAspectId)modelAspectIds.next();
        ViewAspectAdapter viewAspectAdapter =
          (ViewAspectAdapter)viewAspectAdapterMap.get(modelAspectId);

        // set editability according to whether the model aspect is readonly
        // and according to the ViewAspectAdapter's EditRule
        EditRule editRule = viewAspectAdapter.getEditRule();
        boolean editable =
          ((model != null) &&
           !model.isReadonlyAspect(modelAspectId) &&
           ((editRule == null) ||
            editRule.isEditableInEditor(viewAspectAdapter, this)));

        viewAspectAdapter.setEditable(editable);
      }
  }

  /**
   *  Return the current model-aspect-value for the specified ModelAspectId.
   *  The value returned is provided directly by the ViewValueConverter, so
   *  it represents what's on the screen rather than what's stored in the
   *  model object being edited.
   */
  public Object getCurrentModelAspectValue(ModelAspectId modelAspectId)
    throws ViewValueConversionException, NoSuchAspectException
  {
    ViewAspectAdapter viewAspectAdapter =
      (ViewAspectAdapter)viewAspectAdapterMap.get(modelAspectId);
    return(viewAspectAdapter.getModelAspectValue());
  }

  /**
   *  Return the current view-aspect-value for the specified ModelAspectId.
   *  The value returned is provided directly by the ViewValueConverter, so
   *  it represents what's on the screen rather than what's stored in the
   *  model object being edited.
   */
  public Object getCurrentViewAspectValue(ModelAspectId modelAspectId)
    throws ViewValueConversionException, NoSuchAspectException
  {
    ViewAspectAdapter viewAspectAdapter =
      (ViewAspectAdapter)viewAspectAdapterMap.get(modelAspectId);
    return(viewAspectAdapter.getViewAspectValue());
  }

  /**
   *  Return the Action used to initiate apply behavior.
   */
  public Action getApplyAction()
  {
    return(applyAction);
  }

  /**
   *  Return the Action used to initiate reset behavior.
   */
  public Action getResetAction()
  {
    return(resetAction);
  }

  /**
   *  Return the Action used to initiate undo behavior.
   */
  public Action getUndoAction()
  {
    return(undoAction);
  }

  /**
   *  Return the Action used to initiate redo behavior.
   */
  public Action getRedoAction()
  {
    return(redoAction);
  }

  /**
   *  Apply changes from managed ViewAspectAdapters to the edited model object.
   *  If an exception is thrown while changing the state of the model object
   *  or if the updated state fails validation, the model object is reverted
   *  to its state prior to the call.
   *  This method fires PropertyChangeEvents for each aspect change, first
   *  to its VetoableChangeListeners and then to its PropertyChangeListeners.
   *
   *  @throws NoSuchAspectException if any of the ViewAspectAdapters have
   *    ModelAspectIds that are not recognized by the ModelObjectAdapter
   *  @throws ViewValueConversionException if any of the ViewAspectAdapters
   *    are unable to convert their view values to model aspect values
   *  @throws PropertyVetoException if an aspect change is vetoed by a
   *    VetoableChangeListener
   *  @throws ModelValidationException if the new state of the model object,
   *    after the edits have been applied, is rejected.
   */
  public synchronized void apply()
    throws NoSuchAspectException, ViewValueConversionException,
           PropertyVetoException, CannotDoException, ModelValidationException
  {
    UndoableModelAspectEdit[] aspectEdits = null;

    // Collect all the modified values from the view and the saved values
    // from the model, packaged up as UndoableModelAspectEdits
    try
      {
        aspectEdits = collectAspectEdits();
      }
    catch (ViewValueConversionException badViewValue)
      {
        // TEMPORARY: report this here anyway, even if it might be handled
        //System.err.println("Bad view value: " + badViewValue);
        throw(badViewValue);
      }

    // create PropertyChangeEvents for each of the edited aspects
    PropertyChangeEvent[] propertyChangeEvents =
      getPropertyChangeEvents(aspectEdits);

    // notify VetoableChangeListeners about the changes
    for (int i = 0, n = propertyChangeEvents.length; i < n; i++)
    {
        vetoableChangeSupport.fireVetoableChange(propertyChangeEvents[i]);
    }

    // Use an UndoableModelEdit to collect all of the applied changes,
    // both for backing out partial changes on failure and for notifying
    // UndoableEditListeners of the whole thing on success.
    UndoableModelEdit compoundModelEdit =
      new UndoableModelEdit("modify model", modelObjectAdapter);

    // try to apply the changes to the model
    boolean succeeded = false;
    boolean wasNew = editModelAsNewObject;
    Object model = modelObjectAdapter.getModelObject();
    Throwable failureException = null;

    try
      {
        for (int i = 0, n = aspectEdits.length; i < n; i++)
          {
            UndoableModelAspectEdit aspectEdit = aspectEdits[i];
            aspectEdit.doEdit(false);
            compoundModelEdit.addEdit(aspectEdit);
          }

        // must disable apply and reset BEFORE firing ModelEditEvents
        // must also stop treating the model as NEW before firing events
        enableApplyAndReset(false);
        editModelAsNewObject = false;

        //isValidating field is used in the set
        isValidating = true;

        if (wasNew)
          {
            modelObjectAdapter.validateNewModel();
            modelObjectAdapter.propagateModelCreation();
            modelEditEventSupport.fireModelCreated(ModelEditEvent.APPLY, model);
          }
        else
          {
            modelObjectAdapter.validateModel();
            modelObjectAdapter.propagateModelEdit();
            modelEditEventSupport.fireModelEdited(ModelEditEvent.APPLY, model);
          }

        succeeded = true;
      }
    catch (Throwable e)
      {
        failureException = e;
      }
    finally
    {
        isValidating = false;
        // close off the compound edit for batch undo and redo
        compoundModelEdit.end();

    }
    
    if (!succeeded)
    {
        // TBD: is it necessary to fire PropertyChangeEvents here to
        // notify listeners that we're setting the state back again?

        editModelAsNewObject = wasNew;
        enableApplyAndReset(true);

        //            if (!wasNew) {      // no need to undo changes to new object
        try {
            //System.out.println("Apply failed - undoing partial changes");
            compoundModelEdit.undo();
        }
        catch (Exception errorInUndo) 
        {
                // now what!!??
            System.err.println("ERROR: Cannot undo failed apply: " + errorInUndo);
        }
        //            }
    }
    else
    {
        //!! System.out.println("Apply succeeded - firing off final events.");

        // Reset the state of all the ViewAspectAdapters, but note that the
        // object is not new anymore.
        setModelInternal(modelObjectAdapter, false);

        // Notify PropertyChangeListeners.
        for (int i = 0, n = propertyChangeEvents.length; i < n; i++)
        {
            propertyChangeSupport.firePropertyChange(propertyChangeEvents[i]);
        }

        // Notify UndoableEditListeners, except when applying changes to
        // a formerly new object.
        if (!wasNew)
        {
            undoableEditSupport.postEdit(compoundModelEdit);
        }

        // save the object's undo history for next time
        savedUndoHistory.put(model, objectEdits);
    }

    if (failureException != null)
    {
        if (failureException instanceof NoSuchAspectException)
        {
            throw((NoSuchAspectException)failureException);
        }
        else if (failureException instanceof PropertyVetoException)
        {
            throw((PropertyVetoException)failureException);
        }
        else if (failureException instanceof CannotDoException)
        {
            throw((CannotDoException)failureException);
        }
        else if (failureException instanceof ModelValidationException)
        {
            throw((ModelValidationException)failureException);
        }
        else
        {
            throw(new ModelValidationException(failureException));
        }
    }
  }

  /**
   *  Discard tentative changes from the managed ViewAspectAdapters.
   */
  public synchronized void reset()
    throws ViewValueConversionException, NoSuchAspectException
  {
    setModelInternal(modelObjectAdapter, editModelAsNewObject);
    enableApplyAndReset(false);
  }

  /**
   *  Discard edit changes so that there are no pending changes.
   *  In the case of a non-new object, this is the same as reset,
   *  but in the case of a new object, the new object is discarded
   *  and replaced by null as the model.
   */
  public synchronized void discardEdit()
  {
    try {
      if (editModelAsNewObject)
    {
        setModelInternal(null, false);
    }
    else
    {
        setModelInternal(modelObjectAdapter, editModelAsNewObject);
    }

      enableApplyAndReset(false);
    }
    catch (Exception e) {
      throw(new WrapperRuntimeException(e));
    }
  }

  /**
   *  Collect UndoableModelAspectEdits for all of the aspect changes.
   *
   *  If we're editing a "new" object, also collect edits from all
   *  editable ViewAspectAdapters even if they haven't fired change
   *  events, since those values should be used for initialization too.
   */
  private UndoableModelAspectEdit[] collectAspectEdits()
    throws ViewValueConversionException
  {
    int numChangedAspects = (editModelAsNewObject ? allAspects.size() :
                             aspectChanges.size());
    HashSet aspectIds = (editModelAsNewObject ? allAspects : aspectChanges);

    ArrayList collector = new ArrayList(numChangedAspects);
    Iterator aspects = aspectIds.iterator();

    for (int i = 0; aspects.hasNext(); i++)
      {
        ModelAspectId aspectId = (ModelAspectId)aspects.next();
        ViewAspectAdapter  viewAspectAdapter  = getViewAspectAdapter(aspectId);
        ModelAspectAdapter modelAspectAdapter = getModelAspectAdapter(aspectId);

        if ((viewAspectAdapter.isEditable() || aspectChanges.contains(aspectId)) &&
            !modelAspectAdapter.isReadonlyAspect())
          {
            UndoableModelAspectEdit aspectEdit =
              viewAspectAdapter.makeUndoableModelAspectEdit(modelObjectAdapter);
            if (aspectEdit != null)
            {
                collector.add(aspectEdit);
            }
          }
      }

    UndoableModelAspectEdit[] edits =
      new UndoableModelAspectEdit[collector.size()];
    collector.toArray(edits);

    return(edits);
  }

  /**
   *  Collect PropertyChangeEvents for all the UndoableModelAspectEdits
   */
  private PropertyChangeEvent[]
    getPropertyChangeEvents(UndoableModelAspectEdit[] aspectEdits)
  {
    int numEdits = aspectEdits.length;
    PropertyChangeEvent[] result = new PropertyChangeEvent[numEdits];

    for (int i = 0; i < numEdits; i++)
    {
        result[i] = aspectEdits[i].getPropertyChangeEvent();
    }

    return(result);
  }

  /**
   *  Register a PropertyChangeListener.
   *
   *  @param listener a PropertyChangeListener
   */
  public void addPropertyChangeListener(PropertyChangeListener listener)
  {
    propertyChangeSupport.addPropertyChangeListener(listener);
  }

  /**
   *  Unregister a PropertyChangeListener.
   *
   *  @param listener a PropertyChangeListener
   */
  public void removePropertyChangeListener(PropertyChangeListener listener)
  {
    propertyChangeSupport.removePropertyChangeListener(listener);
  }

  /**
   *  Register a VetoableChangeListener.
   *
   *  @param listener a VetoableChangeListener
   */
  public void addVetoableChangeListener(VetoableChangeListener listener)
  {
    vetoableChangeSupport.addVetoableChangeListener(listener);
  }

  /**
   *  Unregister a VetoableChangeListener.
   *
   *  @param listener a VetoableChangeListener
   */
  public void removeVetoableChangeListener(VetoableChangeListener listener)
  {
    vetoableChangeSupport.removeVetoableChangeListener(listener);
  }

  /**
   *  Register an UndoableEditListener.
   *
   *  @param listener an UndoableEditListener
   */
  public final void addUndoableEditListener(UndoableEditListener listener)
  {
    undoableEditSupport.addUndoableEditListener(listener);
  }

  /**
   *  Unregister an UndoableEditListener.
   *
   *  @param listener an UndoableEditListener
   */
  public void removeUndoableEditListener(UndoableEditListener listener)
  {
    undoableEditSupport.removeUndoableEditListener(listener);
  }

  /**
   *  Register a ModelEditListener.
   */
  public void addModelEditListener(ModelEditListener listener)
  {
    modelEditEventSupport.addModelEditListener(listener);
  }

  /**
   *  Unregister a ModelEditListener.
   */
  public void removeModelEditListener(ModelEditListener listener)
  {
    modelEditEventSupport.removeModelEditListener(listener);
  }

  /**
   *  Register a ModelEditFailureListener.
   */
  public void addModelEditFailureListener(ModelEditFailureListener listener)
  {
    modelEditFailureSupport.addModelEditFailureListener(listener);
  }

  /**
   *  Unregister a ModelEditFailureListener.
   */
  public void removeModelEditFailureListener(ModelEditFailureListener listener)
  {
    modelEditFailureSupport.removeModelEditFailureListener(listener);
  }

  /**
   *  Discard all UndoableEdits from ViewAspectAdapters.
   */
  private void discardFieldEdits()
  {
    if (fieldEdits != null)
      {
        //!! System.out.println("discarding fieldEdits");
        fieldEdits.end();
        fieldEdits = null;
      }
  }

  private void enableUndoAndRedo()
  {
    boolean canUndo = (objectEdits.canUndo() ||
                       ((fieldEdits != null) && fieldEdits.canUndo()));
    boolean canRedo = (objectEdits.canRedo() ||
                       ((fieldEdits != null) && fieldEdits.canRedo()));
    undoAction.setEnabled(canUndo);
    redoAction.setEnabled(canRedo);
  }

  private void enableApplyAndReset(boolean enabled)
  {
    boolean wasEnabled  = applyAction.isEnabled();
    boolean enableApply = (enabled || editModelAsNewObject);

    applyAction.setEnabled(enableApply);
    resetAction.setEnabled(enabled);

    // notify PropertyChangeListeners
    if (wasEnabled != enableApply) {
      PropertyChangeEvent propChangeEvent =
        new PropertyChangeEvent(this, HAS_UNAPPLIED_CHANGES_PROPERTY,
                                new Boolean(wasEnabled),
                                new Boolean(enableApply));
      propertyChangeSupport.firePropertyChange(propChangeEvent);
    }
  }

  /**
   *  Return whether there are any unapplied changes in this ModelEditMediator.
   */
  public boolean hasUnappliedChanges()
  {
    return(resetAction.isEnabled()); // ||
//           (getEditModelAsNewObject() && applyAction.isEnabled()));
  }


  ///////////////////////////////////////////////////////////////////////////
  ///
  ///    Representation
  ///
  ///////////////////////////////////////////////////////////////////////////

  /** The name of the modelObject property used in PropertyChangeEvents */
  public final static String    MODEL_OBJECT_PROPERTY   = "modelObject";

  /** The name of the unappliedChanges property used in PropertyChangeEvents */
  public final static String    HAS_UNAPPLIED_CHANGES_PROPERTY =
                                                        "hasUnappliedChanges";

  /** the Action for initiating apply behavior */
  private Action                applyAction;

  /** the Action for initiating reset behavior */
  private Action                resetAction;

  /** the Action for initiating undo behavior */
  private Action                undoAction;

  /** the Action for initiating redo behavior */
  private Action                redoAction;

  /** a map of model aspect names to ViewAspectAdapters */
  private HashMap               viewAspectAdapterMap;

  /** the currently edited ModelObjectAdapter */
  private ModelObjectAdapter    modelObjectAdapter;

  /** manages UndoableEdits for undo and redo */
  private UndoManager           objectEdits;

  /** manages UndoableEdits from ViewAspectAdapters for undo and redo */
  private UndoManager           fieldEdits;

  /** maintains undo-history for previously edited objects */
  private WeakHashMap           savedUndoHistory;

  /** indicates whether the model is edited as a new object */
  protected boolean               editModelAsNewObject;

  /** support for PropertyChangeListeners */
  private PropertyChangeSupport propertyChangeSupport;

  /** support for VetoableChangeListeners */
  private VetoableChangeSupport vetoableChangeSupport;

  /** support for UndoableEditListeners */
  private UndoableEditSupport   undoableEditSupport;

  /** support for ModelEditListeners */
  private ModelEditEventSupport modelEditEventSupport;

  /** support for ModelEditFailureListeners */
  private ModelEditFailureSupport modelEditFailureSupport;

  /** a Set of the ModelAspectIds for aspects that have changed */
  protected HashSet               aspectChanges;

  /** a Set of the ModelAspectIds for all ViewAspectAdapters */
  protected HashSet               allAspects;

  /** indicates if we're setting a model and should disregard ChangeEvents */
  protected transient boolean     settingModel;

  /** indicates if we're in the middle of processing undo or redo */
  protected transient boolean     inUndoOrRedo;

  /**
   * This field is introduced to fix the bug# Sonic00035561
   * This was introduced to avoid the problem
   * When all the rows are deleted and a new row is created it would fail by throwing an exception.
   *
   * It indicates if we are int eh middle of applying changes to the model.
   * */
  private boolean isValidating;


  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  Inner classes for Action definitions
  ///
  /////////////////////////////////////////////////////////////////////////////

  private class ApplyAction extends AbstractAction
  {
    ApplyAction()
    {
      super("Apply");
    }
    @Override
    public void actionPerformed(ActionEvent event)
    {
      try {
        apply();
      }
      catch (Exception exception) {
        //System.err.println("Error in apply: " + exception);
        Object model = modelObjectAdapter.getModelObject();
        modelEditFailureSupport.fireModelEditFailure(model, exception);
      }
      enableUndoAndRedo();
    }
  }

  private class ResetAction extends AbstractAction
  {
    ResetAction()
    {
      super("Reset");
    }
    @Override
    public void actionPerformed(ActionEvent event)
    {
      try {
        reset();
      }
      catch (Exception e) {
        System.err.println("Error in reset: " + e);
        e.printStackTrace();
      }

      enableUndoAndRedo();
    }
  }

  private class UndoAction extends AbstractAction
  {
    UndoAction()
    {
      super("Undo");
    }
    @Override
    public void actionPerformed(ActionEvent event)
    {
      try {
        inUndoOrRedo = true;

        if ((fieldEdits != null) && fieldEdits.canUndo()) {
          fieldEdits.undo();
        }
        else if (objectEdits.canUndo()) {
          objectEdits.undo();
          reset();
          modelObjectAdapter.validateModel();
          modelObjectAdapter.propagateModelEdit();
          Object model = modelObjectAdapter.getModelObject();
          modelEditEventSupport.fireModelEdited(ModelEditEvent.UNDO, model);
        }
        else {
          System.out.println("ERROR: cannot really undo anything!");
        }

        enableUndoAndRedo();
      }
      catch (Exception e) {
        Object model = modelObjectAdapter.getModelObject();
        modelEditFailureSupport.fireModelEditFailure(model, e);
        System.err.println("Error in undo: " + e);
//          e.printStackTrace();
      }
      finally {
        inUndoOrRedo = false;
      }
    }
  }

  private class RedoAction extends AbstractAction
  {
    RedoAction()
    {
      super("Redo");
    }
    @Override
    public void actionPerformed(ActionEvent event)
    {
      try {
        inUndoOrRedo = true;

        if (objectEdits.canRedo()) {
          objectEdits.redo();
          reset();
          modelObjectAdapter.validateModel();
          modelObjectAdapter.propagateModelEdit();
          Object model = modelObjectAdapter.getModelObject();
          modelEditEventSupport.fireModelEdited(ModelEditEvent.REDO, model);
        }
        else if ((fieldEdits != null) && fieldEdits.canRedo()) {
          fieldEdits.redo();
        }
        else {
          System.out.println("ERROR: cannot really redo anything!");
        }

        enableUndoAndRedo();
      }
      catch (Exception e) {
        Object model = modelObjectAdapter.getModelObject();
        modelEditFailureSupport.fireModelEditFailure(model, e);
        System.err.println("Error in redo: " + e);
//          e.printStackTrace();
      }
      finally {
        inUndoOrRedo = false;
      }
    }
  }

}
