/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */

package modelobjects.framework;

import java.lang.reflect.Constructor;

/**
 *  This class is a customizable ViewValueConverter for converting Number
 *  model values to Strings and back again.
 */
public class NumberToStringConverter implements ViewValueConverter
{
  protected boolean m_allowNull;
  protected Class   m_modelClass;


  public NumberToStringConverter(Class modelClass)
  {
    this(modelClass, true);
  }

  public NumberToStringConverter(Class modelClass, boolean allowNull)
  {
    m_modelClass = modelClass;
    m_allowNull  = allowNull;
  }

  @Override
public Object viewValueToModel(Object viewValue)
      throws ViewValueConversionException
  {
      if (viewValue instanceof String)
      {
          if (m_allowNull && ((viewValue == null) || viewValue.equals("")))
        {
            return null;
        }

          try
          {
              String sval = (String)viewValue;

              Constructor c = m_modelClass.getConstructor(new Class[] { String.class });

              return c.newInstance(new Object[] { viewValue });
          }
          catch (Exception e)
          {
              throw new ViewValueConversionException("Failed to convert <" + viewValue + "> to a " + m_modelClass.getName());
          }
      }

      // Not a valid type!
      //
      String viewType = (viewValue != null) ? viewValue.getClass().getName() : "null";

      throw new ViewValueConversionException(viewType + " is not an allowed type - expecting "
                                                      + getViewClass().getName()
                                                      + " (" + viewValue + ")");
  }

  @Override
public Object modelValueToView(Object modelValue)
       throws ViewValueConversionException
  {
    if (modelValue == null)
    {
        return "";
    }

    /*
     * A bit lazy here but it seems ok...not checking that the modelValue is
     * the correct type. We're just converting it to a string.
     */
    return modelValue.toString();
  }

  /**
   *  Returns an array of all the String values that may be used as view
   *  aspect values, or null if no such constraint applies.
   */
  @Override
public String[] getViewStringValues()
  {
      return null;
  }

  /**
   *  Returns whether or not conversion from view values to model values
   *  is supported by the ViewValueConverter.  If not, ViewAspectAdapters
   *  using this ViewValueConverter cannot be edited.
   */
  @Override
public boolean isViewValueToModelSupported()
  {
      return true;
  }

  /**
   *  Return the model type returned by viewValueToModel()
   */
  @Override
public Class getModelClass()
  {
      return m_modelClass;
  }

  /**
   *  Return the view type returned by modelValueToView()
   */
  @Override
public Class getViewClass()
  {
      return String.class;
  }

}
