/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */

package modelobjects.framework;

import java.math.BigDecimal;

/**
 *  This class is a customizable ViewValueConverter for converting Number
 *  model values to Strings and back again.
 */
public class PercentageToBigDecimalConverter implements ViewValueConverter
{
  protected BigDecimalToStringConverter m_converter;
  protected int                         m_scale;
  protected boolean                     m_allowNull;


  public PercentageToBigDecimalConverter(BigDecimalToStringConverter converter, int scale)
  {
    this(converter, scale, true);
  }

  public PercentageToBigDecimalConverter(BigDecimalToStringConverter converter, int scale, boolean allowNull)
  {
    m_converter = converter;
    m_scale     = scale;
    m_allowNull = allowNull;
  }

  @Override
public Object viewValueToModel(Object viewValue)
      throws ViewValueConversionException
  {
      BigDecimal value = (BigDecimal)m_converter.viewValueToModel(viewValue);

      if (m_allowNull && (value == null))
    {
        return null;
    }

// Old method:
//      BigDecimal newValue = value.divide(new BigDecimal(100.00), m_scale + 2, BigDecimal.ROUND_HALF_UP);
//      return newValue;

      // We don't want to enforce a maximum number of digits to the right of
      // the decimal point...we want scale to be used as a maximum...but we
      // DO want trailing zero's to be removed, i.e. 0.80 should be 0.8
      //
      double scaleFactor = Math.pow(10, m_scale);  // Used for shifting scale in and out of fractional part
      double tmp = value.doubleValue() / 100;      // Convert from percentage to fractional part
      tmp = tmp * scaleFactor;                     // Multiple out fractional part
      tmp = (double)Math.round(tmp) / scaleFactor; // Remove (round) fractional part and remove scale factor

      return new BigDecimal(Double.toString(tmp));
  }

  @Override
public Object modelValueToView(Object modelValue)
       throws ViewValueConversionException
  {
    if (modelValue == null)
    {
        return "";
    }

    if (modelValue instanceof BigDecimal)
    {
        BigDecimal value = (BigDecimal)modelValue;

        BigDecimal newValue = value.multiply(new BigDecimal(100.00));

        newValue = newValue.setScale(m_scale, BigDecimal.ROUND_HALF_UP);

        return m_converter.modelValueToView(newValue);
    }

    throw new ViewValueConversionException("Invalid modelValue type: expecting BigDecimal!");
  }

  /**
   *  Returns an array of all the String values that may be used as view
   *  aspect values, or null if no such constraint applies.
   */
  @Override
public String[] getViewStringValues()
  {
      return null;
  }

  /**
   *  Returns whether or not conversion from view values to model values
   *  is supported by the ViewValueConverter.  If not, ViewAspectAdapters
   *  using this ViewValueConverter cannot be edited.
   */
  @Override
public boolean isViewValueToModelSupported()
  {
      return true;
  }

  /**
   *  Return the model type returned by viewValueToModel()
   */
  @Override
public Class getModelClass()
  {
      return BigDecimal.class;
  }

  /**
   *  Return the view type returned by modelValueToView()
   */
  @Override
public Class getViewClass()
  {
      return String.class;
  }

}
