/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */

package modelobjects.framework;

import java.awt.Component;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyEditor;
import java.beans.PropertyEditorManager;

import javax.swing.JComboBox;
import javax.swing.JTextField;

import modelobjects.framework.model.ModelAspectId;
import modelobjects.framework.swing.JComboBoxAdapter;
import modelobjects.framework.swing.JTextFieldAdapter;

/**
 *  PropertyEditorViewAspectAdapter is a kind of ViewAspectAdapter
 *  that is based on a JavaBeans PropertyEditor.  It will use either
 *  the PropertyEditor's custom-editor Component if there is one, or
 *  a JComboBox if there's a choice of legal String values, or else
 *  just a JTextField with String conversion based on whatever the
 *  PropertyEditor provides for converting to and from Strings.
 */
public class PropertyEditorViewAspectAdapter extends ViewAspectAdapter
  implements PropertyChangeListener
{
  public static ViewAspectAdapter
    makeViewAspectAdapter(ModelAspectId modelAspectId,
                          EditRule editRule,
                          PropertyEditor propertyEditor,
                          ModelEditMediator mediator)
  {
    Component comp = (propertyEditor.supportsCustomEditor() ?
                      propertyEditor.getCustomEditor() : null);
    String[] choices = propertyEditor.getTags();

    if (comp != null) {
      PropertyEditorViewAspectAdapter aspectAdapter =
        new PropertyEditorViewAspectAdapter(modelAspectId, editRule,
                                            propertyEditor, mediator, comp);
      return(aspectAdapter);
    }

    else if (choices != null) {
      JComboBox comboBox = new JComboBox(choices);

      ViewValueConverter converter =
        new PropertyEditorViewValueConverter(Object.class, propertyEditor);

      JComboBoxAdapter aspectAdapter =
        new JComboBoxAdapter(modelAspectId, comboBox, editRule, converter,
                             mediator);
      return(aspectAdapter);
    }

    else {
      JTextField textField = new JTextField(20);
      
      ViewValueConverter converter =
        new PropertyEditorViewValueConverter(Object.class, propertyEditor);

      JTextFieldAdapter aspectAdapter =
        new JTextFieldAdapter(modelAspectId, textField, editRule, converter,
                             mediator);
      return(aspectAdapter);
    }
  }

  private PropertyEditorViewAspectAdapter(ModelAspectId modelAspectId,
                                          EditRule editRule,
                                          PropertyEditor propertyEditor,
                                          ModelEditMediator mediator,
                                          Component viewComponent)
  {
    super(modelAspectId, editRule, null, mediator);
    this.propertyEditor = propertyEditor;
    this.viewComponent  = viewComponent;

    propertyEditor.addPropertyChangeListener(this);
  }

  @Override
public Component getViewComponent()
  {
    return(viewComponent);
  }

  @Override
public boolean isEditable()
  {
    return(viewComponent.isEnabled());
  }

  @Override
public void setEditable(boolean val)
  {
    viewComponent.setEnabled(val);
  }

  @Override
public Object getViewAspectValue()
  {
    return(propertyEditor.getValue());
  }

  @Override
public String getViewAspectStringValue()
  {
    return(propertyEditor.getAsText());
  }
  
  @Override
protected void setViewAspectValue(Object viewAspectValue)
    throws IllegalArgumentException
  {
    propertyEditor.setValue(viewAspectValue);
  }

  @Override
public void setViewAspectStringValue(String stringValue)
  {
    propertyEditor.setAsText(stringValue);
  }

  @Override
public void propertyChange(PropertyChangeEvent event)
  {
    fireChangeEvent();
  }

  private PropertyEditor propertyEditor;
  private Component      viewComponent;

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  unit test
  ///
  /////////////////////////////////////////////////////////////////////////////

  public static void main(String[] args)
  {
    PropertyEditor boolEditor =
      PropertyEditorManager.findEditor(boolean.class);
    System.out.println("bolean editor = " + boolEditor);

    if (boolEditor == null) {
      boolEditor = PropertyEditorManager.findEditor(Boolean.class);
      System.out.println("second try - boolean editor = " + boolEditor);
    }

    if (boolEditor != null) {
      ModelAspectId aspectId = ModelAspectId.forName("Foo");
      ViewAspectAdapter viewAspectAdapter = PropertyEditorViewAspectAdapter.
        makeViewAspectAdapter(aspectId, EditRule.ALWAYS, boolEditor, null);
      System.out.println("aspect adapter = " + viewAspectAdapter);
    }

    System.exit(0);
  }
}
