/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */

package modelobjects.framework;

import java.beans.PropertyEditor;

/**
 *  PropertyEditorViewValueConverter implements ViewValueConverter
 *  using a JavaBeans PropertyEditor.  The view-type is always String.
 */
public class PropertyEditorViewValueConverter implements ViewValueConverter
{
  public PropertyEditorViewValueConverter(Class modelClass,
                                          PropertyEditor propertyEditor)
  {
    this.modelClass     = modelClass;
    this.propertyEditor = propertyEditor;
  }

  /**
   *  Convert the specified view aspect value in a view component to a model
   *  aspect value for updating the model object.
   *
   *  @param viewValue the view aspect value to convert
   *  @throws ViewValueConversionException if the conversion cannot be
   *    performed.
   */
  @Override
public Object viewValueToModel(Object viewValue)
       throws ViewValueConversionException
  {
    try {
      propertyEditor.setAsText((String)viewValue);
      return(propertyEditor.getValue());
    }
    catch (RuntimeException rte) {
      String message = rte.getMessage();
      if ((message == null) || (message.length() == 0))
    {
        message = rte.toString();
    }
      throw(new ViewValueConversionException(message));
    }
  }

  /**
   *  Convert the model aspect value to a view aspect value for display
   *  or edit in a view component.
   *
   *  @param viewValue the view aspect value to convert
   *  @throws ViewValueConversionException if the conversion cannot be
   *    performed.
   */
  @Override
public Object modelValueToView(Object modelValue)
       throws ViewValueConversionException
  {
    try {
      propertyEditor.setValue(modelValue);
      return(propertyEditor.getAsText());
    }
    catch (RuntimeException rte) {
      String message = rte.getMessage();
      if ((message == null) || (message.length() == 0))
    {
        message = rte.toString();
    }
      throw(new ViewValueConversionException(message));
    }
  }

  /**
   *  Returns an array of all the String values that may be used as view
   *  aspect values, or null if no such constraint applies.
   */
  @Override
public String[] getViewStringValues()
  {
    return(propertyEditor.getTags());
  }

  /**
   *  Returns whether or not conversion from view values to model values
   *  is supported by the ViewValueConverter.  If not, ViewAspectAdapters
   *  using this ViewValueConverter cannot be edited.
   */
  @Override
public boolean isViewValueToModelSupported()
  {
    return(true);
  }

  /**
   *  Return the model type returned by viewValueToModel()
   */
  @Override
public Class getModelClass()
  {
    return(modelClass);
  }

  /**
   *  Return the view type returned by modelValueToView()
   */
  @Override
public Class getViewClass()
  {
    return(String.class);
  }

  private Class                 modelClass;
  private PropertyEditor        propertyEditor;
}
