/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */

package modelobjects.framework;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyVetoException;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.FocusManager;
import javax.swing.JTable;
import javax.swing.SwingUtilities;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import modelobjects.framework.model.ModelObjectAdapter;
import modelobjects.util.swing.ModelListTableModel;
import modelobjects.util.swing.RelayAction;

public class TableFormMediator
  implements ModelEditListener, ListSelectionListener, PropertyChangeListener
{
  public TableFormMediator(final JTable table,
                           final EditFormManager formManager)
  {
    this.table       = table;
    this.tableModel  = (ModelListTableModel)table.getModel();
    this.formManager = formManager;

    modelEditEventSupport = new ModelEditEventSupport(this);
    modelEditFailureSupport = new ModelEditFailureSupport(this);

    // listen for events from both table and form
    formManager.addModelEditListener(this);
    table.getSelectionModel().addListSelectionListener(this);

    formManager.addPropertyChangeListener(this);

    newInstanceAction = new RelayAction(formManager.getNewInstanceAction()) {
        @Override
        public void actionPerformed(ActionEvent event) {
          table.clearSelection();
          super.actionPerformed(event);
          SwingUtilities.invokeLater(new FocusOnForm());
        }
      };
    deleteRowAction  = new DeleteRowAction();

    enableActions();
  }

  /**
   *  Register a ModelEditListener.
   */
  public void addModelEditListener(ModelEditListener listener)
  {
    modelEditEventSupport.addModelEditListener(listener);
  }

  /**
   *  Unregister a ModelEditListener.
   */
  public void removeModelEditListener(ModelEditListener listener)
  {
    modelEditEventSupport.removeModelEditListener(listener);
  }

  /**
   *  Register a ModelEditFailureListener.
   */
  public void addModelEditFailureListener(ModelEditFailureListener listener)
  {
    modelEditFailureSupport.addModelEditFailureListener(listener);
  }

  /**
   *  Unregister a ModelEditFailureListener.
   */
  public void removeModelEditFailureListener(ModelEditFailureListener listener)
  {
    modelEditFailureSupport.removeModelEditFailureListener(listener);
  }

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  Utility methods
  ///
  /////////////////////////////////////////////////////////////////////////////

  /**
   *  Return the Action used to delete a selected row in the table.
   */
  public Action getDeleteRowAction()
  {
    return(deleteRowAction);
  }

  /**
   *  Return the Action used to create and edit new model instances.
   */
  public Action getNewInstanceAction()
  {
    return(newInstanceAction);
  }

  /**
   *  Handle unapplied edits, either by discarding them (by just returning
   *  true), applying them (by invoking the EditFormManager's applyAction)
   *  and then returning true, or by returning false, which indicates that
   *  the form should not be made to stop editing the current object.
   */
  protected boolean handleUnappliedChanges()
  {
    if (true) {
      System.out.println("Apply or reset changes first!");
      return(false);
    }
    else {
      System.out.println("automatically applying unapplied changes...");
      formManager.getApplyAction().actionPerformed(null);
      return(true);
    }
  }

  /**
   *  Revert table selection to match what's being edited in the form.
   */
  protected void revertTableSelection()
  {
    if (rowModelInForm != null) {
      int index = tableModel.getIndexOf(rowModelInForm);
      table.setRowSelectionInterval(index, index);
    }
    else {
      table.clearSelection();
    }
  }

  /**
   *  Check with the user to make sure it's OK to delete the specified
   *  object.  The default implementation just returns true without
   *  actually checking.
   */
  public boolean confirmDelete(Object modelObject)
  {
    return(true);
  }


  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  Listener methods
  ///
  /////////////////////////////////////////////////////////////////////////////

  /**
   *  Handle a change in the table selection.
   */
  @Override
public void valueChanged(ListSelectionEvent listSelectionEvent)
  {
    if (listSelectionEvent.getValueIsAdjusting() ||
        table.getSelectionModel().getValueIsAdjusting()) {
      return;      // don't handle selection changes when "adjusting"
    }

    try {
      int numSelectedRows = table.getSelectedRowCount();

      if (numSelectedRows != 1) {
        rowModelInForm = null;
        EditForm activeForm = formManager.getActiveEditForm();
        if (activeForm != null)
        {
            formManager.editModelObject(null, activeForm.getEditTypeKey());
        }
      }
      else {
        int row = table.getSelectedRow();
        if ((row >= 0) && (row < tableModel.getRowCount())) {
          Object rowModel = tableModel.getRowModel(row);

          if (rowModel == rowModelInForm) {
            return;     // reselected same object - possibly because of sorting
          }

          if (!formManager.hasUnappliedChanges() || handleUnappliedChanges()) {
            rowModelInForm = rowModel;
            Object editTypeKey = formManager.getEditTypeKey(rowModel);
            formManager.editModelObject(rowModel, editTypeKey);
          }
          else {
            revertTableSelection();
          }
        }
      }
    }
    catch (PropertyVetoException e) {
      revertTableSelection();
    }
    catch (ViewValueConversionException e) {
      revertTableSelection();
    }

    enableActions();
  }

  /**
   *  Handle a ModelEditEvent from the active form.
   */
  @Override
public void modelEdited(ModelEditEvent modelEditEvent)
  {
    if (modelEditEvent.getObjectWasNew())
    {
      modelCreated(modelEditEvent);
    }
    else
    {
      tableModel.rowChanged(table.getSelectedRow());

      // forward the event to this TableFormMediator's ModelEditListeners too
      modelEditEventSupport.fireModelEdited(modelEditEvent);
    }
  }

  /**
   *  Forward ModelEditEvents from the ModelEditMediator to this EditForm's
   *  ModelEditListeners.
   */
  @Override
public void modelCreated(ModelEditEvent modelEditEvent)
  {
    Object modelObject = modelEditEvent.getModelObject();

    // check if the row is already in the table
    int row = tableModel.getIndexOf(modelObject);

    if(row < 0)
    {
        tableModel.insertRowMaintainSort(modelObject);
    }
    else
    {
        tableModel.rowChanged(row);
    }

    row = tableModel.getIndexOf(modelObject);
    table.setRowSelectionInterval(row, row);

    // forward the event to this TableFormMediator's ModelEditListeners too
    modelEditEventSupport.fireModelCreated(modelEditEvent);
  }

  /**
   *  Forward ModelEditEvents from the ModelEditMediator to this EditForm's
   *  ModelEditListeners.
   */
  @Override
public void modelDeleted(ModelEditEvent modelEditEvent)
  {
    // this is never initiated by the form, so it will never happen

    // forward the event to this TableFormMediator's ModelEditListeners too
    modelEditEventSupport.fireModelDeleted(modelEditEvent);
  }

  /**
   *  Forward ModelEditFailureEvents from the ModelEditMediator to
   *  this EditForm's ModelEditFailureListeners.
   */
  public void modelEditFailed(ModelEditFailureEvent event)
  {
    modelEditFailureSupport.fireModelEditFailure(event);
  }

  @Override
public void propertyChange(PropertyChangeEvent propertyChangeEvent)
  {
    Object eventSource = propertyChangeEvent.getSource();

    if (eventSource == formManager)
    {
      hasPendingEdits = formManager.hasUnappliedChanges();
      enableActions();
    }
  }

  final void enableActions()
  {
    int numSelectedRows = table.getSelectedRowCount();
    deleteRowAction.setEnabled((numSelectedRows == 1) && !hasPendingEdits);
  }

  class DeleteRowAction extends AbstractAction
  {
    public DeleteRowAction()
    {
      super("Delete...");
    }

    @Override
    public void actionPerformed(ActionEvent event)
    {
      if (table.getSelectedRowCount() == 1) {
        int row = table.getSelectedRow();
        Object rowModel = tableModel.getRowModel(row);

        if (confirmDelete(rowModel)) {
          try {
            // first validate the delete using the ModelObjectAdapter
            ModelObjectAdapter rowModelAdapter =
              formManager.getModelObjectAdapter();
            if ((rowModelAdapter != null) &&
                (rowModelAdapter.getModelObject() == rowModel)) {
              rowModelAdapter.validateModelDeletion();
              rowModelAdapter.propagateModelDeletion();
            }

            modelEditEventSupport.fireModelDeleted(ModelEditEvent.DELETE,
                                                   rowModel, false);
            table.clearSelection();
            tableModel.deleteRow(row);

            int rowCount = tableModel.getRowCount();
            //!!System.out.println("after delete: row-count = " + rowCount);
            if (rowCount > row)
            {
                table.setRowSelectionInterval(row, row);
            }
            else if (rowCount > 0)
            {
                table.setRowSelectionInterval(row-1, row-1);
            }
          }
          catch (Exception e) {
            // fire a ModelEditFailureEvent
            modelEditFailureSupport.fireModelEditFailure(rowModel, e);
          }
        }
        enableActions();
      }
    }
  }

  /**
   *  FocusOnForm is used in a call to SwingUtilities.invokeLater to
   *  move the focus to the form for editing a new instance.
   */
  class FocusOnForm implements Runnable
  {
    FocusOnForm() {
    }
    @Override
    public void run() {
      Component comp = formManager.getActiveEditForm().getFormComponent();
      FocusManager.getCurrentManager().focusNextComponent(comp);
    }
  }


  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  private representation
  ///
  /////////////////////////////////////////////////////////////////////////////

  /** the JTable associated with the EditFormManager */
  private       JTable                          table;

  /** the JTable's ModelListTableModel */
  private       ModelListTableModel             tableModel;

  /** the EditFormManager associated with the JTable, used to edit instances */
  private       EditFormManager                 formManager;

  /** the Object being edited */
  private       Object                          rowModelInForm;

  /** action for deleting rows from table */
  private       Action                          deleteRowAction;

  /** wrapper for EditFormManager's newInstanceAction */
  private       RelayAction                     newInstanceAction;

  /** indicates whether there are any pending edits in the form */
  private       boolean                         hasPendingEdits;

  /** support for ModelEditListeners */
  private       ModelEditEventSupport           modelEditEventSupport;

  /** support for ModelEditFailureListeners */
  private ModelEditFailureSupport modelEditFailureSupport;

}
