/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */
package modelobjects.framework;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyVetoException;

import modelobjects.framework.model.ModelAspectId;
import modelobjects.framework.model.ModelObjectAdapter;
import modelobjects.framework.model.NoSuchAspectException;
import modelobjects.util.undo.CannotDoException;
import modelobjects.util.undo.CannotRedoException;
import modelobjects.util.undo.CannotUndoException;
import modelobjects.util.undo.DoableUndoableEdit;
import modelobjects.util.undo.PresentableUndoableEdit;

public class UndoableModelAspectEdit extends PresentableUndoableEdit
  implements DoableUndoableEdit
{
  public UndoableModelAspectEdit(String presentationName,
                                 ModelObjectAdapter modelObjectAdapter,
                                 ModelAspectId modelAspectId,
                                 Object oldValue,
                                 Object newValue)
  {
    super(presentationName);

    this.modelObjectAdapter = modelObjectAdapter;
    this.modelAspectId      = modelAspectId;
    this.oldValue           = oldValue;
    this.newValue           = newValue;
  }

  /**
   *  Apply this UndoableModelAspectEdit for the first time (or possibly
   *  after an undo).
   */
  @Override
public void doEdit()
    throws CannotDoException
  {
    doEdit(true);
  }

  public void doEdit(boolean compareOldWithNew)
    throws CannotDoException
  {
    if (alreadyDone && !canRedo())
    {
      Exception reason = new IllegalStateException("alreadyDone && canRedo() == false");

      throw new CannotDoException("doEdit", reason);
    }

    if (!compareOldWithNew || (newValue != oldValue))
    {
      try
      {
        modelObjectAdapter.setAspectValue(modelAspectId, newValue);
      }
      catch (PropertyVetoException e)
      {
        throw(new CannotDoException(e));
      }
      catch (NoSuchAspectException e)
      {
        throw(new CannotDoException(e));
      }
    }
  }

  /**
   *  Undo the UndoableEdit.  Subclasses should add the rest of their
   *  undo behavior after calling super.undo().
   *  @throws CannotUndoException if the canUndo() returns false
   */
  @Override
public void undo()
    throws CannotUndoException
  {
    super.undo();

    if (newValue != oldValue) {
      try {
        modelObjectAdapter.setAspectValue(modelAspectId, oldValue);
      }
      catch (PropertyVetoException e) {
        throw(new CannotUndoException(e));
      }
      catch (NoSuchAspectException e) {
        throw(new CannotUndoException(e));
      }
    }
  }

  /**
   *  Redo the UndoableEdit.  Subclasses should add the rest of their
   *  redo behavior after calling super.redo().
   *  @throws CannotUndoException if the canRedo() returns false
   */
  @Override
public void redo()
    throws CannotRedoException
  {
    super.redo();

    if (newValue != oldValue) {
      try {
        modelObjectAdapter.setAspectValue(modelAspectId, newValue);
      }
      catch (PropertyVetoException e) {
        throw(new CannotRedoException(e));
      }
      catch (NoSuchAspectException e) {
        throw(new CannotRedoException(e));
      }
    }
  }

  /**
   *  Return the ModelAspectId
   */
  public ModelAspectId getModelAspectId()
  {
    return(modelAspectId);
  }


  /**
   *  Handle notification that this UndoableEdit will no longer be used.
   */
  @Override
public void die()
  {
    super.die();
    oldValue = null;
    newValue = null;
  }

  /**
   *  Return a PropertyChangeEvent representing the model aspect edit.
   */
  public PropertyChangeEvent getPropertyChangeEvent()
  {
    return(new PropertyChangeEvent(modelObjectAdapter,
                                   modelAspectId.getName(),
                                   oldValue, newValue));
  }

  private ModelObjectAdapter    modelObjectAdapter;
  private ModelAspectId         modelAspectId;
  private Object                oldValue;
  private Object                newValue;
  private boolean               alreadyDone;
}
