/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */
package modelobjects.framework;

import java.beans.PropertyVetoException;

import javax.swing.undo.AbstractUndoableEdit;

import modelobjects.framework.model.ModelObjectAdapter;
import modelobjects.util.undo.CannotRedoException;
import modelobjects.util.undo.CannotUndoException;

/**
 *  UndoableSetModelEdit is a kind of UndoableEdit used for keeping track
 *  of changes to models set in ModelEditMediators.
 */
public class UndoableSetModelEdit extends AbstractUndoableEdit
{
  /**
   *  Construct an UndoableSetModelEdit with the specified presentation name,
   *  for the specified ModelEditMediator, and for the old and new models.
   */
  public UndoableSetModelEdit(String presentationName,
                              ModelEditMediator modelEditMediator,
                              ModelObjectAdapter oldModel, boolean oldIsNew,
                              ModelObjectAdapter newModel, boolean newIsNew)
  {
    this.presentationName   = presentationName;
    this.modelEditMediator  = modelEditMediator;

    this.oldModel           = oldModel;
    this.oldIsNew           = oldIsNew;
    this.newModel           = newModel;
    this.newIsNew           = newIsNew;
  }

  /**
   *  Undo the UndoableModelEdit.
   *
   *  @throws CannotUndoException if the undo cannot be done, or if it fails.
   */
  @Override
public void undo()
    throws CannotUndoException
  {
    if (!canUndo()) {
      Exception reason = new IllegalStateException("canUndo == false");
      throw(new CannotUndoException(getUndoPresentationName(), reason));
    }

    Exception undoFailure =
      flipState(newModel, newIsNew, oldModel, oldIsNew, true);

    if (undoFailure != null) {
      undoFailure.printStackTrace();
      throw(new CannotUndoException(getUndoPresentationName(), undoFailure));
    }

    super.undo();
  }

  /**
   *  Redo the UndoableModelEdit
   *
   *  @throws CannotRedoException if the redo cannot be done, or if it fails.
   */
  @Override
public void redo()
    throws CannotRedoException
  {
    if (!canRedo()) {
      Exception reason = new IllegalStateException("canRedo == false");
      throw(new CannotRedoException(getRedoPresentationName(), reason));
    }

    Exception redoFailure =
      flipState(oldModel, oldIsNew, newModel, newIsNew, true);

    if (redoFailure != null) {
      throw(new CannotUndoException(getRedoPresentationName(), redoFailure));
    }

    super.redo();
  }

  /**
   *  Return the presentation name for the UndoableModelEdit
   */
  @Override
public String getPresentationName()
  {
    return(presentationName);
  }

  /**
   *  Return the undo presentation name for the UndoableModelEdit
   */
  @Override
public String getUndoPresentationName()
  {
    return("undo " + presentationName);
  }

  /**
   *  Return the redo presentation name for the UndoableModelEdit
   */
  @Override
public String getRedoPresentationName()
  {
    return("redo " + presentationName);
  }

  Exception flipState(ModelObjectAdapter fromModel, boolean fromIsNew,
                      ModelObjectAdapter toModel,   boolean toIsNew,
                      boolean isUndo)
  {
    boolean succeeded = false;
    Exception failure = null;

    try {
      if (toIsNew)
    {
        modelEditMediator.editModelAsNewObject(toModel);
    }
    else
    {
        modelEditMediator.editModelObject(toModel);
    }

      succeeded = true;
    }
    catch (PropertyVetoException veto) {
      failure = veto;
    }
    catch (ViewValueConversionException badVal) {
      failure = badVal;
    }
    finally {
      if (!succeeded) {
        try {
          if (fromIsNew)
        {
            modelEditMediator.editModelAsNewObject(fromModel);
        }
        else
        {
            modelEditMediator.editModelObject(fromModel);
        }
        }
        catch (PropertyVetoException veto) {
          // TBD: How should this be reported?!
        }
        catch (ViewValueConversionException badVal) {
          // TBD: How should this be reported?!
        }
      }
    }

    return(failure);
  }

  @Override
public void die()
  {
    super.die();
    modelEditMediator = null;
    oldModel = null;
    newModel = null;
  }

  ///////////////////////////////////////////////////////////////////////////
  ///
  ///   Representation
  ///
  ///////////////////////////////////////////////////////////////////////////

  private String                presentationName;
  private ModelEditMediator     modelEditMediator;
  private ModelObjectAdapter    oldModel;
  private boolean               oldIsNew;
  private ModelObjectAdapter    newModel;
  private boolean               newIsNew;
}
