/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */

package modelobjects.framework;

import java.awt.Component;

import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.UndoableEditListener;
import javax.swing.undo.UndoableEdit;
import javax.swing.undo.UndoableEditSupport;

import modelobjects.framework.model.ModelAspectId;
import modelobjects.framework.model.ModelObjectAdapter;
import modelobjects.util.ChangeEventSupport;

/**
 *  This class serves as an adapter to provide a uniform interface for all
 *  view components used to display and edit aspects of model objects in
 *  the ModelObjects framework.
 */
public abstract class ViewAspectAdapter
{
  /**
   *  Construct a ViewAspectAdapter to view or edit the specified model aspect.
   *
   *  @param modelAspectId the ModelAspectId of the model aspect that this
   *    ViewAspectAdapter is used to view or edit.
   */
  protected ViewAspectAdapter(ModelAspectId modelAspectId)
  {
    this(modelAspectId, null, null, null);
  }

  /**
   *  Construct a ViewAspectAdapter to view or edit the specified model aspect
   *  using the specified EditRule and ViewValueConverter, and managed by the
   *  specified ModelEditMediator.
   *
   *  @param modelAspectId the ModelAspectId of the model aspect that this
   *    ViewAspectAdapter is used to view or edit.
   *  @param editRule the EditRule used to decide when the ViewAspectAdapter
   *    should be editable.
   *  @param viewValueConverter the ViewValueConverter used to convert between
   *    model aspect values and view aspect values.
   *  @param modelEditMediator the ModelEditMediator that manages the
   *    interactions between the ViewAspectAdapter and the model object.
   */
  protected ViewAspectAdapter(ModelAspectId modelAspectId,
                              EditRule editRule,
                              ViewValueConverter viewValueConverter,
                              ModelEditMediator modelEditMediator)
  {
    this.modelAspectId = modelAspectId;

    changeEventSupport = new ChangeEventSupport(this);

    setEditRule(editRule);
    setViewValueConverter(viewValueConverter);
    setModelEditMediator(modelEditMediator);
  }

  /**
   *  Return whether the adapted view component is current editable.
   */
  public abstract boolean isEditable();

  /**
   *  Assign whether the adapted view component should be editable.
   *
   *  @param editable true if the component should be editable
   */
  public abstract void setEditable(boolean editable);

  /**
   *  Return the AWT / Swing Component associated with this ViewAspectAdapter
   */
  public abstract Component getViewComponent();

  /**
   *  Return the current view aspect value, without conversion.
   */
  public abstract Object getViewAspectValue();

  /**
   *  Return the current view aspect value as a String, or null if the view
   *  component does not support a String representation of the view value.
   */
  public abstract String getViewAspectStringValue();

  /**
   *  Set the viewAspectValue to the specified value, which should be
   *  appropriate for the view component.
   *
   *  @param viewAspectValue the value to view or edit in the view component
   *  @throws IllegalArgumentException if the viewAspectValue is inappropriate
   *    for the view component
   */
  protected abstract void setViewAspectValue(Object viewAspectValue)
    throws IllegalArgumentException;

  /**
   *  Set the viewAspectValue to the specified String value, which should be
   *  appropriate for the view component.
   *
   *  @param viewStringValue the value to view or edit in the view component
   *  @throws IllegalArgumentException if the viewStringValue is inappropriate
   *    for the view component
   */
  protected abstract void setViewAspectStringValue(String viewStringValue)
    throws IllegalArgumentException;

  /**
   *  Return the model aspect value represented by the current view aspect
   *  value, using the ViewValueConverter to perform the conversion.
   *
   *  @throws ViewValueConversionException if the current view aspect value
   *    cannot be converted to a model aspect value.
   */
  public  Object getModelAspectValue()
    throws ViewValueConversionException
  {
    Object viewAspectValue = null;

    try
    {
      viewAspectValue = getViewAspectValue();
    }
    catch (Exception e)
    {
      throw new ViewValueConversionException(e, getModelAspectId());
    }

    ViewValueConverter converter = getViewValueConverter();

    if (converter == null)
    {
        ;
    }
    else
    if (converter.isViewValueToModelSupported())
    {
      try
      {
        viewAspectValue = converter.viewValueToModel(viewAspectValue);
      }
      catch (ViewValueConversionException e)
      {
        e.setModelAspectId(getModelAspectId());

        throw e;
      }
    }
    else
    {
        throw new ViewValueConversionException("View value to Model value conversion not supported",
                                                 getModelAspectId());
    }

    return viewAspectValue;
  }

  /**
   *  Set the model aspect value for the ViewAspectAdapter, using the
   *  ViewValueConverter to convert the model aspect value to a view aspect
   *  value appropriate for display and editing in the view component.
   *
   *  @param modelAspectValue the model aspect value to be viewed or edited,
   *    after conversion by the ViewValueConverter
   *
   *  @throws ViewValueConversionException if the specified model aspect value
   *    cannot be converted to a view aspect value.
   */
  public void setModelAspectValue(Object modelAspectValue)
    throws ViewValueConversionException
  {
    this.oldModelAspectValue = modelAspectValue;

    ViewValueConverter converter = getViewValueConverter();

    try
    {
        if (converter == null)
        {
            setViewAspectValue(modelAspectValue);
        }
        else if (converter.getViewClass() == String.class)
        {
            String viewAspectValue =
                ((modelAspectValue == null) ? "" :
                 (String) converter.modelValueToView(modelAspectValue));
            setViewAspectStringValue(viewAspectValue);
        }
        else
        {
            Object viewAspectValue = converter.modelValueToView(modelAspectValue);
            setViewAspectValue(viewAspectValue);
        }
    }
    catch (ViewValueConversionException e)
    {
        e.setModelAspectId(getModelAspectId());
        throw e;
    }
  }

  /**
   *  Return the name of the model aspect that this ViewAspectAdapter is used
   *  to view or edit.
   */
  public ModelAspectId getModelAspectId()
  {
    return(modelAspectId);
  }

  /**
   *  Return the EditRule used to decide when the ViewAspectAdapter
   *  should be editable.
   */
  public EditRule getEditRule()
  {
    return(editRule);
  }

  /**
   *  Assign the EditRule used to decide when the ViewAspectAdapter
   *  should be editable.
   */
  public final void setEditRule(EditRule editRule)
  {
    this.editRule = editRule;
  }

  /**
   *  Return the ViewValueConverter used to convert between model aspect
   *  values and view aspect values, or null if no conversion is performed.
   */
  public ViewValueConverter getViewValueConverter()
  {
    return(viewValueConverter);
  }

  /**
   *  Assign the ViewValueConverter used to convert between model aspect
   *  values and view aspect values, or null if no conversion is to be
   *  performed.
   */
  public final void setViewValueConverter(ViewValueConverter viewValueConverter)
  {
    this.viewValueConverter = viewValueConverter;
  }

  /**
   *  Return the ModelEditMediator that manages the interactions between
   *  the ViewAspectAdapter and the model object.
   */
  public ModelEditMediator getModelEditMediator()
  {
    return(modelEditMediator);
  }

  /**
   *  Assign the ModelEditMediator that manages the interactions between
   *  the ViewAspectAdapter and the model object.
   */
  public final void setModelEditMediator(ModelEditMediator modelEditMediator)
  {
    if (this.modelEditMediator != null)
    {
        this.modelEditMediator.removeViewAspectAdapter(this);
    }

    this.modelEditMediator = modelEditMediator;

    if (this.modelEditMediator != null)
    {
        this.modelEditMediator.addViewAspectAdapter(this);
    }
  }

  /**
   *  Return an UndoableModelAspectEdit representing the edits made to the
   *  model aspect value since the last time setModelAspectValue was called.
   *  This is a Factory method which may return null if it determines that no
   *  changes were made.  Different subclasses of ViewAspectAdapter may return
   *  different subclasses of UndoableModelAspectEdit, as appropriate for the
   *  kinds of edits made.
   *  This implementation returns an UndoableModelAspectEdit that contains
   *  only the old aspect value and new aspect value, and does not record
   *  any modifications that may have been made within the aspect value.
   */
  public UndoableModelAspectEdit
    makeUndoableModelAspectEdit(ModelObjectAdapter modelObjectAdapter)
    throws ViewValueConversionException
  {
    Object newModelAspectValue = getModelAspectValue();

    if (newModelAspectValue == oldModelAspectValue)
    {
        return(null);
    }

    return(new UndoableModelAspectEdit("changed " + modelAspectId,
                                       modelObjectAdapter,
                                       modelAspectId,
                                       oldModelAspectValue,
                                       newModelAspectValue));
  }

  /**
   *  Return the last model aspect value assigned by setModelAspectValue.
   */
  protected Object getOldModelAspectValue()
  {
    return(oldModelAspectValue);
  }

  /**
   *  Register a ChangeListener to be notified of ChangeEvents when the
   *  view aspect value is modified by the view component.
   */
  public void addChangeListener(ChangeListener listener)
  {
    changeEventSupport.addChangeListener(listener);
  }

  /**
   *  Unregister a ChangeListener.
   */
  public void removeChangeListener(ChangeListener listener)
  {
    changeEventSupport.removeChangeListener(listener);
  }

  /**
   *  Fire a ChangeEvent to registered ChangeListeners.  This method should
   *  be called by subclasses whenever the view aspect value is modified by
   *  the view component.  An exception should be made, however, when the
   *  view value is changed as a result of a call to setViewAspectValue or
   *  setViewAspectStringValue.
   */
  protected void fireChangeEvent()
  {
    changeEventSupport.fireChangeEvent(changeEvent);
  }

  /**
   *  Register an UndoableEditListener
   */
  public void addUndoableEditListener(UndoableEditListener listener)
  {
    if (undoableEditSupport == null)
    {
        undoableEditSupport = new UndoableEditSupport(this);
    }
    undoableEditSupport.addUndoableEditListener(listener);
  }

  /**
   *  Unregister an UndoableEditListener
   */
  public void removeUndoableEditListener(UndoableEditListener listener)
  {
    if (undoableEditSupport != null)
    {
        undoableEditSupport.removeUndoableEditListener(listener);
    }
  }

  /**
   *  Fire an UndoableEditEvent to UndoableEditListeners
   */
  protected void fireUndoableEditEvent(UndoableEdit anEdit)
  {
    if (undoableEditSupport != null)
    {
        undoableEditSupport.postEdit(anEdit);
    }
  }


  private Object                        oldModelAspectValue;

  private final ModelAspectId           modelAspectId;

  private EditRule                      editRule;

  private ViewValueConverter            viewValueConverter;

  private ModelEditMediator             modelEditMediator;

  private ChangeEventSupport            changeEventSupport;

  private UndoableEditSupport           undoableEditSupport;

  private ChangeEvent                   changeEvent = new ChangeEvent(this);
}
