/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */

package modelobjects.framework.model;

import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;

/**
 *  This kind of ModelDescriptor uses JavaBeans introspection to find all
 *  the Beans properties defined by the model class, and automatically adds
 *  ModelAspectAdapters for each property, treating them as aspects.
 *  Special handling is provided for properties whose types implement
 *  java.util.List.
 */
public class JavaBeanModelDescriptor extends ModelDescriptor
{
  /**
   *  Construct a JavaBeanModelDescriptor for the specified model object type.
   *
   *  @param beanClass the application model type adapted by the
   *    JavaBeanModelDescriptor.
   */
  public JavaBeanModelDescriptor(Class beanClass)
    throws IntrospectionException
  {
    this(beanClass, Object.class, beanClass);
  }

  /**
   *  Construct a JavaBeanModelDescriptor for the specified model object type.
   *
   *  @param beanClass the application model type adapted by the
   *    JavaBeanModelDescriptor.
   *  @param stopClass the ancestor superclass of the beanClass at which
   *    to stop collecting BeanInfo.
   */
  public JavaBeanModelDescriptor(Class beanClass, Class stopClass)
    throws IntrospectionException
  {
    this(beanClass, stopClass, beanClass);
  }

  /**
   *  Construct a JavaBeanModelDescriptor for the specified model object type,
   *  and using the specified edit-type metadata object as a lookup key.
   *
   *  @param beanClass the application model type adapted by the
   *    JavaBeanModelDescriptor.
   *  @param editTypeKey an object (normally a class-level metadata object of
   *    some kind) that represents the type adapted by the ModelDescriptor.
   */
  public JavaBeanModelDescriptor(Class beanClass, Object editTypeKey)
    throws IntrospectionException
  {
    this(beanClass, Object.class, editTypeKey);
  }

  /**
   *  Construct a JavaBeanModelDescriptor for the specified model object type,
   *  and using the specified edit-type metadata object as a lookup key.
   *
   *  @param beanClass the application model type adapted by the
   *    JavaBeanModelDescriptor.
   *  @param stopClass the ancestor superclass of the beanClass at which
   *    to stop collecting BeanInfo.
   *  @param editTypeKey an object (normally a class-level metadata object of
   *    some kind) that represents the type adapted by the ModelDescriptor.
   */
  public JavaBeanModelDescriptor(Class beanClass, Class stopClass,
                                 Object editTypeKey)
    throws IntrospectionException
  {
    super(beanClass, editTypeKey);

    // get the BeanInfo for the model class
    BeanInfo beanInfo = Introspector.getBeanInfo(beanClass, stopClass);
    PropertyDescriptor[] propDescs = beanInfo.getPropertyDescriptors();

    // add ModelAspectAdapters for each property
    for (int i = 0, n = propDescs.length; i < n; i++)
      {
        PropertyDescriptor prop = propDescs[i];
        Class propType = prop.getPropertyType();
        addModelAspectAdapter(new JavaBeanPropertyAdapter(prop));
      }
  }

}
