/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */

package modelobjects.framework.model;


import java.beans.PropertyDescriptor;
import java.beans.PropertyVetoException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import modelobjects.util.WrapperRuntimeException;

/**
 *  This subclass of ModelAspectAdapter uses the getter and setter methods
 *  from a JavaBeans PropertyDescriptor to access the underlying property.
 *  It uses the Java reflection api to invoke the property methods.
 */
public class JavaBeanPropertyAdapter extends ModelAspectAdapter
{
  /**
   *  Construct a JavaBeanPropertyAdapter for the property described by the
   *  specified PropertyDescriptor.
   */
  public JavaBeanPropertyAdapter(PropertyDescriptor propDesc)
  {
    super(ModelAspectId.forName(propDesc.getName()),
          propDesc.getPropertyType());

    this.getterMethod = propDesc.getReadMethod();
    this.setterMethod = propDesc.getWriteMethod();
  }

  /**
   *  Return whether the aspect is a read-only aspect which cannot be
   *  assigned a new value.
   *  This implementation says the aspect is read-only if the property
   *  does not have a setter method.
   */
  @Override
public boolean isReadonlyAspect()
  {
    return(setterMethod == null);
  }

  /**
   *  Get this ModelAspectAdapter's model aspect value from the specified
   *  model object, using the ModelObjectAdapter for additional context if
   *  needed.
   *
   *  @param modelObject the application model object from which to get the
   *    aspect value.
   *  @param modelObjectAdapter the ModelObjectAdapter used as an adapter
   *    for the model object.
   */
  @Override
protected Object getAspectValue(Object modelObject,
                                  ModelObjectAdapter modelObjectAdapter)
  {
    try {
      return(getterMethod.invoke(modelObject, new Object[0]));
    }
    catch (IllegalAccessException illegalX) {
      throw(new WrapperRuntimeException(illegalX));
    }
    catch (InvocationTargetException callX) {
      throw(new WrapperRuntimeException(callX.getTargetException()));
    }
  }

  /**
   *  Set this ModelAspectAdapter's model aspect value in the specified
   *  model object to the specified new value, using the ModelObjectAdapter
   *  for additional context if needed.
   */
  @Override
protected void setAspectValue(Object modelObject,
                                Object newAspectValue,
                                ModelObjectAdapter modelObjectAdapter)
    throws PropertyVetoException, IllegalArgumentException,
           UnsupportedOperationException
  {
    if (setterMethod == null)
    {
        throw(new UnsupportedOperationException("setAspectValue: " +
                                                    getModelAspectId()));
    }

    try {
      setterMethod.invoke(modelObject, new Object[] { newAspectValue });
    }
    catch (IllegalAccessException illegalX) {
      throw(new WrapperRuntimeException(illegalX));
    }
    catch (InvocationTargetException callX) {
      Throwable targetException = callX.getTargetException();

      if (targetException instanceof PropertyVetoException)
    {
        throw((PropertyVetoException)targetException);
    }
    else
    {
        throw(new WrapperRuntimeException(targetException));
    }
    }
  }

  private Method getterMethod;
  private Method setterMethod;
}
