/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */

package modelobjects.framework.model;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Set;
import java.util.Vector;

/**
 *  The class ModelDescriptor is used to provide a uniform interfact to
 *  application model objects within the ModelObjects framework.  Aspects
 *  of model objects are defined by ModelAspectAdapters which are managed
 *  by ModelDescriptors.
 */
public class ModelDescriptor
{
  /**
   *  Construct a ModelDescriptor for the specified model object type,
   *  also using the model type as the edit-type key for registering the
   *  ModelDescriptor.
   *
   *  @param modelType the application model type adapted by the
   *    ModelDescriptor.
   */
  public ModelDescriptor(Class modelType)
  {
    this(modelType, modelType);
  }

  /**
   *  Construct a ModelDescriptor for the specified model object type,
   *  and using the specified edit-type metadata object as a lookup key.
   *
   *  @param modelType the application model type adapted by the
   *    ModelDescriptor.
   *  @param editTypeKey an object (normally a class-level metadata object of
   *    some kind) that represents the type adapted by the ModelDescriptor.
   */
  public ModelDescriptor(Class modelType, Object editTypeKey)
  {
    if (modelType == null)
    {
        throw(new IllegalArgumentException("ModelDescriptor.modelType = null"));
    }
    if (editTypeKey == null)
    {
        throw(new IllegalArgumentException("ModelDescriptor.editTypeKey = null"));
    }

    this.modelType = modelType;
    this.editTypeKey = editTypeKey;
    aspectAdapterList  = new Vector();
    aspectAdapterTable = new Hashtable();

    descriptorRegistry.put(editTypeKey, this);
  }

  /**
   *  Return the ModelDescriptor for the specified model type, or null
   *  if not ModelDescriptor has been created for that model type.
   *  ModelDescriptors are registered automatically when they are constructed.
   *
   *  @param  modelType the application model type adapted by the
   *    ModelDescriptor.
   */
  public static ModelDescriptor forClass(Class modelType)
  {
    return((ModelDescriptor)descriptorRegistry.get(modelType));
  }

  /**
   *  Return the ModelDescriptor for the specified edit-type key, or null
   *  if not ModelDescriptor has been created for that key.
   *  ModelDescriptors are registered automatically when they are constructed.
   *
   *  @param editTypeKey an object (normally a class-level metadata object of
   *  some kind) that represents the type adapted by the ModelDescriptor.
   */
  public static ModelDescriptor forEditTypeKey(Object editTypeKey)
  {
    return((ModelDescriptor)descriptorRegistry.get(editTypeKey));
  }

  /**
   *  Remove the ModelDescriptor for the specified model type from the
   *  ModelDescriptor registry.
   *
   *  @param  editTypeKey the application model type (or other edit-type key)
   *    adapted by the ModelDescriptor to be removed.
   */
  public static void unregisterByType(Object editTypeKey)
  {
    descriptorRegistry.remove(editTypeKey);
  }

  /**
   *  Remove the specified ModelDescriptor from the  ModelDescriptor registry.
   *
   *  @param  modelDesc the ModelDescriptor to be removed.
   */
  public static void unregister(ModelDescriptor modelDesc)
  {
    descriptorRegistry.remove(modelDesc.getModelType());
  }

  /**
   *  Add a ModelAspectAdapter.  If a ModelAspectAdapter with the same
   *  name was already present, it is replace by the new one.
   *
   *  @param aspectAdapter the ModelAspectAdapter to add
   */
  public synchronized final void
    addModelAspectAdapter(ModelAspectAdapter modelAspectAdapter)
  {
    ModelAspectId aspectId = modelAspectAdapter.getModelAspectId();

    ModelAspectAdapter previousAspectAdapter =
      (ModelAspectAdapter)aspectAdapterTable.get(aspectId);

    if (previousAspectAdapter != null)
      {
        int pos = aspectAdapterList.indexOf(aspectAdapterTable.get(aspectId));
        aspectAdapterList.setElementAt(modelAspectAdapter, pos);
      }
    else
      {
        aspectAdapterList.addElement(modelAspectAdapter);
      }

    aspectAdapterTable.put(aspectId, modelAspectAdapter);
  }

  /**
   *  Return a ModelObjectAdapter for the specified model object.
   *  This method follows the Factory Method design pattern.  The default
   *  implementation returns instances of ModelObjectAdapter.
   *
   *  @param modelObject the application model object for which to create
   *    a ModelObjectAdapter
   */
  public ModelObjectAdapter makeModelObjectAdapter(Object modelObject)
  {
    return(new ModelObjectAdapter(modelObject, this));
  }

  /**
   *  Return an ImmutableModelObjectAdapter.  This method follows the
   *  Factory Method design pattern.  The default implementation
   *  returns instances of ImmutableModelObjectAdapter.
   */
  public ImmutableModelObjectAdapter makeModelObjectAdapter()
  {
    return(new ImmutableModelObjectAdapter(this));
  }

  /**
   *  Return the ModelDescriptor's model object type.
   */
  public Class getModelType()
  {
    return(modelType);
  }

  /**
   *  Return the edit-type metadata key by which this ModelDescriptor
   *  is registered.
   */
  public Object getEditTypeKey()
  {
    return(editTypeKey);
  }

  /**
   *  Return the ModelAspectAdapter for the specified ModelAspectId, or
   *  null if no ModelAspectAdapter with that ModelAspectId has been added.
   *
   *  @param aspectId the ModelAspectId of the ModelAspectAdapter
   */
  protected ModelAspectAdapter getModelAspectAdapter(ModelAspectId aspectId)
    throws NoSuchAspectException
  {
    ModelAspectAdapter result =
      (ModelAspectAdapter)aspectAdapterTable.get(aspectId);
    if (result == null)
    {
        throw(new NoSuchAspectException(aspectId.getName()));
    }
    else
    {
        return(result);
    }
  }

  /**
   *  Return whether the specified aspect has the specified default value.
   *  This method delegates to ModelAspectAdapter.isDefaultValue(Object)
   */
  public boolean isDefaultValue(ModelAspectId aspectId, Object value)
    throws NoSuchAspectException
  {
    return(getModelAspectAdapter(aspectId).isDefaultValue(value));
  }

  /**
   *  Return an array of all the ModelAspectAdapters.
   */
  public synchronized ModelAspectAdapter[] getModelAspectAdapters()
  {
    ModelAspectAdapter[] result =
      new ModelAspectAdapter[aspectAdapterList.size()];
    aspectAdapterList.copyInto(result);
    return(result);
  }

  /**
   *  Return a Set containing the ModelAspectIdentifiers for all of the
   *  ModelAspectAdapters managed by the ModelDescriptor.
   */
  protected Set getAspectIds()
  {
    return(aspectAdapterTable.keySet());
  }

  /**
   *  Add a ModelObjectValidator for validating model objects.
   */
  public void addModelObjectValidator(ModelObjectValidator validator)
  {
    modelObjectValidators.add(validator);
  }

  /**
   *  Remove a ModelObjectValidator.
   */
  public void removeModelObjectValidator(ModelObjectValidator validator)
  {
    modelObjectValidators.remove(validator);
  }

  /**
   *  Return an Iterator of this ModelDescriptor's ModelObjectValidators.
   */
  public Iterator getModelObjectValidators()
  {
    return(modelObjectValidators.iterator());
  }

  /**
   *  Return the ModelObjectPropagator for propagating the state of models.
   */
  public ModelObjectPropagator getModelObjectPropagator()
  {
    return(modelObjectPropagator);
  }

  /**
   *  Assign the ModelObjectPropagator for propagating the state of models.
   */
  public void setModelObjectPropagator(ModelObjectPropagator propagator)
  {
    modelObjectPropagator = propagator;
  }


  /**
   *  Validate edits to a model object.
   *  @param modelObjectAdapter the ModelObjectAdapter that holds the model
   *  @exception ModelValidationException if the edited model is not valid
   */
  public void validateModelEdit(ModelObjectAdapter adapter)
    throws ModelValidationException
  {
    for (Iterator i = modelObjectValidators.iterator(); i.hasNext(); )
    {
        ((ModelObjectValidator)i.next()).validateModelEdit(adapter);
    }
  }

  /**
   *  Validate a newly created model object.
   *  @param modelObjectAdapter the ModelObjectAdapter that holds the model
   *  @exception ModelValidationException if the new model is not valid
   */
  public void validateModelCreation(ModelObjectAdapter adapter)
    throws ModelValidationException
  {
    for (Iterator i = modelObjectValidators.iterator(); i.hasNext(); )
    {
        ((ModelObjectValidator)i.next()).validateModelCreation(adapter);
    }
  }

  /**
   *  Validate deletion of a model object.
   *  @param modelObjectAdapter the ModelObjectAdapter that holds the model
   *  @exception ModelValidationException if model cannot be deleted
   */
  public void validateModelDeletion(ModelObjectAdapter adapter)
    throws ModelValidationException
  {
    for (Iterator i = modelObjectValidators.iterator(); i.hasNext(); )
    {
        ((ModelObjectValidator)i.next()).validateModelDeletion(adapter);
    }
  }


  /**
   *  Propagate edits to a model object.
   *  @param modelObjectAdapter the ModelObjectAdapter that holds the model
   *  @exception ModelPropagationException if the edits cannot be propagated
   */
  public void propagateModelEdit(ModelObjectAdapter adapter)
    throws ModelPropagationException
  {
    if (modelObjectPropagator != null)
    {
        modelObjectPropagator.propagateModelEdit(adapter);
    }
  }

  /**
   *  Propagate a newly created model object.
   *  @param modelObjectAdapter the ModelObjectAdapter that holds the model
   *  @exception ModelPropagationException if the create cannot be propagated
   */
  public void propagateModelCreation(ModelObjectAdapter adapter)
    throws ModelPropagationException
  {
    if (modelObjectPropagator != null)
    {
        modelObjectPropagator.propagateModelCreation(adapter);
    }
  }

  /**
   *  Propagate deletion of a model object.
   *  @param modelObjectAdapter the ModelObjectAdapter that holds the model
   *  @exception ModelPropagationException if the delete cannot be propagated
   */
  public void propagateModelDeletion(ModelObjectAdapter adapter)
    throws ModelPropagationException
  {
    if (modelObjectPropagator != null)
    {
        modelObjectPropagator.propagateModelDeletion(adapter);
    }
  }

  ///////////////////////////////////////////////////////////////////////////
  ///
  ///    Representation
  ///
  ///////////////////////////////////////////////////////////////////////////

  private final Class                   modelType;
  private final Object                  editTypeKey;
  private final Vector                  aspectAdapterList;
  private final Hashtable               aspectAdapterTable;

  private final ArrayList               modelObjectValidators = new ArrayList();
  private ModelObjectPropagator         modelObjectPropagator = null;

  private final static Hashtable        descriptorRegistry    = new Hashtable();
}
