/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */
package modelobjects.framework.model;

import java.beans.PropertyVetoException;
import java.util.Hashtable;

/**
 *  ModelObjectAdapter provides a uniform interface for model objects of
 *  different types.  It maintains a reference to an adapted model object,
 *  and a reference to a ModelDescriptor that specifies the public
 *  <i>aspects</i> of the model.  ModelObjectAdapter also provides storage
 *  for <i>annotation</i> aspect values which can be used by either intrinsic
 *  aspects (specified by the ModelDescriptor) or by extrinsice aspects that
 *  are added to the ModelDescriptor by the application.
 *
 *  This class may be subclassed in conjunction with subclasses of
 *  ModelDescriptor, which should return instances of the subclass from
 *  the Factory Method makeModelObjectAdapter().
 */
public class ModelObjectAdapter
{
  /**
   *  Construct a ModelObjectAdapter.
   */
  public ModelObjectAdapter(Object modelObject,
                            ModelDescriptor modelDescriptor)
  {
    if (!modelDescriptor.getModelType().isInstance(modelObject))
    {
        throw new IllegalArgumentException("model is not an instance of " +
                                             modelDescriptor.getModelType() +
                                             " - it is a " +
                                             modelObject.getClass().getName());
    }

    this.modelObject     = modelObject;
    this.modelDescriptor = modelDescriptor;
  }

  /**
   *  Special constructor for ImmutableModelObjectAdapter
   */
  ModelObjectAdapter(ModelDescriptor modelDescriptor)
  {
    if (this instanceof ImmutableModelObjectAdapter) {
      this.modelObject     = null;
      this.modelDescriptor = modelDescriptor;
    }
    else {
      throw(new UnsupportedOperationException
        ("allowed only for ImmutableModelObjectAdapters"));
    }
  }

  /**
   *  Get the model object adapted by the ModelObjectAdapter.
   */
  public Object getModelObject()
  {
    return(modelObject);
  }

  /**
   *  Return the ModelDescriptor for the ModelObjectAdapter.
   */
  public ModelDescriptor getModelDescriptor()
  {
    return(modelDescriptor);
  }

  /**
   *  Return whether the specified aspect is a read-only aspect which cannot
   *  be assigned a new value.
   *
   *  @param aspectId the ModelAspectId of the aspect to check
   *
   *  @throws NoSuchAspectException if no model aspect with the specified
   *    ModelAspectId exists in the ModelDescriptor
   */
  public boolean isReadonlyAspect(ModelAspectId aspectId)
    throws NoSuchAspectException
  {
    ModelAspectAdapter modelAspectAdapter =
      modelDescriptor.getModelAspectAdapter(aspectId);

    return(modelAspectAdapter.isReadonlyAspect());
  }

  /**
   *  Get the value of the named aspect from the model object.
   *
   *  @param aspectId the ModelAspectId of the aspect to return
   *
   *  @throws NoSuchAspectException if no model aspect with the specified
   *    ModelAspectId exists in the ModelDescriptor
   */
  public synchronized Object getAspectValue(ModelAspectId aspectId)
    throws NoSuchAspectException
  {
    ModelAspectAdapter modelAspectAdapter =
      modelDescriptor.getModelAspectAdapter(aspectId);

    return(modelAspectAdapter.getAspectValue(modelObject, this));
  }

  /**
   *  Set the value of the named aspect in the model object to the
   *  specified value.
   *
   *  @param aspectId the ModelAspectId of the aspect to return
   *  @param newAspectValue the new value to assign to the aspect
   *
   *  @throws NoSuchAspectException if no model aspect with the specified
   *    ModelAspectId exists in the ModelDescriptor
   *  @throws PropertyVetoException if the assignment is vetoed
   */
  public synchronized void setAspectValue(ModelAspectId aspectId,
                                          Object newAspectValue)
    throws PropertyVetoException, NoSuchAspectException
  {
    ModelAspectAdapter modelAspectAdapter =
      modelDescriptor.getModelAspectAdapter(aspectId);

    modelAspectAdapter.setAspectValue(modelObject, newAspectValue, this);
  }


  /**
   *  Validate the state of the model object.
   *  This method should be called whenever editing changes are made to a
   *  model that represents an existing, as opposed to newly created, object.
   *
   *  @throws ModelValidationException if the state is not valid.
   */
  public void validateModel()
    throws ModelValidationException
  {
    modelDescriptor.validateModelEdit(this);
  }

  /**
   *  Validate the state of the newly initialized model object.
   *  This method should be called whenever editing changes are made to a
   *  model that represents a newly created and initialized object.
   *
   *  @throws ModelValidationException if the state is not valid.
   */
  public void validateNewModel()
    throws ModelValidationException
  {
    modelDescriptor.validateModelCreation(this);
  }

  /**
   *  Validate a proposed delete of a model object.  This method should
   *  be called whenever a model object is about to be deleted.
   *
   *  @throws ModelValidationException if the model may not be deleted.
   */
  public void validateModelDeletion()
    throws ModelValidationException
  {
    modelDescriptor.validateModelDeletion(this);
  }

  /**
   *  Propagate the edited state of the model object.
   *  This method should be called whenever editing changes are made to a
   *  model that represents an existing, as opposed to newly created, object.
   *
   *  @throws ModelPropagationException if the state cannot be propagated.
   */
  public void propagateModelEdit()
    throws ModelPropagationException
  {
    modelDescriptor.propagateModelEdit(this);
  }

  /**
   *  Propagate the state of the newly initialized model object.
   *  This method should be called whenever editing changes are made to a
   *  model that represents a newly created and initialized object.
   *
   *  @throws ModelPropagationException if the create cannot be propagated.
   */
  public void propagateModelCreation()
    throws ModelPropagationException
  {
    modelDescriptor.propagateModelCreation(this);
  }

  /**
   *  Propagate a proposed delete of a model object.  This method should
   *  be called whenever a model object is about to be deleted.
   *
   *  @throws ModelPropagationException if the delete cannot be propagated.
   */
  public void propagateModelDeletion()
    throws ModelPropagationException
  {
    modelDescriptor.propagateModelDeletion(this);
  }

  /**
   *  Return the cached annotation value for the specified aspect-id, or
   *  null if no value for the aspect-id is present.
   */
  public Object getAnnotationAspectValue(ModelAspectId aspectId)
  {
    if (annotationValues == null)
    {
        return(null);
    }
    else
    {
        return(annotationValues.get(aspectId));
    }
  }

  /**
   *  Return whether or not there is a cached non-null annotation value
   *  for the specified aspect-id.
   */
  public boolean hasAnnotationAspectValue(ModelAspectId aspectId)
  {
    if (annotationValues == null)
    {
        return(false);
    }
    else
    {
        return(annotationValues.containsKey(aspectId));
    }
  }

  /**
   *  Set the cached annotation value for the specified aspect-id.  Returns
   *  the previously cached value, or null if there wasn't one.
   */
  public Object setAnnotationAspectValue(ModelAspectId aspectId, Object value)
  {
    if (annotationValues == null) {
      if (value == null)
    {
        return(null);
    }
      synchronized(this) {
        if (annotationValues == null)
        {
            annotationValues = new Hashtable();
        }
      }
    }

    if (value == null)
    {
        return(annotationValues.remove(aspectId));
    }
    else
    {
        return(annotationValues.put(aspectId, value));
    }
  }


  ///////////////////////////////////////////////////////////////////////////
  ///
  ///  Representation
  ///
  ///////////////////////////////////////////////////////////////////////////

  /**
   *  The model object being adapted.
   */
  protected Object          modelObject;

  /**
   *  The model descriptor for the model object.
   */
  protected ModelDescriptor modelDescriptor;

  /**
   *  Storage for cached values of "annotation" aspects.  It is a Hashtable
   *  that maps ModelAspectId to Object, and is only allocated if used.
   */
  protected Hashtable       annotationValues;
}
