/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */
package modelobjects.framework.swing;

import java.awt.Component;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.util.Enumeration;

import javax.swing.AbstractButton;
import javax.swing.ButtonGroup;
import javax.swing.ButtonModel;

import modelobjects.framework.EditRule;
import modelobjects.framework.ModelEditMediator;
import modelobjects.framework.ViewAspectAdapter;
import modelobjects.framework.ViewValueConverter;
import modelobjects.framework.model.ModelAspectId;

/**
 *  ButtonGroupAdapter is an unusual kind of ViewAspectAdapter that models
 *  an entire ButtonGroup rather than just a single component.
 *  The view-values used by a ButtonGroupAdapter are the ActionCommand
 *  strings associated with the ButtonGroup's buttons.  Therefore, each
 *  button added to the ButtonGroup should have an assigned ActionCommand.
 */
public class ButtonGroupAdapter extends ViewAspectAdapter
  implements ItemListener
{
  /**
   *  Construct a ButtonGroupAdapter for the specified ModelAspectId and
   *  the specified ButtonGroup.
   */
  public ButtonGroupAdapter(ModelAspectId modelAspectId,
                            ButtonGroup buttonGroup)
  {
    super(modelAspectId);
    setButtonGroup(buttonGroup);
    this.editable = true;
  }

  /**
   *  Construct a ButtonGroupAdapter for the specified ModelAspectId,
   *  ButtonGroup, EditRule, ViewValueConverter, and ModelEditMediator.
   */
  public ButtonGroupAdapter(ModelAspectId modelAspectId,
                            ButtonGroup buttonGroup,
                            EditRule editRule,
                            ViewValueConverter viewValueConverter,
                            ModelEditMediator modelEditMediator)
  {
    super(modelAspectId, editRule, viewValueConverter, modelEditMediator);
    setButtonGroup(buttonGroup);
    this.editable = true;
  }

  /**
   *  Set the ButtonGroup of this ButtonGroupAdapter.
   */
  protected final void setButtonGroup(ButtonGroup buttonGroup)
  {
    this.buttonGroup = buttonGroup;
    
    Enumeration buttons = buttonGroup.getElements();
    while (buttons.hasMoreElements()) {
      AbstractButton button = (AbstractButton)buttons.nextElement();
      button.addItemListener(this);
    }
  }

  /**
   *  Returns null, since there is no single Component that really represents
   *  the ButtonGroup.
   */
  @Override
public Component getViewComponent()
  {
    return(null);
  }

  /**
   *  Return whether the buttons in the group are enabled.
   */
  @Override
public boolean isEditable()
  {
    return(editable);
  }

  /**
   *  Enable or disable all of the buttons in the ButtonGroup.
   */
  @Override
public void setEditable(boolean editable)
  {
    if (editable != this.editable) {
      // enable or disable all of the participating buttons
      Enumeration buttons = buttonGroup.getElements();
      while (buttons.hasMoreElements()) {
        AbstractButton button = (AbstractButton)buttons.nextElement();
        button.setEnabled(editable);
      }
      this.editable = editable;
    }
  }

  /**
   *  Delegates to getViewAspectStringValue.
   */
  @Override
public Object getViewAspectValue()
  {
    return(getViewAspectStringValue());
  }

  /**
   *  The view-aspect-string-value of a ButtonGroup is the ActionCommand of the
   *  selected button's ButtonModel, or null if there is no selected button.
   */
  @Override
public String getViewAspectStringValue()
  {
    ButtonModel buttonModel = buttonGroup.getSelection();
    return((buttonModel == null) ? null : buttonModel.getActionCommand());
  }

  /**
   *  Delegates to setViewAspectStringValue.
   */
  @Override
protected void setViewAspectValue(Object viewAspectValue)
    throws IllegalArgumentException
  {
    setViewAspectStringValue((String)viewAspectValue);
  }

  /**
   *  Select the button from the buttonGroup whose action-command matches
   *  the specified view-string-value.
   */
  @Override
protected void setViewAspectStringValue(String viewStringValue)
    throws IllegalArgumentException
  {
    Enumeration buttons = buttonGroup.getElements();
    AbstractButton selectedButton = null;
    while (buttons.hasMoreElements()) {
      AbstractButton button = (AbstractButton)buttons.nextElement();
      String cmd = button.getActionCommand();
      button.setSelected((cmd != null) && cmd.equals(viewStringValue));
    }
  }

  @Override
public void itemStateChanged(ItemEvent e)
  {
    fireChangeEvent();
  }

  private ButtonGroup buttonGroup;
  private boolean     editable;
}
