/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */

package modelobjects.framework.swing;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.awt.event.ContainerEvent;
import java.awt.event.ContainerListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyVetoException;
import java.util.ArrayList;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.DefaultDesktopManager;
import javax.swing.DesktopManager;
import javax.swing.JComponent;
import javax.swing.JDesktopPane;
import javax.swing.JFrame;
import javax.swing.JInternalFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JScrollPane;
import javax.swing.Scrollable;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.event.InternalFrameAdapter;
import javax.swing.event.InternalFrameEvent;

public class DesktopPane extends JDesktopPane
  implements ComponentListener, ContainerListener, Scrollable
{
  private final static  int             CASCADE         = 1;
  private final static  int             TILE            = 2;
  private final static  int             DEFAULT_OFFSETX = 24;
  private final static  int             DEFAULT_OFFSETY = 24;
  private final static  int             UNUSED_HEIGHT   = 48;
  private final static  int             NON_WINDOW_ITEM_COUNT = 3;

  private               JScrollPane     scrollpane;
  private               int             nextX;		// Next X position
  private               int             nextY;		// Next Y position
  private               int             offsetX         = DEFAULT_OFFSETX;
  private               int             offsetY         = DEFAULT_OFFSETY;
  private               int             currentLayoutSetting = 0;
  private               boolean         recomputeSizeInhibited;
  private               ArrayList       internalFrames;
  private               JMenu           internalFramesMenu;
  private       InternalFramesWatcher   internalFramesWatcher;

  /**
   * Constructor
   */
  public DesktopPane()
  {
    super();
    addContainerListener();
    internalFrames = new ArrayList();

    internalFramesMenu = new JMenu("Window");
    JMenuItem tileItem = new JMenuItem(new TileAction());
    JMenuItem cascadeItem = new JMenuItem(new CascadeAction());
    internalFramesMenu.addSeparator();
    internalFramesMenu.add(tileItem);
    internalFramesMenu.add(cascadeItem);

    internalFramesWatcher = new InternalFramesWatcher();
    setDoubleBuffered();

    // replace the desktop manager with a better one (not needed in JDK 1.4)
    //!! setDesktopManager(new ImprovedDesktopMgr());
  }

  private void addContainerListener() {
      addContainerListener(this);
  }
  
  private void setDoubleBuffered() {
      setDoubleBuffered(true);
  }
  
  public JMenu getInternalFramesMenu() {
    return(internalFramesMenu);
  }

  /**
   * This method allows child frames to
   * be added with automatic cascading
   */
  public void addCascaded(Component comp)
  {
    // First add the component in the correct layer
    super.add(comp);
		
    // Now do the cascading
    if (comp instanceof JInternalFrame) {
      this.cascade(comp);
    }

    // Move it to the front
    this.moveToFront(comp);
  }
	
  // Layout all of the children of this container
  // so that they are cascaded.
  public void cascadeAll()
  {
    Component[] comps = getComponents();
    int count = comps.length;
    nextX = 0;
    nextY = 0;

    for (int i = count - 1; i >= 0; i--) {
      Component comp = comps[i];
      if (comp instanceof JInternalFrame && comp.isVisible()) {
        cascade(comp);
      }
    }

    setCurrentLayoutSetting(CASCADE);
  }

  /**
   * Layout all of the children of this container
   * so that they are tiled.
   */
  public void tileAll()
  {
    DesktopManager manager = getDesktopManager();
    if (manager == null) {
      // No desktop manager - do nothing
      return;
    }
			
    Component[] comps = getComponents();
    Component comp;
    int count = 0;

    // Count and handle only the internal frames
    for (int i = 0; i < comps.length; i++) {
      comp = comps[i];
      if (comp instanceof JInternalFrame && comp.isVisible()) {
        count++;
      }
    }

    if (count != 0) {
      double root = Math.sqrt((double)count);
      int rows = (int)root;
      int columns = count/rows;
      int spares = count - (columns * rows);

      Dimension paneSize = getSize();
      int columnWidth = paneSize.width/columns;

      // We leave some space at the bottom that doesn't get covered
      int availableHeight = paneSize.height - UNUSED_HEIGHT;
      int mainHeight = availableHeight/rows;
      int smallerHeight = availableHeight/(rows + 1);
      int rowHeight = mainHeight;
      int x = 0;
      int y = 0;
      int thisRow = rows;
      int normalColumns = columns - spares;

      for (int i = comps.length - 1; i >= 0; i--) {
        comp = comps[i];
        if (comp instanceof JInternalFrame && comp.isVisible()) {
          manager.setBoundsForFrame((JComponent)comp, x, y,
                                    columnWidth, rowHeight);
          y += rowHeight;
          if (--thisRow == 0) {
            // Filled the row
            y = 0;
            x += columnWidth;

            // Switch to smaller rows if necessary
            if (--normalColumns <= 0) {
              thisRow = rows + 1;
              rowHeight = smallerHeight;
            }
            else {
              thisRow = rows;
            }
          }
        }
      }
    }

    setCurrentLayoutSetting(TILE);
  }

  /**
   *
   */
  public void setCascadeOffsets(int offsetX, int offsetY) {
    this.offsetX = offsetX;
    this.offsetY = offsetY;
  }

  /**
   *
   */
  public void setCascadeOffsets(Point pt) {
    this.offsetX = pt.x;
    this.offsetY = pt.y;
  }

  /**
   *
   */
  public Point getCascadeOffsets()
  {
    return new Point(offsetX, offsetY);
  }

  /**
   *  Place a component so that it is cascaded
   *  relative to the previous one
   */
  protected void cascade(Component comp)
  {
    Dimension paneSize = getSize();

    int targetWidth = 3 * paneSize.width/4;
    int targetHeight = 3 * paneSize.height/4;

    DesktopManager manager = getDesktopManager();
    if (manager == null) {
      comp.setBounds(0, 0, targetWidth, targetHeight);
      return;
    }
		
    if (nextX + targetWidth > paneSize.width ||
        nextY + targetHeight > paneSize.height) {
      nextX = 0;
      nextY = 0;
    }

    manager.setBoundsForFrame((JComponent)comp, nextX, nextY,
                              targetWidth, targetHeight);
		
    nextX += offsetX;
    nextY += offsetY;
  }

  void recomputeSize()
  {
    if (recomputeSizeInhibited)
    {
        return;
    }

    try {
      recomputeSizeInhibited = true;

      Dimension viewportSize =
        ((scrollpane == null) ? new Dimension(0, 0) :
         scrollpane.getViewport().getSize());

      int width  = viewportSize.width;
      int height = viewportSize.height;

      Component comps[] = getComponents();
      for (int i = 0, n = comps.length; i < n; i++) {
        Component comp = comps[i];
        if (comp.isVisible()) {
          Rectangle r = comp.getBounds();
          width  = Math.max(width,  r.x + r.width);
          height = Math.max(height, r.y + r.height);
        }
      }

      Dimension size = getSize();
      if ((width != size.width) || (height != size.height)) {
        Dimension newSize = new Dimension(width, height);
        setSize(newSize);
        setPreferredSize(newSize);
      }

      if (scrollpane != null) {
        scrollpane.invalidate();
        scrollpane.validate();
      }
    }
    finally {
      recomputeSizeInhibited = false;
    }
  }

  @Override
public void componentAdded(ContainerEvent event)
  {
    Component comp = event.getChild();
    comp.addComponentListener(this);

    if (comp instanceof JInternalFrame) {
      if (!internalFrames.contains(comp)) {
        JInternalFrame iframe = (JInternalFrame)comp;
        internalFrames.add(iframe);
        //internalFramesMenu.add(new OpenInternalFrameAction(iframe));
        internalFramesMenu.insert(new OpenInternalFrameAction(iframe),internalFramesMenu.getItemCount()- NON_WINDOW_ITEM_COUNT);
        assignMnemonics(internalFramesMenu, null);
        iframe.addInternalFrameListener(internalFramesWatcher);
      }
    }

    recomputeSize();
  }

  @Override
public void componentRemoved(ContainerEvent event)
  {
    Component comp = event.getChild();
    comp.removeComponentListener(this);
    recomputeSize();
  }

  @Override
public void componentMoved(ComponentEvent event)
  {
    recomputeSize();
  }

  @Override
public void componentResized(ComponentEvent event)
  {
    recomputeSize();
  }

  @Override
public void componentShown(ComponentEvent event)
  {
    recomputeSize();
  }

  @Override
public void componentHidden(ComponentEvent event)
  {
    recomputeSize();
  }

  @Override
public Dimension getPreferredScrollableViewportSize()
  {
    return(new Dimension(700, 500));
  }

  @Override
public int getScrollableUnitIncrement(Rectangle visibleRect,
                                        int orientation, int direction)
  {
    boolean vertical = (orientation == SwingConstants.VERTICAL);
    return(vertical ? 25 : 40);
  }

  @Override
public int getScrollableBlockIncrement(Rectangle visibleRect,
                                         int orientation, int direction)
  {
    int fullSize = ((orientation == SwingConstants.VERTICAL) ?
                    visibleRect.height : visibleRect.width);
    return(Math.max(10, (int)(fullSize * 0.9)));
  }

  @Override
public boolean getScrollableTracksViewportWidth()
  {
    return(false);
  }

  @Override
public boolean getScrollableTracksViewportHeight()
  {
    return(false);
  }

  @Override
public void setBounds(int x, int y, int w, int h)
  {
    if ((x != getX()) || (y != getY()) ||
        (w != getWidth()) || (h != getHeight())) {
      super.setBounds(x, y, w, h);

      // wiggle things again so the scrollpane really lays out correctly
      recomputeSize();
    }
  }

  private void refreshSettings()
  {
    if(getCurrentLayoutSetting() == CASCADE)
    {
        cascadeAll();
    }
    else if(getCurrentLayoutSetting() == TILE)
    {
        tileAll();
    }
  }

  private void setCurrentLayoutSetting(int setting)
  {
    currentLayoutSetting = setting;
  }

  private int getCurrentLayoutSetting()
  {
    return(currentLayoutSetting);
  }

  @Override
public void addNotify()
  {
    super.addNotify();
    
    // if we're in a JScrollPane (and we probably are) keep track of it
    JScrollPane scrollpane =
      (JScrollPane)SwingUtilities.getAncestorOfClass(JScrollPane.class, this);
    if ((scrollpane != null) && (getParent().getParent() == scrollpane)) {
      this.scrollpane = scrollpane;
      scrollpane.getViewport().addComponentListener(this);
    }
  }

  void removeMenuItemFor(JInternalFrame internalFrame)
  {
    JMenu menu = internalFramesMenu;

    String name = internalFrame.getTitle();
    int itemCount = menu.getItemCount() - NON_WINDOW_ITEM_COUNT;
    for (int i = 0; i < itemCount; i++) {
      JMenuItem menuItem = menu.getItem(i);
      if (name.equals(menuItem.getText())) {
        menu.remove(menuItem);
        break;
      }
    }
  }

  class InternalFramesWatcher extends InternalFrameAdapter
  {
    @Override
    public void internalFrameClosed(InternalFrameEvent e)
    {
      JInternalFrame iframe = e.getInternalFrame();
      internalFrames.remove(iframe);
      removeMenuItemFor(iframe);
    }
  }

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  menu utilities
  ///
  /////////////////////////////////////////////////////////////////////////////

  /**
   *  Automatically assigns keyboard mnemonics to everything in the specified
   *  MenuContainer and all of its components.
   */
  public static void assignMnemonics(JMenuBar menubar)
  {

    String usedChars = "";
    int count = menubar.getMenuCount();
    for (int i = 0; i < count; i++) {
      JMenu menu = menubar.getMenu(i);
      usedChars = assignMnemonics(menu, usedChars);
    }
  }

  public static String assignMnemonics(JMenuItem menuItem, String usedChars)
  {

    if (menuItem instanceof JMenu) {
      JMenu menu = (JMenu)menuItem;
      String menuUsedChars = "";
      int itemCount = menu.getItemCount() - NON_WINDOW_ITEM_COUNT;
      for (int i = 0; i < itemCount; i++) {
        JMenuItem item = menu.getItem(i);
        if (item != menuItem)
        {
            menuUsedChars = assignMnemonics(item, menuUsedChars);
        }
      }
    }

    if (usedChars != null) {
      String label = menuItem.getText();
      int numChars = ((label == null) ? 0 : label.length());

      for (int i = 0; i < numChars; i++) {
        char c;
        c = label.charAt(i);
        char upper = Character.toUpperCase(c);
        int pos = usedChars.indexOf(upper);
        if (pos == -1) {
          menuItem.setMnemonic(c);
          return(usedChars + upper);
        }
      }
    }
    
    return(usedChars);
  }


  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  unit test
  ///
  /////////////////////////////////////////////////////////////////////////////

  public static void main(String[] args)
  {
    JFrame window = new JFrame("DesktopPane Test");
    window.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    window.setBounds(300, 300, 700, 400);

    DesktopPane desktopPane = new DesktopPane();
    desktopPane.setSize(800, 600);
    JScrollPane desktopScroller = new JScrollPane(desktopPane);
    window.getContentPane().add(desktopScroller, BorderLayout.CENTER);

    JMenuBar menubar = new JMenuBar();
    JMenu windowsMenu = desktopPane.internalFramesMenu;
    windowsMenu.setMnemonic('w');
    menubar.add(windowsMenu);
    window.setJMenuBar(menubar);

    window.validate();
    window.setVisible(true);

    JInternalFrame f1 = new JInternalFrame("Fred", true, true, true, true);
    f1.setBounds(100, 100, 200, 200);
    // f1.setBackground(Color.blue);
    desktopPane.add(f1);
    f1.setVisible(true);

    JInternalFrame f2 = new JInternalFrame("Barney", true, true, true, true);
    f2.setBounds(300, 300, 200, 200);
    // f1.setBackground(Color.blue);
    desktopPane.add(f2);
    f2.setVisible(true);
  }


  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  inner class OpenInternalFrameAction
  ///
  /////////////////////////////////////////////////////////////////////////////

  static class OpenInternalFrameAction extends AbstractAction
    implements PropertyChangeListener
  {
    OpenInternalFrameAction(JInternalFrame internalFrame)
    {
      super(internalFrame.getTitle());

      this.internalFrame = internalFrame;
      internalFrame.addPropertyChangeListener(this);
    }

    @Override
    public void actionPerformed(ActionEvent event)
    {
      try {
        internalFrame.setIcon(false);
        internalFrame.show();
        internalFrame.setSelected(true);
      }
      catch (PropertyVetoException veto) {
        // cannot uniconify
      }
    }

    @Override
    public void propertyChange(PropertyChangeEvent event)
    {
      if (JInternalFrame.TITLE_PROPERTY.equals(event.getPropertyName())) {
        putValue(NAME, (String)event.getNewValue());
      }
    }

    private JInternalFrame internalFrame;
  }


  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  inner class ImprovedDesktopMgr is just supposed to make things faster
  ///
  /////////////////////////////////////////////////////////////////////////////

  static class ImprovedDesktopMgr extends DefaultDesktopManager
  {
    private transient JComponent          modComp;
    private transient Rectangle           modRect;
    private transient JDesktopPane        desktop;
    private transient Graphics            graphics;

    public ImprovedDesktopMgr()
    {
      super();
      modRect  = null;
      modComp  = null;
      graphics = null;
    }

    @Override
    public void beginDraggingFrame(JComponent f)
    {
      startMods(f);
      drawModBounds();
    }

    @Override
    public void dragFrame(JComponent f, int newX, int newY)
    {
      if (modComp == f) {
          drawModBounds();
          modRect.x = newX;
          modRect.y = newY;
          drawModBounds();
      }
      else {
        System.out.println("Wrong component!");
      }
    }

    @Override
    public void endDraggingFrame(JComponent f)
    {
      if (modComp == f) {
        drawModBounds();

        setBoundsForFrame(f, modRect.x, modRect.y,
                          modRect.width, modRect.height);
        modComp = null;
        modRect = null;
        graphics.dispose();
        graphics = null;
        desktop.repaint();
        desktop = null;
      }
      else {
        System.out.println("Wrong component!");
      }
    }

    @Override
    public void beginResizingFrame(JComponent f, int direction)
    {
      startMods(f);
      drawModBounds();
    }

    @Override
    public void resizeFrame(JComponent f,
                            int newX, int newY, int newWidth, int newHeight)
    {
      if (modComp == f) {
        drawModBounds();
        modRect.x = newX;
        modRect.y = newY;
        modRect.width = newWidth;
        modRect.height = newHeight;
        drawModBounds();
      }
      else {
        System.out.println("Wrong component!");
      }
    }

    @Override
    public void endResizingFrame(JComponent f)
    {
      if (modComp == f) {
        drawModBounds();
        setBoundsForFrame(f, modRect.x, modRect.y,
                          modRect.width, modRect.height);
        modComp = null;
        modRect = null;
        graphics.dispose();
        graphics = null;
        desktop.repaint();
        desktop = null;
      }
      else {
        System.out.println("Wrong component!");
      }
    }
  
    private void startMods(JComponent f)
    {
      modRect  = f.getBounds();
      modComp  = f;
      desktop  = (JDesktopPane)
        SwingUtilities.getAncestorOfClass(JDesktopPane.class, f);
      graphics = desktop.getGraphics();
      graphics.setXORMode(Color.white);
    }

    private void drawModBounds()
    {
      Rectangle r = modRect;
      for (int i = 0; i < 4; i++)
    {
        graphics.drawRect(r.x + i, r.y + i, r.width - 2*i, r.height - 2*i);
    }
    }
  }


  /***********
   *INNERCLASS
   *
   **********/
  class TileAction extends AbstractAction {

    public TileAction() {
        putProperties();
    }
    
    private void putProperties() {
        putValue(Action.NAME, "Tile");
        //putValue(Action.SMALL_ICON, getIcon(SMALL_ICON_EXIT));
        //putValue(LARGE_ICON, getIcon(LARGE_ICON_EXIT));
        putValue(Action.SHORT_DESCRIPTION, "Tile Windows");
        putValue(Action.LONG_DESCRIPTION, "Tile Windows");
        putValue(Action.MNEMONIC_KEY, new Integer('T'));
        putValue(Action.ACTION_COMMAND_KEY, "tile-command");
    }

    @Override
    public void actionPerformed(ActionEvent event) {
      tileAll();
    }

  }
  
  

  /***********
   *INNERCLASS
   *
   **********/
  class CascadeAction extends AbstractAction {

    public CascadeAction() {
        putProperties();
    }

    private void putProperties() {
        putValue(Action.NAME, "Cascade");
        //putValue(Action.SMALL_ICON, getIcon(SMALL_ICON_EXIT));
        //putValue(LARGE_ICON, getIcon(LARGE_ICON_EXIT));
        putValue(Action.SHORT_DESCRIPTION, "Cascade windows");
        putValue(Action.LONG_DESCRIPTION, "Cascade windows");
        putValue(Action.MNEMONIC_KEY, new Integer('C'));
        putValue(Action.ACTION_COMMAND_KEY, "cascade-command");
    }
    @Override
    public void actionPerformed(ActionEvent event) {
      cascadeAll();
    }


  }
}
