/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */
package modelobjects.framework.swing;

import java.awt.Component;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;

import javax.swing.ComboBoxEditor;
import javax.swing.JComboBox;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.text.JTextComponent;

import modelobjects.framework.EditRule;
import modelobjects.framework.ModelEditMediator;
import modelobjects.framework.ViewAspectAdapter;
import modelobjects.framework.ViewValueConverter;
import modelobjects.framework.model.ModelAspectId;

public class JComboBoxAdapter extends ViewAspectAdapter
  implements ItemListener, DocumentListener
{
  public JComboBoxAdapter(ModelAspectId modelAspectId,
                          JComboBox viewComponent)
  {
    super(modelAspectId);
    viewComponent.setName(modelAspectId.toString());
    setViewComponent(viewComponent);
  }

  public JComboBoxAdapter(ModelAspectId modelAspectId,
                          JComboBox viewComponent,
                          EditRule editRule,
                          ViewValueConverter viewValueConverter,
                          ModelEditMediator modelEditMediator)
  {
    super(modelAspectId, editRule, viewValueConverter, modelEditMediator);
    viewComponent.setName(modelAspectId.toString());
    setViewComponent(viewComponent);
  }

  private void setViewComponent(JComboBox viewComponent)
  {
    //if (viewComponent.isEditable())
    //throw(new IllegalArgumentException("Cannot adapt editable JComboBox"));

    this.viewComponent = viewComponent;

    ViewValueConverter converter = getViewValueConverter();
    if (converter != null)
      {
        String[] viewStringValues = converter.getViewStringValues();
        if (viewStringValues != null)
        {
            setViewStringValues(viewStringValues);
        }
      }

    // add the listener after items are added so we don't fire bogus events
    viewComponent.addItemListener(this);

    // always set a document listener on the editor so that we
    // can catch changes via the keyboard (if the combobox is editable)
    JTextComponent tc = getEditorTextComponent();
    if (tc != null)
    {
        tc.getDocument().addDocumentListener(this);
    }
  }

  @Override
public Component getViewComponent()
  {
    return(viewComponent);
  }

  @Override
public boolean isEditable()
  {
    return(viewComponent.isEnabled());
  }

  @Override
public void setEditable(boolean editable)
  {
    viewComponent.setEnabled(editable);
  }

  private JTextComponent getEditorTextComponent()
  {
    ComboBoxEditor editor = viewComponent.getEditor();

    if (editor.getEditorComponent() instanceof JTextComponent)
    {
        return (JTextComponent)editor.getEditorComponent();
    }

    return null;
  }

  @Override
public Object getViewAspectValue()
  {
    return(viewComponent.getSelectedItem());
  }

  @Override
public String getViewAspectStringValue()
  {
    return(viewComponent.getSelectedItem().toString());
  }

  @Override
protected void setViewAspectValue(Object viewAspectValue)
    throws IllegalArgumentException
  {
    if (viewAspectValue == null)
    {
        return;
    }
    if (viewAspectValue instanceof String &&
        ((String)viewAspectValue).length() <= 0)
    {
        return;
    }

    viewComponent.setSelectedItem(viewAspectValue);
  }

  @Override
protected void setViewAspectStringValue(String viewStringValue)
    throws IllegalArgumentException
  {
    if (viewStringValue == null ||
        viewStringValue.length() <= 0)
    {
        return;
    }

    viewComponent.setSelectedItem(viewStringValue);
  }

  public void setViewStringValues(String viewStringValues[])
  {
    viewComponent.removeAllItems();
    int n = ((viewStringValues == null) ? 0 : viewStringValues.length);
    for (int i = 0; i < n; i++)
    {
        viewComponent.addItem(viewStringValues[i]);
    }
  }

  //  ItemListener interface methods

  @Override
public void itemStateChanged(ItemEvent event)
  {
    // For every change in selection the combobox fires TWO events (deselected
    // and selected) - we should only issue a single ChangeEvent for the
    // selected change.
    if (event.getStateChange() == ItemEvent.SELECTED)
    {
        fireChangeEvent();
    }
  }

  //  DocumentListener interface methods

  @Override
public void changedUpdate(DocumentEvent event)
  {
    fireChangeEvent();
  }

  @Override
public void insertUpdate(DocumentEvent event)
  {
    fireChangeEvent();
  }

  @Override
public void removeUpdate(DocumentEvent event)
  {
    fireChangeEvent();
  }

  private JComboBox viewComponent;
}
