/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */
package modelobjects.framework.swing;

import java.awt.Component;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JTable;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;

import modelobjects.framework.EditRule;
import modelobjects.framework.ModelEditMediator;
import modelobjects.framework.ViewAspectAdapter;
import modelobjects.framework.model.ModelAspectId;
import modelobjects.util.ObjectCopier;
import modelobjects.util.swing.IModelTableModel;

/**
 *  JTableListAspectAdapter is a special kind of ViewAspectAdapter that
 *  is used for List-valued model-aspects that are edited in JTables.
 *  This class should be used only with care, since it must make a complete
 *  copy of the model-aspect List and of the contents of the List in order
 *  to be sure that edits made in the table (inserts, deletes, cell-edits)
 *  are both discardable in the case of reset, and undoable and redoable.
 *
 *  Note that JTableListAspectAdapter can only be used with JTables that
 *  use ModelListTableModel for their TableModels.
 *
 *  JTableListAspectAdapter supports PropertyChangeListeners that may want
 *  to track when it is editable in order to enable/disable editing actions.
 */
public class JTableListAspectAdapter extends ViewAspectAdapter
  implements TableModelListener
{
  /**
   *  Construct a JTableListAspectAdapter.
   *
   *  @param modelAspectId the ModelAspectId of the model-aspect being edited
   *  @param table the JTable component in which the editing occurs
   *  @param tableModel the ModelListTableModel of the JTable
   *  @param rowModelCopier the ObjectCopier used to make copies of row models
   *  @param editRule the editRule for the ViewAspectAdapter
   *  @param modelEditMediator the ModelEditMediator managing this adapter
   */
  public JTableListAspectAdapter(ModelAspectId     modelAspectId,
                                 JTable            table,
                                 IModelTableModel  tableModel,
                                 ObjectCopier      rowModelCopier,
                                 EditRule          editRule,
                                 ModelEditMediator modelEditMediator)
  {
    super(modelAspectId, editRule, null, modelEditMediator);
    table.setName(modelAspectId.toString());
    this.table          = table;
    this.tableModel     = tableModel;
    this.rowModelCopier = rowModelCopier;

    tableModel.addTableModelListener(this);
    propertyChangeSupport = new PropertyChangeSupport(this);
  }

  @Override
public Component getViewComponent()
  {
    return(table);
  }

  @Override
public boolean isEditable()
  {
    return(editable);
  }

  @Override
public void setEditable(boolean editable)
  {
    boolean wasEditable = this.editable;
    this.editable = editable;
    if (editable != wasEditable)
    {
        propertyChangeSupport.firePropertyChange(EDITABLE_PROPERTY,
                                                   wasEditable, editable);
    }
  }

  @Override
public Object getViewAspectValue()
  {
    return(tableModel.getContents());
  }

  @Override
public String getViewAspectStringValue()
  {
    throw(new IllegalArgumentException("String values not supported"));
  }

  @Override
protected void setViewAspectValue(Object viewAspectValue)
    throws IllegalArgumentException
  {
    List elems = (List)viewAspectValue;
    int numElems = ((elems == null) ? 0 : elems.size());

    ArrayList contents = new ArrayList(numElems);
    for (int i = 0; i < numElems; i++) {
      contents.add(rowModelCopier.copyObject(elems.get(i)));
    }

    tableModel.setContents(contents);
  }

  @Override
protected void setViewAspectStringValue(String viewStringValue)
    throws IllegalArgumentException
  {
    throw(new IllegalArgumentException("String values not supported"));
  }

  @Override
public void tableChanged(TableModelEvent e)
  {
    // ignore TableStructureChange event - data hasn't changed
    if ((e.getFirstRow() == TableModelEvent.HEADER_ROW) &&
        (e.getLastRow() == TableModelEvent.HEADER_ROW) &&
        (e.getColumn() == TableModelEvent.ALL_COLUMNS))
    {
        return;
    }

    fireChangeEvent();
  }

  public void addPropertyChangeListener(PropertyChangeListener listener)
  {
    propertyChangeSupport.addPropertyChangeListener(listener);
  }

  public void removePropertyChangeListener(PropertyChangeListener listener)
  {
    propertyChangeSupport.removePropertyChangeListener(listener);
  }

  private JTable               table;
  protected IModelTableModel   tableModel;
  protected ObjectCopier       rowModelCopier;
  private boolean              editable;

  private PropertyChangeSupport propertyChangeSupport;

  public  static final String   EDITABLE_PROPERTY       = "editable";
}
