/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */
package modelobjects.framework.swing;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.PrintWriter;
import java.io.Serializable;
import java.io.StringWriter;
import java.text.DateFormat;
import java.util.Date;
import java.util.TimeZone;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.swing.Action;
import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextPane;
import javax.swing.KeyStroke;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.event.CaretEvent;
import javax.swing.event.CaretListener;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.text.BadLocationException;
import javax.swing.text.Document;
import javax.swing.text.JTextComponent;
import javax.swing.text.Keymap;
import javax.swing.text.SimpleAttributeSet;
import javax.swing.text.Style;
import javax.swing.text.StyleConstants;
import javax.swing.text.StyledDocument;
import javax.swing.text.TabSet;
import javax.swing.text.TabStop;
import javax.swing.text.TextAction;

/**
 *  MessagePane is a special Swing component for displaying messages of
 *  various kinds.  It is a subclass of JTextPane, designed to be used in
 *  any of a variety of contexts.
 */
public class MessagePane extends JTextPane
{
  /**
   *  Construct a MessagePane
   */
  public MessagePane()
  {
    super();
    prepareMessagePane();
  }
  
  private void prepareMessagePane() {
    setEditable(false);

    initActions();
    initTextStyles();

    TextEventHandler eventHandler = new TextEventHandler();
    addCaretListener(eventHandler);
    getDocument().addDocumentListener(eventHandler);
  }

  /**
   *  Return this MessagePane's cut-action.
   */
  public Action getCutAction()
  {
    return(cutAction);
  }

  /**
   *  Return this MessagePane's copy-action.
   */
  public Action getCopyAction()
  {
    return(copyAction);
  }

  /**
   *  Return this MessagePane's clear-action.
   */
  public Action getClearAction()
  {
    return(clearAction);
  }

  /**
   *  Return this MessagePane's select-all-action.
   */
  public Action getSelectAllAction()
  {
    return(selectAllAction);
  }

  /**
   *  Add a status message, possibly accompanied by an Throwable.
   */
  public synchronized
  void addStatusMessage(String message, Throwable throwable)
  {
    addMessage(this, getIcon(STATUS_ICON), getStyle(STATUS_STYLE),
               message, throwable);
  }

  /**
   *  Add an informatory message, possibly accompanied by an Throwable.
   */
  public synchronized
  void addInformatoryMessage(String message, Throwable throwable)
  {
    addMessage(this, getIcon(INFORM_ICON), getStyle(INFORMATION_STYLE),
               message, throwable);
  }

  /**
   *  Add a warning message, possibly accompanied by an Throwable.
   */
  public synchronized
  void addWarningMessage(String message, Throwable throwable)
  {
    addMessage(this, getIcon(WARNING_ICON), getStyle(WARNING_STYLE),
               message, throwable);
  }

  /**
   *  Add an error message, possibly accompanied by an Throwable.
   */
  public synchronized
  void addErrorMessage(String message, Throwable throwable)
  {
    addMessage(this, getIcon(ERROR_ICON), getStyle(ERROR_STYLE),
               message, throwable);
  }

  /**
   *  Append to the previous message, possibly accompanied by an Throwable.
   *  This method will not add a new message-icon.
   */
  public synchronized
  void addToLastMessage(String message, Throwable throwable)
  {
    SwingUtilities.invokeLater(new MessageAppender(message, throwable));
  }

  protected void initActions()
  {
    Action[] actions = this.getActions();

    clearAction = new TextAction("Clear") {
        @Override
        public void actionPerformed(ActionEvent event) {
          JTextComponent textComp = getTextComponent(event);
          textComp.selectAll();
          textComp.setText("");

          // remove any buttons (or other components) that were left behind
          for (int i = getComponentCount() - 1; i >= 0; i--) {
            Component comp = getComponent(i);
            if (comp instanceof JButton)
            {
                remove(comp);
            }
          }
        }
      };

    copyAction = new TextAction("Copy") {
        @Override
        public void actionPerformed(ActionEvent event) {
          getTextComponent(event).copy();
        }
      };

    cutAction = new TextAction("Cut") {
        @Override
        public void actionPerformed(ActionEvent event) {
          JTextComponent textComp = getTextComponent(event);
          textComp.copy();
          Document doc = textComp.getDocument();
          try {
            int p1 = textComp.getSelectionStart();
            int p2 = textComp.getSelectionEnd();
            doc.remove(Math.min(p1, p2), Math.abs(p2-p1));
          }
          catch (BadLocationException badLoc) {
          }
        }
      };

    selectAllAction = new TextAction("Select All") {
        @Override
        public void actionPerformed(ActionEvent event) {
          getTextComponent(event).selectAll();
          getTextComponent(event).requestFocus();
        }
      };
  }

  private KeyStroke getActionKeyBinding(Action action)
  {
    Keymap keymap = this.getKeymap();
    KeyStroke boundKeys[] = keymap.getBoundKeyStrokes();
    int numKeyStrokes = ((boundKeys == null) ? 0 : boundKeys.length);

    for (int i = 0; i < numKeyStrokes; i++) {
      if (keymap.getAction(boundKeys[i]) == action)
    {
        return(boundKeys[i]);
    }
    }

    return(null);
  }

  /**
   *  Initialize the text styles used for formatting messages
   */
  protected void initTextStyles()
  {
    final int numTabStops = 8;
    TabStop tabStops[] = new TabStop[numTabStops];
    for (int i = 0; i < numTabStops; i++)
    {
        tabStops[i] = new TabStop(tab * i);
    }
    TabSet tabs = new TabSet(tabStops);

    MessagePane textPane = this;

    Style baseStyle = textPane.getLogicalStyle();
    Style normalStyle = textPane.addStyle(NORMAL_STYLE, baseStyle);

    StyleConstants.setFontSize    (normalStyle, normalFontSize);
    StyleConstants.setFontFamily  (normalStyle, "SansSerif");
    StyleConstants.setBold        (normalStyle, false);
    StyleConstants.setItalic      (normalStyle, false);
    StyleConstants.setAlignment   (normalStyle, alignment);
    StyleConstants.setRightIndent (normalStyle, rightMargin);
    StyleConstants.setLeftIndent  (normalStyle, margin);
    StyleConstants.setTabSet      (normalStyle, tabs);
    StyleConstants.setSpaceAbove  (normalStyle, 8);
    StyleConstants.setSpaceBelow  (normalStyle, 0);

    Style indented = textPane.addStyle(INDENT_STYLE, normalStyle);
    StyleConstants.setLeftIndent(indented, tab);

    // declare the other styles
    Style style = textPane.addStyle(STATUS_STYLE, indented);
    StyleConstants.setForeground(style, Color.black);

    style = textPane.addStyle(INFORMATION_STYLE, indented);
    StyleConstants.setForeground(style, Color.black);
    StyleConstants.setItalic(style, true);
    
    style = textPane.addStyle(WARNING_STYLE, indented);
    StyleConstants.setBold(style, true);
    StyleConstants.setItalic(style, true);
    StyleConstants.setForeground(style, Color.blue);

    style = textPane.addStyle(ERROR_STYLE, indented);
    StyleConstants.setBold(style, true);
    StyleConstants.setFontSize(style, normalFontSize + 2);
    StyleConstants.setForeground(style, Color.red);

    style = textPane.addStyle(STACKTRACE_STYLE, normalStyle);
    StyleConstants.setFontFamily(style, "Monospaced");
    StyleConstants.setFontSize(style, normalFontSize - 2);
    StyleConstants.setForeground(style, Color.blue);
    StyleConstants.setBold(style, false);
    StyleConstants.setItalic(style, false);
    StyleConstants.setSpaceAbove(style, 0);
    StyleConstants.setSpaceBelow(style, 0);
    StyleConstants.setLeftIndent(style, margin + tab);

    style = textPane.addStyle(HEADER_STYLE, style);
    StyleConstants.setForeground(style, Color.darkGray);
    StyleConstants.setSpaceAbove(style, 16);
    StyleConstants.setLeftIndent(style, margin);
  }

  /**
   *  Return the named Icon corresponding to a message style.
   */
  protected Icon getIcon(String iconName)
  {
    if (!iconsReady) {
      informIcon  = UIManager.getIcon(INFORM_ICON);
      statusIcon  = UIManager.getIcon(STATUS_ICON);
      warningIcon = UIManager.getIcon(WARNING_ICON);
      errorIcon   = UIManager.getIcon(ERROR_ICON);
      iconsReady  = true;
    }

    if (iconName.equals(STATUS_ICON))
    {
        return(statusIcon);
    }
    else if (iconName.equals(INFORM_ICON))
    {
        return(informIcon);
    }
    else if (iconName.equals(WARNING_ICON))
    {
        return(warningIcon);
    }
    else if (iconName.equals(ERROR_ICON))
    {
        return(errorIcon);
    }
    else
    {
        return(null);
    }
  }

  /**
   *  Add a new message with the specified icon and style, possibly
   *  accompanied by an throwable.
   */
  protected void addMessage(JTextPane textPane, Icon icon, Style style,
                            String msg, Throwable throwable)
  {
    // don't write "ready" messages to log
    if ((msg == null) || msg.equals("") || msg.equals("Ready"))
    {
        return;
    }

    if ((System.getProperty("NOMESSAGELOG") != null)) {
      System.out.println(msg);
    }
    else {
      SwingUtilities.invokeLater(new MessageAdder(textPane, icon, style,
                                                  msg, throwable));
    }
  }

  protected void addStackTraceButton(int textStart, Icon icon, Style style,
                                     String message, Throwable throwable)
  {
    String label = "Show Stack Trace...";
    JButton button = new JButton(label);
    button.setOpaque(true);
    button.addActionListener(new StackTraceButtonHandler(icon, style,
                                                         message, throwable));
    button.putClientProperty(EXCEPTION_PROP, throwable);
    button.putClientProperty(MESSAGE_PROP, message);

    SimpleAttributeSet componentAttrs = new SimpleAttributeSet();
    StyleConstants.setComponent(componentAttrs, button);
    Document doc = getStyledDocument();
    int end = doc.getLength();
    setCaretPosition(end);

    try {
      Style prevStyle = getLogicalStyle();
      doc.insertString(end, " ", componentAttrs);
      doc.insertString(doc.getLength(), "\n", prevStyle);
    }
    catch (BadLocationException badLoc) {
      badLoc.printStackTrace();
    }     
  }

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  representation
  ///
  /////////////////////////////////////////////////////////////////////////////

  protected Action              cutAction;
  protected Action              copyAction;
  protected Action              clearAction;
  protected Action              selectAllAction;

  private transient Icon        informIcon;
  private transient Icon        statusIcon;
  private transient Icon        warningIcon;
  private transient Icon        errorIcon;
  private transient boolean     iconsReady      = false;

  private final static int      alignment       = StyleConstants.ALIGN_LEFT;
  private final static int      normalFontSize  = 14;
  private final static float    tab             = 40;
  private final static float    margin          = 2;
  private final static float    rightMargin     = 20;

  private final static String   INFORM_ICON     = "OptionPane.informationIcon";
  private final static String   STATUS_ICON     = "none";
  private final static String   WARNING_ICON    = "OptionPane.warningIcon";
  private final static String   ERROR_ICON      = "OptionPane.errorIcon";

  private final static String   EXCEPTION_PROP  = "exception";
  private final static String   MESSAGE_PROP    = "message";

  private final static String   NORMAL_STYLE            = "normal";
  private final static String   INDENT_STYLE            = "indented";
  private final static String   INFORMATION_STYLE       = "inform";
  private final static String   STATUS_STYLE            = "status";
  private final static String   WARNING_STYLE           = "warning";
  private final static String   ERROR_STYLE             = "error";
  private final static String   STACKTRACE_STYLE        = "stacktrace";
  private final static String   HEADER_STYLE            = "headerInfo";


  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  unit test
  ///
  /////////////////////////////////////////////////////////////////////////////

  public static void main(String[] args)
    throws Exception
  {
    JFrame window = new JFrame("MessagePane Test");
    window.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    window.setBounds(200, 200, 600, 500);
    MessagePane msgPane = new MessagePane();

    JPanel buttonPanel = new JPanel();
    buttonPanel.add(new JButton(msgPane.getCutAction()));
    buttonPanel.add(new JButton(msgPane.getCopyAction()));
    buttonPanel.add(new JButton(msgPane.getClearAction()));
    buttonPanel.add(new JButton(msgPane.getSelectAllAction()));

    window.getContentPane().add(new JScrollPane(msgPane), BorderLayout.CENTER);
    window.getContentPane().add(buttonPanel, BorderLayout.SOUTH);
    window.setVisible(true);

    if ((args.length == 1) && args[0].equals("-logger")) {
      MessagePaneLogHandler logHandler = new MessagePaneLogHandler(msgPane);
      Logger testLogger = Logger.getLogger("test");
      testLogger.setLevel(Level.ALL);
      testLogger.addHandler(logHandler);

      testLogger.log(Level.FINE, "OK, we're just starting up now...",
                     (Throwable)null);
      testLogger.log(Level.INFO, "Now here's some information for you!",
                     (Throwable)null);
      testLogger.log(Level.WARNING, "Let this be a warning to you!",
                     new Exception("Condition YELLOW"));
      testLogger.log(Level.SEVERE, "Danger, Danger Will Robinson!",
                     new Exception("Condition RED"));
    }
    else {
      msgPane.addStatusMessage("Ok, are we having fun yet?", null);
      msgPane.addInformatoryMessage("Hey, here's some info.", null);
      msgPane.addWarningMessage("Hey, this is a warning, buddy!",
                                new Exception("That was a warning"));
      msgPane.addErrorMessage("Hey, this is an error!",
                              new Exception("That was an error"));
    }
  }


  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  inner class TextEventHandler
  ///
  /////////////////////////////////////////////////////////////////////////////

  protected class TextEventHandler implements CaretListener, DocumentListener
  {
    @Override
    public void caretUpdate(CaretEvent e)
    {
      int p1 = getSelectionStart();
      int p2 = getSelectionEnd();


      cutAction.setEnabled(p1 != p2);
      copyAction.setEnabled(p1 != p2);
    }

    @Override
    public void insertUpdate(DocumentEvent e)
    {
      clearAction.setEnabled(true);
      selectAllAction.setEnabled(true);
    }

    @Override
    public void removeUpdate(DocumentEvent e) 
    {
      int end = getDocument().getLength();
      clearAction.setEnabled(end > 0);
      selectAllAction.setEnabled(end > 0);
    }

    @Override
    public void changedUpdate(DocumentEvent e)
    {
      // no action needed
    }
  }

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  inner class MessageAdder
  ///
  /////////////////////////////////////////////////////////////////////////////

  protected class MessageAdder implements Runnable
  {
    JTextPane   textPane;
    Icon        icon;
    Style       style;
    String      message;
    Throwable   throwable;
    String      threadName;
    Date        eventTime;
    long        millis;

    MessageAdder(JTextPane textPane, Icon icon, Style style,
                 String message, Throwable throwable)
    {
      if ((message != null) && !message.endsWith("\n"))
    {
        message += "\n";
    }

      this.textPane  = textPane;
      this.icon      = icon;
      this.style     = style;
      this.message   = message;
      this.throwable = throwable;

      threadName     = Thread.currentThread().getName();
      eventTime      = new Date();
      millis         = System.currentTimeMillis();
    }

    @Override
    public void run()
    {
      try {
        StyledDocument doc = textPane.getStyledDocument();
        int end = doc.getLength();

        // these two are just SimpleAttributeSets rather than full Styles
        SimpleAttributeSet iconAttrs = new SimpleAttributeSet();

        // start a new paragraph with a newline unless it's the first one
        // or unless we've already got a newline at the end
        if ((end > 0) && (textPane.getText(end-1, 1).charAt(0) != '\n'))
        {
            doc.insertString(end++, "\n", this.style);
        }

        // start off with the icon
        if (icon != null)
        {
            StyleConstants.setIcon(iconAttrs, icon);
        }

        // set style for header line with icon.
        textPane.setCaretPosition(end);
        Style headerStyle = getStyle(HEADER_STYLE);
        textPane.setLogicalStyle(headerStyle);

        // move to the end of the document and set the current style
        doc.insertString(end, " ", iconAttrs);

        String headerText = "\t";
        if (textPane == MessagePane.this) {
          // format the current time
          DateFormat df = DateFormat.getTimeInstance(DateFormat.LONG);
          df.setTimeZone(TimeZone.getDefault());  //!!! BUG WORKAROUND !!!//

          String time = df.format(eventTime);
          if (true)
        {
            time += (" + " + (millis % 1000) + " msec");
        }

          // tab to the indent and insert the time and thread name
          // using the info-header style
          headerText = "\t" + time + " -- thread " + threadName + "\n";
        }
        doc.insertString(end+1, headerText, headerStyle);

        // switch styles for the real message
        textPane.setCaretPosition(doc.getLength());
        textPane.setLogicalStyle(this.style);
        textPane.setParagraphAttributes(this.style, true);

        // now insert the message in the specified style
        doc.insertString(end + headerText.length() + 1, message, this.style);

        // if an throwable accompanied the message, add a stack-trace button
        if (throwable != null) {
          addStackTraceButton(end + headerText.length() + 1,
                              icon, style, message, throwable);
        }
      }
      catch (BadLocationException badLoc) {
        badLoc.printStackTrace();
      }
      catch (Exception e) {
        System.out.println("Unexpected exception in MessageAdder: " + e);
      }
    }
  }


  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  inner class MessageAppender
  ///
  /////////////////////////////////////////////////////////////////////////////

  protected class MessageAppender implements Runnable
  {
    String      message;
    Throwable   throwable;

    MessageAppender(String message, Throwable throwable)
    {
      this.message   = message;
      this.throwable = throwable;
    }

    @Override
    public void run()
    {
      MessagePane textPane = MessagePane.this;
      StyledDocument doc = textPane.getStyledDocument();

      int end = doc.getLength();
      textPane.setCaretPosition(end);
      Style style = doc.getLogicalStyle(end);

      try {
        // tab to the normal indent and do the rest of the message
        doc.insertString(end, message, style);

        if (throwable != null)
        {
            addStackTraceButton(end, null, style, message, throwable);
        }

        // assign the logical style at the end in case there's an append
        textPane.setLogicalStyle(style);
      }
      catch (BadLocationException e) {
        e.printStackTrace();
      }
    }
  }


  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  inner class StackTraceButtonHandler
  ///
  /////////////////////////////////////////////////////////////////////////////

  private class StackTraceButtonHandler implements ActionListener, Serializable
  {
    private Icon        icon;
    private Style       style;
    private String      message;
    private Throwable   throwable;

    StackTraceButtonHandler(Icon icon, Style style, String message,
                            Throwable throwable)
    {
      this.icon      = icon;
      this.style     = style;
      this.message   = message;
      this.throwable = throwable;
    }

    @Override
    public void actionPerformed(ActionEvent evt)
    {
      JButton button = (JButton)evt.getSource();
      Style stacktraceStyle = getStyle(STACKTRACE_STYLE);

      try {
        if (true)
        {
            throwable = new Exception(throwable.getMessage());
        }

        StringWriter writer = new StringWriter();
        throwable.printStackTrace(new PrintWriter(writer));
        String stackTrace = writer.toString();

        JTextPane stackTracePane = new JTextPane() {
            @Override
            public boolean getScrollableTracksViewportWidth() { return(false); }
          };
        stackTracePane.setEditable(false);
        stackTracePane.getStyledDocument().insertString(0, stackTrace,
                                                        stacktraceStyle);
        JScrollPane scroller =
          new JScrollPane(stackTracePane,
                          JScrollPane.VERTICAL_SCROLLBAR_ALWAYS,
                          JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
        scroller.setPreferredSize(new Dimension(650, 300));
        stackTracePane.setCaretPosition(0);

        JPanel outerPanel = new JPanel();
        JTextPane messagePane = new JTextPane();
        messagePane.setBackground(UIManager.getColor("Panel.background"));
        messagePane.setEditable(false);
        (new MessageAdder(messagePane, icon, style, message, null)).run();

        outerPanel.setLayout(new BorderLayout());
        outerPanel.add(messagePane);
        outerPanel.add(scroller, BorderLayout.SOUTH);

        String exceptionMessage = throwable.getMessage();
        if (exceptionMessage == null)
        {
            exceptionMessage = "Exception";
        }

        JOptionPane.showMessageDialog(MessagePane.this, outerPanel,
                                      "Stack Trace for: " + exceptionMessage,
                                      JOptionPane.PLAIN_MESSAGE);

        MessagePane.this.requestFocus();
      }
      catch (BadLocationException e) {
        e.printStackTrace();
      }
    }
  }
  
}
