/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */
package modelobjects.framework.swing;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.text.DateFormat;
import java.util.Date;
import java.util.TimeZone;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import javax.swing.Timer;
import javax.swing.border.Border;

import modelobjects.layout.PartitionLayout;

/**
 *  StatusBar is a component that is intended to display various kinds
 *  of status information in an application.  It supports a variable
 *  number of application-specific sub-fields that can have different
 *  widths, as well as a standard optional time field, a progress-bar
 *  field to show work-in-progress, and a general status-text field.
 *  It also can accomodate a cancel-button for cancelling cancellable
 *  threads.
 */
public class StatusBar extends JPanel
  implements ActionListener, PropertyChangeListener
{
  /**
   *  Construct a StatusBar with no cancel-button and with a default
   *  progress-bar.
   */
  public StatusBar()
  {
    this(null);
  }

  /**
   *  Construct a StatusBar with the specified cancel-button and with a
   *  default progress-bar.
   */
  public StatusBar(JButton cancelButton)
  {
    this(cancelButton, makeDefaultProgressBar());
  }

  /**
   *  Construct a StatusBar, providing a cancel-button and progress-bar,
   *  either or both of which may be null.
   */
  public StatusBar(JButton cancelButton, JProgressBar progressBar)
  {
    super();
    this.cancelButton = cancelButton;
    this.progressBar  = progressBar;
    buildComponents();
  }

  /**
   *  Return this StatusBar's JProgressBar
   */
  public JProgressBar getProgressBar()
  {
    return(progressBar);
  }

  /**
   *  Return this StatusBar's cancel button.
   */
  public JButton getCancelButton()
  {
    return(cancelButton);
  }

  /**framework/swing/
   *  Create a default JProgressBar for use in a StatusBar.
   */
  public static JProgressBar makeDefaultProgressBar()
  {
    JProgressBar progressBar = new JProgressBar(0,8);
    progressBar.setBorder(BorderFactory.createLoweredBevelBorder());
    progressBar.setString("Working...");
    return(progressBar);
  }

  /**
   *  Return whether the progress-bar should be active whenever the
   *  cancel-button is enabled.
   */
  public boolean getProgressBarTracksCancelButton()
  {
    return(progressBarTracksCancelButton);
  }

  /**
   *  Assign whether the progress-bar should be active whenever the
   *  cancel-button is enabled.
   */
  public void setProgressBarTracksCancelButton(boolean val)
  {
    if (progressBarTracksCancelButton != val) {
      progressBarTracksCancelButton = val;

      if ((cancelButton != null) && (progressBar != null)) {
        if (val)
        {
            cancelButton.addPropertyChangeListener(this);
        }
        else
        {
            cancelButton.removePropertyChangeListener(this);
        }
      }
    }
  }

  /**
   *  Display the specified message in the main status-text field.
   */
  public void showStatusMessage(String message)
  {
    String text = ((message == null) ? "" : message);
    updateText(statusArea, text);
  }

  /**
   *  Set the width of the specified application-specific status field.
   *  There are nine status fields numbered from 0 to 8.  The width is
   *  given in number of characters.  A width of zero causes the field
   *  to be hidden.  A width greater than zero causes the field to be
   *  made visible with the specified width.
   */
  public void setStatusFieldSizeInChars(int statusFieldNumber, int numChars)
  {
    Dimension sz = new Dimension(40, 16);
    JComponent comp = extras[statusFieldNumber];
    Insets m = comp.getInsets();

    if (numChars > 0) {
      Font f = comp.getFont();
      if (f != null) {
        FontMetrics fm = getFontMetrics(f);
        if (fm != null)
        {
            sz = new Dimension(fm.charWidth('N') * numChars, fm.getHeight());
        }
      }
    }

    sz.width  += m.left + m.right;
    sz.height += m.top + m.bottom;

    comp.setPreferredSize(sz);
    comp.setVisible(numChars > 0);

    invalidate();
    validate();
  }

  /**
   *  Return the contents of the specified application-specified status field.
   */
  public String getStatusField(int statusFieldNumber)
  {
    return(extras[statusFieldNumber].getText());
  }

  /**
   *  Assign the contents of the specified application-specified status field.
   */
  public void setStatusField(int statusFieldNumber, String text)
  {
    updateText(extras[statusFieldNumber], text);
  }

  /**
   *  Show or hide the time-of-day field, activating or deactivating it as
   *  appropriate.
   */
  public void showTimeOfDay(boolean show)
  {
    if (show) {
      extras[timeFieldIndex].setHorizontalAlignment(JLabel.CENTER);
      setStatusFieldSizeInChars(timeFieldIndex, 10);
      if (wallclockTimer == null) {
        wallclockTimer = new Timer(1000, this);
        wallclockTimer.setRepeats(true);
      }
      wallclockTimer.start();
    }
    else {
      if (wallclockTimer != null)
    {
        wallclockTimer.stop();
    }
      setStatusFieldSizeInChars(timeFieldIndex, 0);
    }
  }

  /**
   *  Override to stop and remove timers also.
   */
  @Override
public void removeNotify()
  {
    if (progressTimer != null) {
      progressTimer.stop();
      progressTimer = null;
    }
    if (wallclockTimer != null) {
      wallclockTimer.stop();
      wallclockTimer = null;
    }

    super.removeNotify();
  }

  /**
   *  Update the specified JTextField or JLabel to have the specified
   *  String value by using SwingUtilities.invokeLater, so that the
   *  modification will happen only in the Swing event thread.
   */
  protected void updateText(JComponent comp, String text)
  {
    SwingUtilities.invokeLater(new TextUpdater(comp, text));
  }

  /**
   *  Start or stop the progress meter, indicating work-in-progress or
   *  its completion.
   */
  public synchronized void monitorProgress(boolean start)
  {
    if (start) {
      if (true) {       // JDK 1.4 only
        progressBar.setStringPainted(true);
        progressBar.setIndeterminate(true);
      }
      else {
        if (progressTimer == null) {
          progressTimer = new Timer(150, this);
          progressTimer.setRepeats(true);
        }
        if (!progressTimer.isRunning()) {
          progressTimer.start();
          progressBar.setStringPainted(true);
        }
      }
    }
    else {
      if (true) {       // JDK 1.4 only
        progressBar.setStringPainted(false);
        progressBar.setIndeterminate(false);
      }
      else {
        if (progressTimer != null && progressTimer.isRunning()) {
          progressTimer.stop();
          progressBar.setStringPainted(false);
          progressBar.setValue(0);
        }
      }
    }
  }

  /**
   *  If there is a cancel-button and a progress-bar, tie the progress-bar
   *  to when the cancel-button is enabled and disabled.
   */
  @Override
public void propertyChange(PropertyChangeEvent event)
  {
    if ((cancelButton != null) && (progressBar != null) &&
        (event.getSource() == cancelButton) &&
        (event.getPropertyName().equals("enabled"))) {
      monitorProgress(cancelButton.isEnabled());
    }
  }

  /**
   *  Handle a "tick" event either from the wallclock-timer or from the
   *  progress-timer by updating the corresponding field.
   */
  @Override
public void actionPerformed(ActionEvent event)
  {
    if ((event.getSource() == wallclockTimer)) {
      DateFormat df = DateFormat.getTimeInstance(DateFormat.MEDIUM);
      df.setTimeZone(TimeZone.getDefault());  //!!! BUG WORKAROUND !!!//
      String time = df.format(new Date());
      if (!getStatusField(timeFieldIndex).equals(time))
    {
        setStatusField(timeFieldIndex, time);
    }
    }
    else if (event.getSource() == progressTimer) {
      if (progressBar.getValue() < progressBar.getMaximum())
    {
        progressBar.setValue(progressBar.getValue() + 1);
    }
    else
    {
        progressBar.setValue(0);
    }
    }
  }

  /**
   *  Set up the layout and composition of this StatusBar.
   */
  private void buildComponents()
  {
    this.setLayout(new BorderLayout(2, 2));

    JPanel p = new JPanel();
    p.setLayout(new PartitionLayout(false, "Rppppppppppp", 2, 1, 1, "status"));
    this.add(p, BorderLayout.CENTER);

    statusArea = new JTextField("");
    statusArea.setEditable(false);
    statusArea.setOpaque(true);
    statusArea.setBackground(Color.white);
    statusArea.setBorder(STATUS_FIELD_BORDER);

    // first the cancel-button if there is one
    if (cancelButton != null)
    {
        add(cancelButton, BorderLayout.WEST);
    }

    // next add the status-text area
    p.add(statusArea);

    // next add the "extra" fields
    if (progressBar != null) {
      numExtras--;
      timeFieldIndex--;
    }
    extras = new JLabel[numExtras];
    for (int i = 0; i < numExtras; i++) {
      JLabel lbl = extras[i] = new JLabel();
      lbl.setBorder(STATUS_FIELD_BORDER);
      lbl.setVisible(false);
      p.add(lbl);
    }

    // and finally the progress-bar if there is one
    if (progressBar != null)
    {
        p.add(progressBar);
    }
  }

  /////////////////////////////////////////////////////////////////////////////

  static class TextUpdater implements Runnable
  {
    private Component comp;
    private String    text;
    
    public TextUpdater(JComponent comp, String text)
    {
      this.comp = comp;
      this.text = text;
    }

    @Override
    public void run()
    {
      if (comp instanceof JTextField)
    {
        ((JTextField)comp).setText(text);
    }
    else if (comp instanceof JLabel)
    {
        ((JLabel)comp).setText(text);
    }
    }
  }

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  representation
  ///
  /////////////////////////////////////////////////////////////////////////////

  private               int             numExtras = 11;
  private               int             timeFieldIndex = 10;

  private transient     Timer           wallclockTimer;
  private transient     Timer           progressTimer;

  private               JTextField      statusArea;
  private               JLabel          extras[];

  private               JButton         cancelButton;
  private               JProgressBar    progressBar;
  private               boolean         progressBarTracksCancelButton;

  private static        Border          STATUS_FIELD_BORDER =
    BorderFactory.createCompoundBorder
        (BorderFactory.createLoweredBevelBorder(),
         BorderFactory.createEmptyBorder(0, 2, 0, 2));

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  unit test
  ///
  /////////////////////////////////////////////////////////////////////////////

  public static void main(String[] args)
  {
    JFrame window = new JFrame("StatusBar Test");
    window.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    window.setBounds(300, 300, 700, 400);
    JButton cancelButton = new JButton("Cancel");
    final StatusBar statusBar = new StatusBar(cancelButton);
    statusBar.setStatusFieldSizeInChars(0, 2);
    statusBar.setStatusFieldSizeInChars(1, 4);
    statusBar.setStatusFieldSizeInChars(2, 6);
    statusBar.setStatusFieldSizeInChars(3, 8);

    statusBar.setStatusField(0, "NN");
    statusBar.setStatusField(1, "NNNN");
    statusBar.setStatusField(2, "NNNNNN");
    statusBar.setStatusField(3, "Whassup?");
    
    statusBar.showTimeOfDay(true);

    cancelButton.addActionListener(new ActionListener() {
      boolean running = false;
      @Override
    public void actionPerformed(ActionEvent e) {
        running = !running;
        statusBar.monitorProgress(running);        
      }
    });

    window.getContentPane().add(statusBar, BorderLayout.SOUTH);
    window.setVisible(true);
  }
  
}
