package modelobjects.framework.test;

import java.awt.Color;
import java.awt.Component;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.beans.PropertyEditor;

import javax.swing.AbstractAction;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTextField;

import modelobjects.framework.EditFormManager;
import modelobjects.framework.EditRule;
import modelobjects.framework.ModelEditEvent;
import modelobjects.framework.ModelEditException;
import modelobjects.framework.ModelEditFailureEvent;
import modelobjects.framework.ModelEditFailureListener;
import modelobjects.framework.ModelEditListener;
import modelobjects.framework.ModelEditMediator;
import modelobjects.framework.PropertyEditorViewAspectAdapter;
import modelobjects.framework.ViewAspectAdapter;
import modelobjects.framework.model.JavaBeanModelDescriptor;
import modelobjects.framework.model.ModelAspectId;
import modelobjects.framework.model.ModelDescriptor;
import modelobjects.framework.model.ModelObjectAdapter;
import modelobjects.framework.model.ModelValidationException;
import modelobjects.framework.swing.JCheckBoxAdapter;
import modelobjects.framework.swing.JTextFieldAdapter;
import modelobjects.layout.PartitionLayout;
import modelobjects.util.NumberPropertyEditor;

/**
 *This sample shows how to build a simple form. Perform the following steps to
 *deal with your model:
 *<ul>
 *<li>Create your application model object (in this case TestPerson.java)
 *<li>Create a ModelDescriptor class for your model (PersonModelDescriptor). If your model is a JavaBean, your ModelDescriptor can extend JavaBeanModelDescriptor. In this class, you'll need to override the makeModelObjectAdapter() factory method to return your specific type of ModelObjectAdapter.
<li>Extend the ModelObjectAdapter class to create your specific instance of ModelObjectAdapter. This is the class that is responsible for validating edits (i.e. making sure required fields are filled) to the application model object. If the validateModel() method of this class doesn't throw an exception, then the modelEdited() event will be successfully fired to the ModelEditMediator.
 */
public class SimpleFormTest {

  JFrame frame;
  ModelDescriptor       personDescriptor;
  ModelEditMediator     modelEditMediator;
  EditFormManager       formManager;
  MyFormListener myFormListener;
  boolean m_applyAccepted = false;

  /**
   *Constructor
   */
  public SimpleFormTest() {
    
    frame = new JFrame("Test Form");

    modelEditMediator = new ModelEditMediator();
    myFormListener = new MyFormListener();
    modelEditMediator.addModelEditListener(myFormListener);
    modelEditMediator.addModelEditFailureListener(myFormListener);

    //Create the model descriptor
    try {
      personDescriptor = new PersonDescriptor();

    } catch(Exception e) {
      e.printStackTrace();
    }

    JPanel entryPanel = makeEntryPanel();
    JPanel buttonPanel = makeButtonPanel();
    JPanel mainPanel = new JPanel();
    mainPanel.setLayout(new PartitionLayout(true,"p,p",1));
    mainPanel.add(entryPanel);
    mainPanel.add(buttonPanel);

    frame.getContentPane().add(mainPanel);

    frame.addWindowListener(new WindowAdapter() {
        @Override
        public void windowClosing(WindowEvent event) {
          frame.dispose();
          System.exit(1);
        }
      });
  }

  /**
   *
   */
  private JPanel makeEntryPanel() {
    JPanel entryPanel = new JPanel();
    entryPanel.setBorder(BorderFactory.createRaisedBevelBorder());
    entryPanel.setLayout(new PartitionLayout(false,"p,r",4));
    
    entryPanel.add(new JLabel("First Name:"));
    JTextField firstNameField = new JTextField(25);
    entryPanel.add(firstNameField);
    
    entryPanel.add(new JLabel("Last Name:"));
    JTextField lastNameField = new JTextField(25);
    entryPanel.add(lastNameField);


    JTextFieldAdapter firstNameFieldAdapter =
      new JTextFieldAdapter(ModelAspectId.forName("firstName"),
                            firstNameField,     // viewComponent
                            EditRule.ALWAYS,    // editRule
                            null,               // viewValueConverter
                            modelEditMediator);


    JTextFieldAdapter lastNameFieldAdapter =
      new JTextFieldAdapter(ModelAspectId.forName("lastName"),
                            lastNameField,     // viewComponent
                            EditRule.ALWAYS,    // editRule
                            null,               // viewValueConverter
                            modelEditMediator);

    //Let's use the property editor for number
    //We could have also used a JTextFieldAdapter with a IntToStringConverter
    PropertyEditor numberPropertyEditor =
      new NumberPropertyEditor("age", java.lang.Integer.class, new Integer(0), new Integer(100));

    ViewAspectAdapter ageAdapter =
      PropertyEditorViewAspectAdapter.
      makeViewAspectAdapter(ModelAspectId.forName("age"),
                            EditRule.ALWAYS,    // editRule
                            numberPropertyEditor,  // PropertyEditor
                            modelEditMediator);

    entryPanel.add(new JLabel("Age:"));
    Component ageField = ageAdapter.getViewComponent();
    entryPanel.add(ageField);

    /*If we wanted to use a
      entryPanel.add(new JLabel("Age:"));
      JTextField ageField = new JTextField(25);
      entryPanel.add(ageField);

      JTextFieldAdapter ageAdapter =
      new JTextFieldAdapter(ModelAspectId.forName("age"),
                            ageField,     // viewComponent
                            EditRule.ALWAYS,    // editRule
                            new IntToStringConverter(),// viewValueConverter
                            modelEditMediator);*/


    entryPanel.add(new JLabel("Happy:"));
    JCheckBox happyCheckbox = new JCheckBox("Happy");
    entryPanel.add(happyCheckbox);
    
    JCheckBoxAdapter checkBoxAdapter =
      new JCheckBoxAdapter(ModelAspectId.forName("happy"),
                            happyCheckbox,     // viewComponent
                            EditRule.ALWAYS,    // editRule
                            null,               // viewValueConverter
                            modelEditMediator);

    return(entryPanel);
  }

  /**
   *
   */
  private JPanel makeButtonPanel() {

    JPanel buttonPanel = new JPanel();
    buttonPanel.setBorder(BorderFactory.createRaisedBevelBorder());
    buttonPanel.setLayout(new FlowLayout());
    
    JButton okButton = new JButton(new OKAction());
    JButton applyButton = new JButton(modelEditMediator.getApplyAction());
    JButton resetButton = new JButton(modelEditMediator.getResetAction());
    JButton newButton = new JButton(new NewPersonAction());

    buttonPanel.add(okButton);
    buttonPanel.add(applyButton);
    buttonPanel.add(resetButton);
    buttonPanel.add(newButton);

    return(buttonPanel);
  }

  public boolean applyAccepted() {
    return(m_applyAccepted);
  }

  /***********
   *INNERCLASS
   *
   **********/
  class OKAction extends AbstractAction
  {
    OKAction() {
      super("OK");
    }

    @Override
    public void actionPerformed(ActionEvent event) {

      modelEditMediator.getApplyAction().actionPerformed(event);

      //if no error occured, exit the window
      if(applyAccepted()) {
        frame.dispose();
        System.exit(1);
      } else {

      }
    }
  }


  /**
   *New Person Action
   */
  class NewPersonAction extends AbstractAction
  {
    NewPersonAction() {
      super("New...");
    }

    @Override
    public void actionPerformed(ActionEvent event) {

      if(modelEditMediator.hasUnappliedChanges()) {
        System.out.println("unapplied changes exist, can't create new");
        return;

      } else {
        try {
          //Create the model object
          TestPerson person = new TestPerson();

          //Make the modelObjectAdapter from the instance
          ModelObjectAdapter personModelAdapter = personDescriptor.makeModelObjectAdapter(person);
      
          //This actaully sets the model for the form
          modelEditMediator.editModelAsNewObject(personModelAdapter);        

        } catch(Exception e) {
          e.printStackTrace();
        }
      }
    }

  }//end class


  /***********
   *INNERCLASS
   *
   **********/
  class MyFormListener implements ModelEditListener, ModelEditFailureListener {
    
    public MyFormListener() {
    }

    @Override
    public void modelCreated(ModelEditEvent event) throws ModelEditException {
      System.out.println("in modelCreated event="+event);
      System.out.println(event.getModelObject().toString());
    }

    @Override
    public void modelEdited(ModelEditEvent event) throws ModelEditException {
      System.out.println("in modelEdited event ID="+event.getEventID());
      System.out.println(event.getModelObject().toString());
      m_applyAccepted = true;
    }

    @Override
    public void modelDeleted(ModelEditEvent event) throws ModelEditException {
      System.out.println("in modelDeleted event ID="+event.getEventID());
      System.out.println(event.getModelObject().toString());
    }

    @Override
    public void modelEditFailed(ModelEditFailureEvent event) {
      System.out.println("in modelEditFailed event="+event);
      System.out.println(event.getModelObject().toString());
      m_applyAccepted = false;

      JOptionPane.showMessageDialog(frame, "Model Edit Failed: "+event.getException().getMessage(), "Alert", JOptionPane.ERROR_MESSAGE); 
    }
                                                                
  }



  /***********
   *INNERCLASS
   *
   **********/
  class PersonDescriptor extends JavaBeanModelDescriptor {

    public PersonDescriptor() throws java.beans.IntrospectionException {
      super(modelobjects.framework.test.TestPerson.class);
    }

    @Override
    public ModelObjectAdapter makeModelObjectAdapter(Object modelObject) {
      return(new PersonModelObjectAdapter(modelObject, this));
    }
  }

  /***********
   *INNERCLASS
   *
   **********/
  class PersonModelObjectAdapter extends ModelObjectAdapter {

    public PersonModelObjectAdapter(Object modelObject, ModelDescriptor modelDescriptor) {
      super(modelObject, modelDescriptor);
    }

    /**
     *This is where the validation logic occurs
     */
    @Override
    public void validateModel() throws ModelValidationException {
      TestPerson person = (TestPerson)getModelObject();
      
      //make sure all of the required fields are set
      StringBuffer errMsg = new StringBuffer("The following errors occurred:");
      boolean err = false;
      
      if(person.getFirstName() == null || person.getFirstName().trim().equals("")) {
        err = true;
        errMsg = errMsg.append("\n'First Name' is a required field");
        
      }

      if(person.getLastName() == null || person.getLastName().trim().equals("")) {
        err = true;
        errMsg = errMsg.append("\n'Last Name' is a required field");
      }

      if(err) {
        throw(new ModelValidationException(errMsg.toString()));
      }
    }

    /**
     *
     */
    @Override
    public void validateNewModel() throws ModelValidationException {
      validateModel();
    }
  }

  /**
   *
   */
  public static void main(String[] args) {

    try {
      SimpleFormTest test = new SimpleFormTest();

      //Create the model object
      TestPerson person = new TestPerson("Bill", "Filler", 34, true, Color.black);

      //Make the modelObjectAdapter from the instance
      ModelObjectAdapter personModelAdapter = test.personDescriptor.makeModelObjectAdapter(person);
      
      //This actaully sets the model for the form
      test.modelEditMediator.editModelObject(personModelAdapter);

      test.frame.setBounds(100, 100, 400, 400);
      test.frame.addNotify();
      test.frame.validate();
      test.frame.setSize(test.frame.getPreferredSize());
      test.frame.setVisible(true);

    } catch(Exception e) {

      e.printStackTrace();
    }

  }



}
