/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */

package modelobjects.framework.threads;

/**
 *  CancellableThread is an extension of Thread that is meant to be easier
 *  to manage in an interactive application where it may be necessary to
 *  allow the user to cancel it once it has started.  It supports
 *  ThreadEventListeners and provides special methods to check on its
 *  status and to request termination.
 *
 *  The Runnable objects that are executed by CancellableThreads *MUST*
 *  cooperate in termination by responding correctly to thread interrupts.
 *  Runnable objects should use CancellableThread.isThreadCancelled() to see
 *  if the Thread they are running in has been cancelled and, if so, to
 *  clean up and exit as quickly as possible.
 */
public class CancellableThread extends Thread
{
  /**
   *  Construct a CancellableThread to run the provided Runnable in a
   *  Thread with the provided thread name.
   */
  public CancellableThread(Runnable innerRunnable, String threadName)
  {
    this(threadName, new RunnableRunner(innerRunnable));
    this.innerRunnable = innerRunnable;
  }

  /**
   *  Really construct the CancellableThread with a RunnableRunner wrapper
   *  around the original runnable, and with the provided thread name.
   */
  private CancellableThread(String threadName, RunnableRunner runnableRunner)
  {
    super(runnableRunner, threadName);
    runnableRunner.setCancellableThread(this);
    this.uniqueID = new Integer(getNextUniqueId());
    this.runnableRunner = runnableRunner;
  }

  /**
   *  Return the inner Runnable object from which this CancellableThread
   *  was constructed.
   */
  public Runnable getInnerRunnable()
  {
    return(innerRunnable);
  }

  /**
   *  Return whether this CancellableThread is currently cancellable.
   *  The default implementation implementation always returns true.
   *  Subclasses may override, for example, to indicate a point of no
   *  return after which it does not wish to be cancelled (hopefully
   *  close to completion anyway).
   */
  public boolean isCancellable()
  {
    return(true);
  }

  /**
   *  Return the RunnableRunner that's running the inner Runnable
   */
  public RunnableRunner getRunnableRunner()
  {
    return(runnableRunner);
  }

  /**
   *  Issue a cancellation request to the thread
   */
  public void cancel()
  {
    synchronized(this) {
      cancelled = true;                 // set status - cannot be cleared
    }
    interrupt();                        // awaken blocked threads
  }

  /**
   *  Return whether the thread has been requested to be cancelled
   */
  public boolean isCancelled()
  {
    if (cancelled)
    {
        return(true);
    }
    synchronized(this) {
      return(cancelled);
    }
  }

  /**
   *  Return whether this CancellableThread was ever started.
   *  Use Thread.isAlive() to see if it's still running.
   */
  public boolean wasStarted()
  {
    return(runnableRunner.wasStarted());
  }

  /**
   *  A convenience method for check if the current thread has been cancelled.
   *  If the thread in which the call is made is not a CancellableThread,
   *  just checks to see if it has been interrupted.
   *  This method calls Thread.sleep(1) just to make sure that other threads
   *  (particularly the AWT-event-thread) get a chance to run.
   */
  public static boolean isThreadCancelled()
  {
    Thread t = Thread.currentThread();

    try {
      Thread.sleep(1);
    }
    catch (InterruptedException e) {
      t.interrupt();
    }

    if (t instanceof CancellableThread)
    {
        return(((CancellableThread)t).isCancelled());
    }
    else
    {
        return(t.isInterrupted());
    }
  }

  /**
   *  Two phase termination.
   *
   *  Returns true if thread completed cleanly on its own; false if it
   *  had to be forcibly eliminated.
   *
   *  Callers may want to take more drastic action if this method
   *  returns false, since uncooperative termination may cause serious
   *  problems for other threads.
   */
  public boolean terminate(long maxWaitToDie)
  {
    if (!isAlive())
     {
        return(true);                     // bypass if already dead
    }

    cancel();                           // issue cancellation request

    try {
      join(maxWaitToDie);               // wait for termination or timeout
    }
    catch (InterruptedException e) {    // ignore interrupt if occurs
    }

    if (!isAlive())
     {
        return(true);                     // successful clean termination
    }

    stop();                             // force hostile termination
    return(false);
  }

  /**
   *  Register a ThreadEventListener
   */
  public void addThreadEventListener(ThreadEventListener listener)
  {
    runnableRunner.addThreadEventListener(listener);
  }

  /**
   *  Register a ThreadEventListener
   */
  public void removeThreadEventListener(ThreadEventListener listener)
  {
    runnableRunner.removeThreadEventListener(listener);
  }

  /**
   *  Return the thread-completion text that should be set as status
   *  when the thread completes.
   */
  public String getThreadCompletedText()
  {
    return(runnableRunner.getThreadCompletedText());
  }

  /**
   *  Assign the thread-completion text that should be set as status
   *  when the thread completes.
   */
  public void setThreadCompletedText(String text)
  {
    runnableRunner.setThreadCompletedText(text);
  }

  /**
   *  Return a id number for this cancellable thread that is guaranteed
   *  to be unique within the current VM execution.
   */
  public Integer getUniqueId()
  {
    return(uniqueID);
  }

  /**
   *  Return the next unique ID number.
   */
  private static synchronized int getNextUniqueId()
  {
    return(uniqueIdCounter++);
  }

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  representation
  ///
  /////////////////////////////////////////////////////////////////////////////

  private static int            uniqueIdCounter = 1;

  private Runnable              innerRunnable;
  private RunnableRunner        runnableRunner;
  private Integer               uniqueID;
  private boolean               cancelled = false;    // acts as a latch

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  inner class RunnableRunner
  ///
  /////////////////////////////////////////////////////////////////////////////

  /**
   *  This class is used to run another Runnable object in a controlled
   *  manner, firing ThreadEvents to registered ThreadEventListeners when
   *  the inner runnable starts and when it completes.
   */
  private static class RunnableRunner implements Runnable
  {
    /**
     *  Construct a RunnableRunner for the specified Runnable object and
     *  the specified thread name.
     */
    RunnableRunner(Runnable innerRunnable)
    {
      this.innerRunnable    = innerRunnable;
      threadListenerSupport = new ThreadListenerSupport();
      threadCompletedText   = "";       // clear status by default
    }

    /**
     *  Return the inner Runnable managed by the RunnableRunner
     */
    Runnable getInnerRunnable()
    {
      return(innerRunnable);
    }

    /**
     *  Execute the provided Runnable object 
     */
    @Override
    public void run()
    {
      started = true;
      Throwable throwable = null;

      try {
        threadListenerSupport.fireThreadStarted(innerRunnable, thread);
        innerRunnable.run();
      }
      catch (RuntimeException e) {
        throwable = e;          // unhandled runtime exception
      }
      catch (ThreadDeath t) {
        throwable = t;          // forcible uncooperative termination
      }
      finally {
        synchronized (this) {
          notifyAll();             // notify any possible waiting thread
        }
        threadListenerSupport.fireThreadCompleted(innerRunnable, thread,
                                                  throwable,
                                                  getThreadCompletedText());
      }
    }

    /**
     *  Register a ThreadEventListener
     */
    public void addThreadEventListener(ThreadEventListener listener)
    {
      threadListenerSupport.addListener(listener);
    }

    /**
     *  Register a ThreadEventListener
     */
    public void removeThreadEventListener(ThreadEventListener listener)
    {
      threadListenerSupport.removeListener(listener);
    }

    /**
     *  Return whether the RunnableRunner's run method was started.
     *  Use Thread.isAlive() to see if it's still running.
     */
    public boolean wasStarted()
    {
      return(started);
    }

    /**
     *  Assign the thread-completion text that should be set as status
     *  when the thread completes.
     */
    public void setThreadCompletedText(String text)
    {
      threadCompletedText = text;
    }

    /**
     *  Return the thread-completion text that should be set as status
     *  when the thread completes.
     */
    public String getThreadCompletedText()
    {
      return(threadCompletedText);
    }

    /**
     *  internal linkage with "enclosing" CancellableThread
     */
    void setCancellableThread(CancellableThread thread)
    {
      this.thread = thread;
    }

    ///
    ///  Representation
    ///

    private boolean               started;
    private Runnable              innerRunnable;
    private CancellableThread     thread;
    private ThreadListenerSupport threadListenerSupport;
    private String                threadCompletedText;
  }
}
