/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */

package modelobjects.framework.threads;

import java.awt.event.ActionEvent;
import java.io.Serializable;
import java.util.ArrayList;

import javax.swing.AbstractAction;
import javax.swing.Action;


/**
 *  CancellableThreadManager keeps track of CancellableThreads and provides
 *  a cancel-action for cancelling them at the user's request.
 */
public class CancellableThreadManager implements Serializable
{
  /**
   *  Construct a CancellableThreadManager
   */
  public CancellableThreadManager()
  {
    threadListenerSupport = new ThreadListenerSupport();
    threadWatcher = new ThreadWatcher();
    cancelAction  = new CancelAction();
  }

  /**
   *  Return the Action that, when invoked, will cancel all CancellableThreads
   *  active in this CancellableThreadManager.
   */
  public Action getCancelAllAction()
  {
    return(cancelAction);
  }

  /**
   *  Create and return a managed CancellableThread.
   *  @param runnable the runnable object to be executed
   *  @param threadName the name to use for the CancellableThread
   */
  public CancellableThread createCancellableThread(Runnable runnable,
                                                   String threadName)
  {
    CancellableThread thread = new CancellableThread(runnable, threadName);
    thread.addThreadEventListener(getThreadWatcher());
    return(thread);
  }

  /**
   *  Create and return a managed CancellableThread.
   *  @param runnable the runnable object to be executed
   *  @param threadName the name to use for the CancellableThread
   *  @param threadCompletedText status text to display on completion
   */
  public CancellableThread createCancellableThread(Runnable runnable,
                                                   String threadName,
                                                   String threadCompletedText)
  {
    CancellableThread thread = createCancellableThread(runnable, threadName);
    thread.setThreadCompletedText(threadCompletedText);
    return(thread);
  }

  /**
   *  Create and return a managed CancellableThread.
   *  @param runnable the runnable object to be executed
   *  @param threadName the name to use for the CancellableThread
   *  @param threadPriority the priority to use for the CancellableThread
   */
  public CancellableThread createCancellableThread(Runnable runnable,
                                                   String threadName,
                                                   int threadPriority)
  {
    CancellableThread thread = createCancellableThread(runnable, threadName);
    thread.setPriority(threadPriority);
    return(thread);
  }

  /**
   *  Create and return a managed CancellableThread.
   *  @param runnable the runnable object to be executed
   *  @param threadName the name to use for the CancellableThread
   *  @param threadPriority the priority to use for the CancellableThread
   */
  public CancellableThread createCancellableThread(Runnable runnable,
                                                   String threadName,
                                                   String threadCompletedText,
                                                   int threadPriority)
  {
    CancellableThread thread = createCancellableThread(runnable, threadName);
    thread.setThreadCompletedText(threadCompletedText);
    thread.setPriority(threadPriority);
    return(thread);
  }

  /**
   *  Register a ThreadEventListener with this CancellableThreadManager.
   */
  public void addThreadEventListener(ThreadEventListener listener)
  {
    threadListenerSupport.addListener(listener);
  }

  /**
   *  Unregister a ThreadEventListener with this CancellableThreadManager.
   */
  public void removeThreadEventListener(ThreadEventListener listener)
  {
    threadListenerSupport.removeListener(listener);
  }

  private ThreadWatcher getThreadWatcher()
  {
    if (threadWatcher == null)
    {
        threadWatcher = new ThreadWatcher();
    }
    return(threadWatcher);
  }

  private ThreadListenerSupport         threadListenerSupport;
  private Action                        cancelAction;
  private transient ThreadWatcher       threadWatcher;
 

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  inner class CancelAction
  ///
  /////////////////////////////////////////////////////////////////////////////

  protected class CancelAction extends AbstractAction
    implements Serializable
  {
    public CancelAction()
    {
      super("Stop");
      setEnabled();
    }

    private void setEnabled() {
        setEnabled(false);
    }
    @Override
    public void actionPerformed(ActionEvent event)
    {
      threadWatcher.cancelAll(20000);
    }
  }

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  inner class ThreadWatcher
  ///
  /////////////////////////////////////////////////////////////////////////////

  /**
   *  The inner class <code>ThreadWatcher</code> is responsible for keeping
   *  track of all running cancellable threads and coordinating with the
   *  <code>CancelButton</code> on the <code>StatusPanel</code>.
   */
  protected class ThreadWatcher implements ThreadEventListener
  {
    public ThreadWatcher()
    {
      activeThreads = new ArrayList();
    }

    @Override
    public void threadStarted(ThreadEvent event)
    {
      synchronized(this) {
        activeThreads.add(event.getThread());
      }

      cancelAction.setEnabled(true);

      threadListenerSupport.fireThreadStarted(event.getRunnable(),
                                              event.getThread());
    }

    public int getNumActiveThreads() {
      return(activeThreads.size());
    }

    @Override
    public void threadCompleted(ThreadEvent event)
    {
      synchronized(this) {
        activeThreads.remove(event.getThread());
      }

      cancelAction.setEnabled(!activeThreads.isEmpty());

      threadListenerSupport.fireThreadCompleted(event.getRunnable(),
                                                event.getThread(),
                                                event.getThrowable());
    }

    public void cancelAll()
    {
      cancelAll(10000);
    }

    /**
     *  Cancel all active CancellableThreads.  This uses a two-pass
     *  algorithm where each CancellableThread is first interrupted in
     *  the first pass, and then terminated in the second pass.
     */
    public void cancelAll(long millisecondsToWait)
    {
      ArrayList threads = null;

      synchronized(this) {
        threads = new ArrayList(activeThreads);
      }

      for (int i = 0, n = threads.size(); i < n; i++) {
        CancellableThread thread = (CancellableThread)threads.get(i);
        if (thread.isAlive()) {
          thread.cancel();
        }
      }

      for (int i = 0, n = threads.size(); i < n; i++) {
        CancellableThread thread = (CancellableThread)threads.get(i);
        if (thread.isAlive())
        {
            thread.terminate(millisecondsToWait);
        }
      }
    }

    private ArrayList activeThreads;
  }

}
