/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */

package modelobjects.layout;

import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Insets;
import java.awt.LayoutManager;
import java.awt.Rectangle;
import java.io.Serializable;
import java.util.StringTokenizer;

/**
 *  This layout manager is designed for high-level layouts much like
 *  Netscape frames.
 *
 *  The container is partitioned into two or more sections, either
 *  horizontally or vertically, each of which specifies its size as
 *  a number of pixels, or to be its preferred size or to be whatever
 *  is left over.
 *
 *  Once constructed, it is assumed that components will be added
 *  left-to-right or top-to-bottom, as appropriate.
 */
public class PartitionLayout implements LayoutManager, Serializable
{
  /**
   *  This should be used to specify the size of a section that is
   *  intended to fill whatever space is leftover.  If more than one
   *  section is specified as REMAINDER, the available space is divided
   *  proportionally according to preferredSize dimensions.
   */
  public static final int                       REMAINDER	= -1;

  /**
   *  This should be used to specify the size of a section that should
   *  be sized to its preferred width or height but should not be sized
   *  proportionally.  This is often a better alternative to specifying
   *  a fixed size in pixels.
   */
  public static final int                       PREFERRED	= -2;

  /**
   *  These constants provide a more intuitive interface for constructing
   *  PartitionLayout instances instead of just using true or false for
   *  the layoutVertically constructor parameter.
   */
  public static final boolean                   TOP_TO_BOTTOM = true;
  public static final boolean                   LEFT_TO_RIGHT = false;




  /**
   *  Create a PartitionLayout that will have to be initialized more
   *  fully using property write methods.
   */
  public PartitionLayout()
  {
    this(false, "", 0);
  }


  /**
   *  Create a PartitionLayout with no margins around the sections.
   *  @param layoutVertically Indicates whether components are laid out
   *		from top to bottom (true) or left to right (false).
   *  @param sectionSizes A array of sizes that must correspond to the
   *		the number of sections in the container.  Sizes are given
   *		either as a number of pixels or as REMAINDER or PREFERRED.
   */
  public PartitionLayout(boolean layoutVertically, int sectionSizes[])
  {
    this(layoutVertically, sectionSizes, 0);
  }

  /**
   *  Create a PartitionLayout with the specified margin around sections.
   *  @param layoutVertically Indicates whether components are laid out
   *		from top to bottom (true) or left to right (false).
   *  @param sectionSizes A array of sizes that must correspond to the
   *		the number of sections in the container.  Sizes are given
   *		either as a number of pixels or as REMAINDER or PREFERRED.
   *  @param gapSize The amount of space to leave between each section
   *		and around the whole container in pixels.
   */
  public PartitionLayout(boolean layoutVertically, int sectionSizes[],
                         int gapSize)
  {
    this(layoutVertically, sectionSizes,
         gapSize, gapSize, gapSize, "Unnamed");
  }

  /**
   *  Create a PartitionLayout with the specified margin around sections.
   *  @param layoutVertically Indicates whether components are laid out
   *		from top to bottom (true) or left to right (false).
   *  @param sectionSizes A array of sizes that must correspond to the
   *		the number of sections in the container.  Sizes are given
   *		either as a number of pixels or as REMAINDER or PREFERRED.
   *  @param gapSize The amount of space to leave between each section
   *		and around the whole container in pixels.
   *  @param instanceName A name to give to the layout manager that will
   *		be used in diagnostic messages.
   */
  public PartitionLayout(boolean layoutVertically, int sectionSizes[],
                         int gapSize, String instanceName)
  {
    this(layoutVertically, sectionSizes,
         gapSize, gapSize, gapSize, instanceName);
  }

  /**
   *  Create a PartitionLayout with the specified margin around sections.
   *  @param layoutVertically Indicates whether components are laid out
   *		from top to bottom (true) or left to right (false).
   *  @param encodedSizes A string compososed of '+' and '|' characters
   *            that specifies the behavior of the partitions either as
   *            REMAINDER or PREFERRED, respectively.
   *  @param gapSize The amount of space to leave between each section
   *		and around the whole container in pixels.
   *  @param instanceName A name to give to the layout manager that will
   *		be used in diagnostic messages.
   */
  public PartitionLayout(boolean layoutVertically, String encodedSizes,
                         int gapSize)
  {
    this(layoutVertically, decodeSizes(encodedSizes),
         gapSize, gapSize, gapSize, "Unnamed");
  }

  /**
   *  Create a PartitionLayout with the specified margin around sections.
   *  @param layoutVertically Indicates whether components are laid out
   *		from top to bottom (true) or left to right (false).
   *  @param encodedSizes A string compososed of '+' and '|' characters
   *            that specifies the behavior of the partitions either as
   *            REMAINDER or PREFERRED, respectively.
   *  @param sepGap The amount of space to put between each section in
   *		pixels.
   *  @param tbEdgeGap The amount of space to put on the top and bottom
   *		of the container in pixels.  tb means top and bottom.
   *  @param lrEdgeGap The amount of space to put on the left and right
   *		of the container in pixels.  lr means left and right.
   *  @param instanceName A name to give to the layout manager that will
   *		be used in diagnostic messages.
   */
  public PartitionLayout(boolean layoutVertically, String encodedSizes,
                         int gapSize, int tbEdgeGap, int lrEdgeGap,
                         String instanceName)
  {
    this(layoutVertically, decodeSizes(encodedSizes),
         gapSize, tbEdgeGap, lrEdgeGap, instanceName);
  }

  /**
   *  Create a PartitionLayout with the specified margin around sections.
   *  @param layoutVertically Indicates whether components are laid out
   *		from top to bottom (true) or left to right (false).
   *  @param sectionSizes A array of sizes that must correspond to the
   *		the number of sections in the container.  Sizes are given
   *		either as a number of pixels or as REMAINDER or PREFERRED.
   *  @param sepGap The amount of space to put between each section in
   *		pixels.
   *  @param tbEdgeGap The amount of space to put on the top and bottom
   *		of the container in pixels.  tb means top and bottom.
   *  @param lrEdgeGap The amount of space to put on the left and right
   *		of the container in pixels.  lr means left and right.
   *  @param instanceName A name to give to the layout manager that will
   *		be used in diagnostic messages.
   */
  public PartitionLayout(boolean layoutVertically, int sectionSizes[],
                         int sepGap, int tbEdgeGap, int lrEdgeGap,
                         String instanceName)
  {
    this.layoutVertically = layoutVertically;
    this.sectionSizes = sectionSizes;
    this.sepGap = sepGap;
    this.splitterGap = 8;		// a good default value
    this.useSplitters = false;
    this.tbEdgeGap = tbEdgeGap;
    this.lrEdgeGap = lrEdgeGap;
    this.instanceName = instanceName;
    this.savedInfo = null;
    this.orientation = new Orientation(layoutVertically);
  }


  /**
   *  Converts a String representation of partition specs to and array of int.
   *  Specification characters may be separated by spaces and/or commas.
   *
   *  The recognized characters are:
   *
   *  PREFERRED:  '|'  '.'  'p'  'P'
   *  REMAINDER:  '+'  '*'  'r'  'R'
   *
   */
  public static int[] decodeSizes(String encoding)
       throws IllegalArgumentException
  {
    String validChars = "|+.*pPrR";

    if (!checkSpec(encoding, validChars))
    {
        return(decodeFancySpec(encoding));
    }

    int sz = encoding.length();
    int decoded[] = new int[sz];

    for (int i = 0; i < sz; i++)
    {
        decoded[i] = translateSpecificationChar(encoding.charAt(i));
    }

    return(decoded);
  }

  static int[] decodeFancySpec(String encoding)
       throws IllegalArgumentException
  {
    StringTokenizer toks = new StringTokenizer(encoding, ",; ", false);

    // first count the tokens
    int numToks = 0;
    for (; toks.hasMoreTokens(); numToks++)
    {
        toks.nextToken();
    }

    int decoded[] = new int[numToks];

    toks = new StringTokenizer(encoding, ",; ", false);
    for (int i = 0; toks.hasMoreTokens(); i++)
      {
        String tok = toks.nextToken();
        char c = tok.charAt(0);
        if (Character.isDigit(c))
        {
            decoded[i] = Integer.parseInt(tok);
        }
        else
        {
            decoded[i] = translateSpecificationChar(c);
        }
      }

    return(decoded);
  }

  static int translateSpecificationChar(char c)
  {
    switch (c)
      {
      case '+': case '*': case 'r': case 'R':
        return(REMAINDER);
      case '|': case '.': case 'p': case 'P':
        return(PREFERRED);
      default:
        throw new IllegalArgumentException
          ("unrecognized partition-spec character: '" + c + "'");
      }
  }

  static boolean checkSpec(String tok, String validChars)
  {
    for (int i = tok.length() - 1; i >= 0; i--)
    {
        if (validChars.indexOf(tok.charAt(i)) == -1)
        {
            return(false);
        }
    }
    return(true);
  }

  public boolean isVertical()
  {
    return(layoutVertically);
  }

  public void setVertical(boolean vertical)
  {
    layoutVertically = vertical;
    orientation = new Orientation(vertical);
  }

  /**
   *  Allows partitions to be set using a String of '+' and '|' characters
   *  designating REMAINDER and PREFERRED respectively.
   */
  public void setPartitionSpecs(String encoding)
  {
    sectionSizes = decodeSizes(encoding);
  }

  public void setPartitionSpecs(int partitionSpecs[])
  {
    sectionSizes = partitionSpecs;
  }

  public String getPartitionSpecs()
  {
    int n = getNumPartitions();
    StringBuffer buf = new StringBuffer();
    for (int i = 0; i < n; i++)
      {
        int spec = getPartitionSpec(i);
        switch (spec)
        {
        case PREFERRED:
          buf.append("P");
          break;
        case REMAINDER:
          buf.append("R");
          break;
        default:
          buf.append(spec);
          break;
        }
        if (i < (n-1))
        {
            buf.append(", ");
        }
      }
    return(new String(buf));
  }

  public int getPartitionSpec(int comp)
  {
    return(sectionSizes[comp]);
  }

  public int getTbEdgeGap()
  {
    return(tbEdgeGap);
  }

  public void setTbEdgeGap(int tbEdgeGap)
  {
    this.tbEdgeGap = tbEdgeGap;
  }

  public int getLrEdgeGap()
  {
    return(lrEdgeGap);
  }

  public void setLrEdgeGap(int lrEdgeGap)
  {
    this.lrEdgeGap = lrEdgeGap;
  }

  public int getSepGap()
  {
    return(sepGap);
  }

  public void setSepGap(int sepGap)
  {
    this.sepGap = sepGap;
  }

  public int getSplitterGap()
  {
    return(splitterGap);
  }

  public void setSplitterGap(int val)
  {
    splitterGap = val;
  }

  public int getNumPartitions()
  {
    return((sectionSizes == null) ? 0 : sectionSizes.length);
  }

  public boolean getUseSplitters()
  {
    return(useSplitters);
  }

  public void setUseSplitters(boolean val)
  {
    useSplitters = val;
  }


  // these two methods of LayoutManager are ignored
  @Override
public void addLayoutComponent(String name, Component c) { }
  @Override
public void removeLayoutComponent(Component c) { }

  /**
   *  Return the minimum layout size for the container
   */
  @Override
public Dimension minimumLayoutSize(Container par)
  {
    return(computeLayoutSize(par, true));
  }

  /**
   *  Return the preferred layout size for the container
   */
  @Override
public Dimension preferredLayoutSize(Container par)
  {
    return(computeLayoutSize(par, false));
  }

  /**
   *  Compute the layout size for a PartitionLayout assuming horizontal
   *  orientation, but using the orientation object to do adjustments.
   *  The height is that of the tallest component plus twice the tbEdgeGap.
   *  The width is computed from the specified sectionSizes except in
   *  the case of REMAINDER components which are computed in terms of
   *  minimumSize or preferredSize as appropriate.  A padding of sepGap
   *  is assumed between the components and a padding of lrEdgeGap is
   *  used on the left and right.
   */
  protected Dimension computeLayoutSize(Container par, boolean minSize)
  {
    Insets insets = par.getInsets();
    int numComps = par.getComponentCount();
    int numSects = Math.min(numComps, sectionSizes.length);

    int rowHeights[] = new int[ceiling(numComps, numSects)];
    int colWidths[]  = new int[numSects];

    if ((savedInfo == null) || (savedInfo.length != numComps))
      {
        savedInfo = new Component[numComps];
      }

    // collect dimensions and compute row heights
    Dimension subDims[] = new Dimension[numComps];
    for (int i = 0; i < numComps; i++)
      {
        int row = i / numSects;
        Component c = par.getComponent(i);
        if (c.isVisible())
          {
            Dimension dim =
              (minSize ? c.getMinimumSize() : c.getPreferredSize());
            // transform component dimensions according to orientation
            dim = orientation.transform(dim);

            subDims[i] = dim;
            rowHeights[row] = Math.max(rowHeights[row], dim.height);
          }
      }

    // compute total height
    int totalHeight = 0;
    boolean haveVisibleRow = false;
    for (int i = 0; i < rowHeights.length; i++)
      {
        int rowHeight = rowHeights[i];
        if (rowHeight > 0)
          {
            totalHeight += rowHeights[i];
            // add the separator gap as appropriate
            if (haveVisibleRow)
            {
                totalHeight += sepGap;
            }
            haveVisibleRow = true;
          }
      }

    // compute the column widths
    for (int i = 0; i < numComps; i++)
      {
        int col = i % numSects;
        Component c = par.getComponent(i);
        if (c.isVisible())
          {
            int sz = sectionSizes[col];
            int width = ((sz < 0) ? subDims[i].width : sz);
            colWidths[col] = Math.max(colWidths[col], width);
          }
      }

    // compute the total width
    int totalWidth = 0;
    boolean haveFirstVisible = false;
    for (int i = 0; i < colWidths.length; i++)
      {
        if (colWidths[i] > 0)
          {
            totalWidth += colWidths[i];
            // add the separator gap as appropriate
            if (haveFirstVisible)
            {
                totalWidth += sepGap;
            }
            haveFirstVisible = true;
          }
      }


    if (useSplitters && (splitterGap != sepGap))
      {
        totalWidth += adjustForSplitterGaps(par, numSects);
      }

    Dimension result =
      orientation.transform(new Dimension(totalWidth, totalHeight));

    // now add in the gaps, etc.
    result.width += lrEdgeGap * 2 + insets.left + insets.right;
    result.height += tbEdgeGap * 2 + insets.top + insets.bottom;

    return(result);
  }

  /**
   *  layout a container
   */
  @Override
public void layoutContainer(Container par)
  {
    int numComps = par.getComponentCount();
    int numSects = Math.min(numComps, sectionSizes.length);

    if (numComps < numSects)
      {
        System.out.println("ERROR: too few components in " + instanceName);
      }

    Dimension size = par.getSize();
    Insets insets = par.getInsets();

    int rowHeights[] = new int[ceiling(numComps, numSects)];
    int colWidths[]  = new int[numSects];


    if ((savedInfo == null) || (savedInfo.length != numComps))
      {
        savedInfo = new Component[numComps];
      }

    // these two are transformed in order to compute things in an
    // orientation independent manner
    Dimension dims = orientation.transform(size);
    Insets margins =
      orientation.transform(new Insets(insets.top + tbEdgeGap,
                                       insets.left + lrEdgeGap,
                                       insets.bottom + tbEdgeGap,
                                       insets.right + lrEdgeGap));

    int xPos = margins.left;
    int yPos = margins.top;

    Dimension insideDimensions =
      new Dimension(dims.width - margins.left - margins.right,
                    dims.height - margins.top - margins.bottom);

    // get the amount of space left over for components
    int remainderHeight = insideDimensions.height;
    int remainderWidth = insideDimensions.width;

    // compute the total number of resizeable sections based on the
    // visibility of components in the first row and the partition specs
    int totalResizeable = 0;
    for (int i = 0; i < numSects; i++)
      {
        try {
          Component comp = par.getComponent(i);
          if (comp.isVisible() && (getPartitionSpec(i) == REMAINDER))
        {
            totalResizeable++;
        }
        }
        catch (ArrayIndexOutOfBoundsException e) {
          System.out.println("Error in " + instanceName + ": " + e);
          throw(e);
        }
      }

    // compute the preferred row heights
    Dimension subDims[] = new Dimension[numComps];
    for (int i = 0; i < numComps; i++)
      {
        int row = i / numSects;
        Component comp = par.getComponent(i);
        if (comp.isVisible())
          {
            // the next line doesn't work with an || instead
            Dimension dim = (((savedInfo[i] == comp) && comp.isValid()) ?
                             comp.getSize() : comp.getPreferredSize());

            // transform component dimensions according to orientation
            dim = orientation.transform(dim);

            subDims[i] = dim;
            rowHeights[row] = Math.max(rowHeights[row], dim.height);
          }
      }

    // compute total height
    int totalHeight = 0;
    int numVisibleRows = 0;
    for (int i = 0; i < rowHeights.length; i++)
      {
        int rowHeight = rowHeights[i];
        if (rowHeight > 0)
          {
            totalHeight += rowHeights[i];
            // add the separator gap as appropriate
            if (numVisibleRows++ > 0)
            {
                totalHeight += sepGap;
            }
          }
      }

    // divide up the space for the rows proportionally
    for (int i = 0; i < rowHeights.length; i++)
      {
        float portion =
          ((float)(remainderHeight * rowHeights[i]) / totalHeight);
        rowHeights[i] = Math.max(1, Math.round(portion));
      }

    // compute the preferred column widths
    for (int i = 0; i < numComps; i++)
      {
        int col = i % numSects;
        Component c = par.getComponent(i);
        if (c.isVisible())
          {
            int width = subDims[i].width;
            colWidths[col] = Math.max(colWidths[col], width);
          }
      }

    // figure out the total number of REMAINDER components and the total
    // amount of REMAINDER space (which will be divided up proportionally)

    int remainderComponentsCombinedWidth = 0;
    int numVisible = 0;

    Component prevComp = null;
    boolean prevResizeable = false;
    int numResizeable = 0;
    int numSplitters = 0;

    for (int i = 0; i < numSects; i++)
      {
        Component comp = par.getComponent(i);
        if (comp.isVisible())
          {
            boolean resizeable = (getPartitionSpec(i) == REMAINDER);
            if (resizeable)
            {
                numResizeable++;
            }

            if (useSplitters)
              {
                // tally splitter AFTER comp
                if (resizeable && (numResizeable < totalResizeable))
                {
                    numSplitters++;
                }
                // tally splitter BEFORE comp
                if (resizeable && !prevResizeable && (numResizeable > 1))
                {
                    numSplitters++;
                }
              }
            prevComp = comp;
            prevResizeable = resizeable;

            numVisible++;

            if (sectionSizes[i] == REMAINDER)
            {
                remainderComponentsCombinedWidth += colWidths[i];
            }
            else if (sectionSizes[i] == PREFERRED)
            {
                remainderWidth -= colWidths[i];
            }
            else
            {
                remainderWidth -= sectionSizes[i];
            }
          }
      }

    // subtract out the separator gaps
    remainderWidth -= (sepGap * Math.max(0, numVisible - 1));
    remainderWidth -= (numSplitters * (splitterGap - sepGap));

    prevComp = null;
    numResizeable = 0;
    prevResizeable = false;

    // compute the actual placement of all the components and reshape
    // them to their required shapes
    for (int i = 0; i < numComps; i++)
      {
        int col = i % numSects;

        Component comp = par.getComponent(i);
        if (!comp.isVisible())
          {
            savedInfo[i] = null;
            continue;
          }

        boolean resizeable = (getPartitionSpec(col) == REMAINDER);
        if (resizeable)
        {
            numResizeable++;
        }

        if (useSplitters)
          {
            // tally splitter that come BEFORE comp
            if (resizeable && !prevResizeable && (numResizeable > 1))
            {
                xPos += (splitterGap - sepGap);
            }
          }

        int width = sectionSizes[col];
        int compWidth = width;
        int preferredColumnWidth = colWidths[col];

        // Save this for next time if it's really showing
        // This works around a bug in the Symantec version of the AWT
        savedInfo[i] = ((comp.isShowing() && comp.isValid()) ? comp : null);

        if (width == REMAINDER)
          {
            if (remainderComponentsCombinedWidth == 0)
              {
                //!! System.out.println("Odd sizes in partition!");
                width = Math.max(50, preferredColumnWidth);
              }
            else
              {
                // give component a proportional share of the remainder
                width = remainderWidth *
                  preferredColumnWidth / remainderComponentsCombinedWidth;
              }
            compWidth = width;
          }
        else if (width == PREFERRED)
          {
            width = preferredColumnWidth;
            compWidth = subDims[i].width;
          }

        Rectangle r = orientation.transform
          (new Rectangle(xPos, yPos, compWidth, rowHeights[i/numSects]));

        if (! r.equals(comp.getBounds()))
          {
            comp.setBounds(r.x, r.y, r.width, r.height);
          }

        int gap = sepGap;
        if (useSplitters)
          {
            // tally splitter that comes AFTER comp
            if (resizeable && (numResizeable < totalResizeable))
            {
                gap = splitterGap;
            }
          }
        prevComp = comp;
        prevResizeable = resizeable;

        // adjust the starting coordinates
        xPos += (width + gap);
        if (((i+1) % numSects) == 0)
          {
            // start a new row
            prevResizeable = false;
            numResizeable = 0;
            prevComp = null;
            yPos += rowHeights[i/numSects] + sepGap;
            xPos = margins.left;
          }
      }
  }

  int adjustForSplitterGaps(Container par, int numSects)
  {
    int numSplitters = 0;

    int totalResizeable = 0;
    for (int i = 0; i < numSects; i++)
      {
        Component comp = par.getComponent(i);
        if (comp.isVisible() && (getPartitionSpec(i) == REMAINDER))
        {
            totalResizeable++;
        }
      }

    Component prevComp = null;
    boolean prevResizeable = false;
    int numResizeable = 0;
    for (int i = 0; i < numSects; i++)
      {
        Component comp = par.getComponent(i);
        if (comp.isVisible())
          {
            boolean resizeable = (getPartitionSpec(i) == REMAINDER);
            if (resizeable)
            {
                numResizeable++;
            }

            // tally splitter AFTER comp
            if (resizeable && (numResizeable < totalResizeable))
            {
                numSplitters++;
            }
            // tally splitter BEFORE comp
            if (resizeable && !prevResizeable && (numResizeable > 1))
            {
                numSplitters++;
            }

            prevComp = comp;
            prevResizeable = resizeable;
          }
      }

    int adjustment = (numSplitters * (splitterGap - sepGap));

    return(adjustment);
  }

  final static int ceiling(int a, int b)
  {
    if ((a == 0) || (b == 0))
    {
        return(0);
    }
    int quot = a / b;
    int rem  = a % b;
    return(quot + ((rem == 0) ? 0 : 1));
  }



  ///
  ///  Representation
  ///

  private boolean                               layoutVertically;
  private int                                   sectionSizes[];
  private String                                instanceName;
  private transient Component                   savedInfo[];
  private Orientation                           orientation;

  // space added on the top and bottom of container
  private int                                   tbEdgeGap;

  // space added on the left and right of container
  private int                                   lrEdgeGap;

  // space added between components
  private int                                   sepGap;

  // space added between components for splitters
  private int                                   splitterGap;

  private boolean                               useSplitters;


}


