/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */

package modelobjects.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;


/**
 *  AnnotatedObject is used as a common superclass for types that can
 *  have a String description and a set of properties, and that can write
 *  themselves out as XML.
 */
public abstract class AnnotatedObject extends XmlWritable
{
  protected AnnotatedObject()
  {
    this("?");
  }

  protected AnnotatedObject(String xmlElementName)
  {
    super(xmlElementName);
    descriptionTagName = "Description";
    propertyTagName = "Property";
  }

  public String getDescription()
  {
    return(description);
  }

  public void setDescription(String description)
  {
    this.description = description;
  }

  public Object getProperty(String propertyName)
  {
    return(getProperty(propertyName, null));
  }

  public Object getProperty(String propertyName, Object defaultValue)
  {
    Object result =
      ((properties == null) ? null : properties.get(propertyName));

    return((result != null) ? result : defaultValue);
  }

  public void setProperty(String propertyName, Object propertyValue)
  {
    if (properties == null)
    {
        properties = new HashMap();
    }
    if (propertyValue == null)
    {
        properties.remove(propertyName);
    }
    else
    {
        properties.put(propertyName, propertyValue);
    }
  }

  public void removeProperty(String propertyName)
  {
    if (properties != null)
    {
        properties.remove(propertyName);
    }
  }

  public Set getPropertyNames()
  {
    return((properties == null) ? Collections.EMPTY_SET : properties.keySet());
  }

  @Override
public List getChildren()
  {
    ArrayList list = new ArrayList();
    if (description != null)
    {
        list.add(makePCDataElement(descriptionTagName, description, true));
    }
    Iterator propNames = getPropertyNames().iterator();
    while (propNames.hasNext()) {
      String name = (String)propNames.next();
      Object val  = properties.get(name);
      list.add(new NamedProperty(propertyTagName, name, val));
    }
    return(list);
  }

  public String getDescriptionTagName()
  {
    return(descriptionTagName);
  }

  public void setDescriptionTagName(String descriptionTagName)
  {
    this.descriptionTagName = descriptionTagName;
  }

  public String getPropertyTagName()
  {
    return(propertyTagName);
  }

  public void setPropertyTagName(String propertyTagName)
  {
    this.propertyTagName = propertyTagName;
  }

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///   private representation
  ///
  /////////////////////////////////////////////////////////////////////////////

  protected String      descriptionTagName;
  protected String      propertyTagName;
  protected String      description;
  protected HashMap     properties;


  /////////////////////////////////////////////////////////////////////////////
  ///
  ///   inner classes
  ///
  /////////////////////////////////////////////////////////////////////////////

  public static class NamedProperty extends XmlWritable
  {
    public NamedProperty(String xmlElementName, String propertyName,
                         Object propertyValue)
    {
      super(xmlElementName);

      if (propertyName == null)
    {
        throw(new IllegalArgumentException("null propertyName"));
    }

      this.propertyName = propertyName;
      this.propertyValue = propertyValue;
    }

    public String getPropertyName()
    {
      return(propertyName);
    }

    public Object getPropertyValue()
    {
      return(propertyValue);
    }

    @Override
    public Map getAttributes()
    {
      Map attrs = super.getAttributes();
      String val =
        ((propertyValue == null) ? "null" : propertyValue.toString());
      attrs.put("name", propertyName);
      attrs.put("value", val);
      return(attrs);
    }

    @Override
    public List getChildren()
    {
      return(Collections.EMPTY_LIST);
    }

    private String  propertyName;
    private Object  propertyValue;
  }

}
