/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */

package modelobjects.util;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.EventListener;
import java.util.EventObject;
import java.util.Hashtable;
import java.util.Vector;


/**
 *  This class provides common functionality for multicast event sources.
 *  It supports most forms of the Event Listener design pattern, including
 *  the case where some listener methods can throw checked exceptions.
 */
public abstract class CommonEventSupport implements Serializable
{
  protected CommonEventSupport()
  {
    this(null);
  }

  protected CommonEventSupport(Object eventSource)
  {
    this.eventSource = eventSource;
    listeners = null;
  }

  protected Object getEventSource()
  {
    return(eventSource);
  }

  public boolean anyListeners()
  {
    return((listeners != null) && (!listeners.isEmpty()));
  }

  protected synchronized void addEventListener(EventListener listener)
  {
    Vector v = ((listeners == null) ? new Vector(2) :
                (Vector)listeners.clone());
    if (!v.contains(listener))
      {
        v.addElement(listener);
        listeners = v;
      }
  }

  protected synchronized void removeEventListener(EventListener listener)
  {
    if ((listeners != null) && listeners.contains(listener))
      {
        Vector v = (Vector)listeners.clone();
        v.removeElement(listener);
        listeners = v;
      }
  }

  protected void fireEvent(Class listenerInterface,
                           String listenerMethodName,
                           EventObject evt)
  {
    if ((listeners == null) || (listeners.isEmpty()))
    {
        return;
    }

    try
      {
        fireEvent(listenerInterface, listenerMethodName, evt,
                  noExpectedExceptions);
      }
    catch (Throwable e)
      {
        // this shouldn't happen
        System.err.println("Error in EventListener method " +
                           listenerMethodName + ": " + e);
      }
  }

  protected void fireEvent(Class listenerInterface,
                           String listenerMethodName,
                           EventObject evt,
                           Class expectedExceptionClass)
       throws Throwable
  {
    if ((listeners == null) || (listeners.isEmpty()))
    {
        return;
    }

    fireEvent(listenerInterface, listenerMethodName, evt,
              new Class[] { expectedExceptionClass });
  }

  protected void fireEvent(Class listenerInterface,
                           String listenerMethodName,
                           EventObject evt,
                           Class expectedExceptionClasses[])
       throws Throwable
  {
    if ((listeners == null) || (listeners.isEmpty()))
    {
        return;
    }

    Method listenerMethod;
    try
      {
        Class paramTypes[] = { evt.getClass() };
        listenerMethod =
          lookupMethod(listenerInterface, listenerMethodName, paramTypes);
      }
    catch (NoSuchMethodException e)
      {
        System.err.println("No such method: " +
                           listenerInterface.getName() + "." +
                           listenerMethodName + ": " + e.getMessage());
        return;
      }

    try
      {
        fireEvent(listenerMethod, evt, expectedExceptionClasses);
      }
    catch (Exception e)
      {
        for (int i = 0; i < expectedExceptionClasses.length; i++)
          {
            if (expectedExceptionClasses[i].isInstance(e))
            {
                throw(e);
            }
          }

        System.err.println("Error in EventListener method " +
                           listenerInterface.getName() + "." +
                           listenerMethodName + ": " + e);
      }
  }

  protected Method lookupMethod(Class listenerInterface,
                                String listenerMethodName,
                                Class paramTypes[])
       throws NoSuchMethodException
  {

    Method m = (Method)MethodCacheLazyHolder.METHOD_CACHE.get(listenerMethodName);

    if ((m == null) || (m.getDeclaringClass() != listenerInterface))
      {
        m = listenerInterface.getMethod(listenerMethodName, paramTypes);
        MethodCacheLazyHolder.METHOD_CACHE.put(listenerMethodName, m);
      }

    return(m);
  }

  protected void fireEvent(Method listenerMethod, EventObject evt,
                           Class expectedExceptionClasses[])
       throws Throwable
  {
    Vector listeners = this.listeners;
    if ((listeners == null) || (listeners.isEmpty()))
    {
        return;
    }

    Object args[] = { evt };
    Exception e = null;

    for (int i = 0; i < listeners.size(); i++)
      {
        try
          {
            listenerMethod.invoke(listeners.elementAt(i), args);
          }
        catch (IllegalAccessException e1)
          { e = e1; }
        catch (InvocationTargetException e2)
          {
            // first see if we're supposed to rethrow the exception
            Throwable targetException = e2.getTargetException();
            for (int j = 0; j < expectedExceptionClasses.length; j++)
              {
                if (expectedExceptionClasses[j].isInstance(targetException))
                {
                    throw(targetException);
                }
              }

            // if not, just report it to System.err
            e = e2;
            System.err.println("Event in EventListener method " +
                               listenerMethod.getName() + ": " +
                               targetException.getMessage());
            targetException.printStackTrace(System.err);
          }
        catch (RuntimeException bug)
          {
            System.err.println("Error in EventListener method " +
                               listenerMethod.getName() + ": " + bug);
            return;
          }

        if (e != null)
          {
            System.err.println("Event listener error in method " +
                               listenerMethod.getName() + ": " +
                               e.getMessage());
            return;
          }
      }
  }

  private void writeObject(ObjectOutputStream stream)
       throws java.io.IOException
  {
    stream.defaultWriteObject();

    Vector listeners = null;
    synchronized (this)
      {
        if (this.listeners != null)
        {
            listeners = (Vector)(this.listeners.clone());
        }
      }

    // write a null-terminated sequence of serializable listeners
    int num = ((listeners == null) ? 0 : listeners.size());

    for (int i = 0; i < num; i++)
      {
        Object listener = listeners.elementAt(i);
        if (listener instanceof Serializable)
        {
            stream.writeObject(listener);
        }
      }
    stream.writeObject(null);
  }

  private void readObject(ObjectInputStream stream)
       throws IOException, ClassNotFoundException
  {
    stream.defaultReadObject();

    while (true)
      {
        Object listener = stream.readObject();
        if (listener == null)
        {
            break;
        }
        if (listeners == null)
        {
            listeners = new Vector(2);
        }
        listeners.addElement(listener);
      }
  }

    /**
     * For debugging support
     * @return string representation of CommonEventSupport
     */
  @Override
public String toString()
  {
    StringBuilder sBuilder = new StringBuilder(getClass().getName());
    sBuilder.append(" ( ");
    sBuilder.append("Event Source : ").append(eventSource).append(" ; ");
    sBuilder.append("Listeners : ").append(listeners);
    sBuilder.append(" )");
    return sBuilder.toString();
  }

  private static class MethodCacheLazyHolder {
      private static final Hashtable METHOD_CACHE = new Hashtable();    
  }
  
  private Object eventSource;
  private transient Vector listeners;
  
  private static final Class noExpectedExceptions[] = new Class[0];
  
}
