/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */

package modelobjects.util;

import java.lang.reflect.Constructor;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;

/**
 *  EnumGroup represents a group of related EnumValues in a manner
 *  similar to an enumeration type.  Each instance of EnumGroup typically
 *  manages a collection of a different subclass of EnumValue.
 */
public class EnumGroup
{
  /**
   *  Construct an EnumGroup with the specified name.
   */
  public EnumGroup(String enumGroupName)
  {
    this(enumGroupName, false);
  }

  /**
   *  Construct an EnumGroup with the specified name, and specifying
   *  whether or not the EnumGroup should be sealed.
   */
  public EnumGroup(String enumGroupName, boolean sealed)
  {
    this(enumGroupName, null, sealed);
  }

  /**
   *  Construct an EnumGroup with the specified name and parent EnumGroup,
   *  and specifying whether or not the EnumGroup should be sealed.
   *  Parent EnumGroups can be used to help implement subclass relationships
   *  between EnumValue subtypes.
   */
  public EnumGroup(String enumGroupName, EnumGroup parentGroup, boolean sealed)
  {
    if (getEnumGroup(enumGroupName) != null) {
      throw(new IllegalStateException("duplicate EnumGroup name: " +
                                      enumGroupName));
    }

    this.enumGroupName = enumGroupName;
    this.parentGroup   = parentGroup;
    this.sealed        = sealed;
    this.instanceMap   = new HashMap();

    ENUM_GROUP_EXTENT.put(enumGroupName, this);
  }

  /**
   *  Record an EnumValue in this EnumGroup, by both name and code.
   *  This method is intended only to be called by EnumValue's constructor.
   */
  void recordEnumValue(EnumValue val)
  {
    recordEnumValue(val, true, true);
  }

  /**
   *  Record an EnumValue in this EnumGroup.  This method is intended only
   *  to be called by EnumValue's constructor.  This method allows the caller
   *  to specify whether the value should be recorded by name, by code, or
   *  by both name and code.
   */
  void recordEnumValue(EnumValue val, boolean byName, boolean byCode)
  {
    if (byName)
    {
        instanceMap.put(val.getName(), val);
    }
    if (byCode)
    {
        instanceMap.put(new Integer(val.getCode()), val);
    }
  }

  /**
   *  Return the name of this EnumGroup.
   */
  public String getEnumGroupName()
  {
    return(enumGroupName);
  }

  /**
   *  Return the EnumValue with the specified integer code recorded in this
   *  EnumGroup.  If more than one EnumValue had the same integer code, the
   *  last one recorded by recordEnumValue will be returned.
   */
  public EnumValue getByCode(int code)
  {
    EnumValue result = (EnumValue)instanceMap.get(new Integer(code));
    if ((result != null) || (parentGroup == null))
    {
        return(result);
    }
    else
    {
        return(parentGroup.getByCode(code));
    }
  }

  /**
   *  Return the EnumValue with the specified name recorded in this
   *  EnumGroup.  If more than one EnumValue had the same name,
   *  the last one recorded by recordEnumValue will be returned.
   */
  public EnumValue getByName(String name)
  {
    EnumValue result = (EnumValue)instanceMap.get(name);
    if ((result != null) || (parentGroup == null))
    {
        return(result);
    }
    else
    {
        return(parentGroup.getByName(name));
    }
  }

  /**
   *  Create an EnumValue suitable for this EnumGroup.
   *
   *  The default method tries to create a instance of EnumValue subclass
   *  if it can find a public constructor with parameter types (int, String).
   *  If no such public constructor can be found, it creates a direct
   *  instance of EnumValue.
   *
   *  Subclasses of EnumGroup may wish to override this method to
   *  create instances of corresponding EnumValue subclasses, or to
   *  return null if the EnumValue subclass is logically sealed.
   */
  protected EnumValue makeEnumValue(String name, int code)
  {
    if (isSealed()) {
      return(null);
    }
    else if (constructSubclassInstance) {
      try {
        if (enumValueCtor == null) {
          // look in subclass for a public constructor that takes (int, String).
          Class enumValueSubclass = Class.forName(enumGroupName);
          Class[] ctorParams = new Class[] { int.class, String.class };
          enumValueCtor = enumValueSubclass.getConstructor(ctorParams);
        }
        Object[] ctorArgs = new Object[] { new Integer(code), name };
        return((EnumValue)enumValueCtor.newInstance(ctorArgs));
      }
      catch (Exception e) {
        // oh well...
        System.err.println
          ("Unable to make instance of " + enumGroupName + ": " + e);
        constructSubclassInstance = false;
      }
    }

    // fall through to 
    return(new EnumValue(this, code, name));
  }

  /**
   *  Look for an EnumValue in this EnumGroup with the specified name,
   *  returning it if it is found, or creating it otherwise by calling
   *  makeEnumValue(String, int).  It is permissible for a subclass of
   *  EnumGroup to override makeEnumValue to return null if the EnumValue
   *  subclass is logically sealed.
   */
  public EnumValue findOrCreateEnumValue(String name, int code)
  {
    EnumValue value = getByName(name);

    if ((value == null) || (value.getCode() != code)) {
      value = makeEnumValue(name, code);
    }

    return(value);
  }

  /**
   *  Return a Set of all the EnumValues in this EnumGroup.  Modification
   *  of this set has no effect on the EnumGroup.
   */
  public Set getValues()
  {
    Set result = ((parentGroup != null) ?
                  parentGroup.getValues() : new HashSet());
    result.addAll(instanceMap.values());
    return(result);
  }

  /**
   *  Return the EnumGroup that was constructed with the specified name,
   *  or null if there is no such EnumGroup.
   */
  public static EnumGroup getEnumGroup(String enumGroupName)
  {
    return(getEnumGroup(enumGroupName, false));
  }

  /**
   *  Return the EnumGroup that was constructed with the specified name,
   *  creating it if necessary and specified, or else return null.
   */
  public static EnumGroup getEnumGroup(String enumGroupName,
                                       boolean createIfNeeded)
  {
    EnumGroup group = (EnumGroup)ENUM_GROUP_EXTENT.get(enumGroupName);
    if ((group == null) && createIfNeeded)
    {
        group = new EnumGroup(enumGroupName);
    }
    return(group);
  }

  /**
   *  Return the default EnumValue for this EnumGroup, or null if there
   *  is no default.  The default value is used during deserialization if
   *  a value is received that is not recorded by this EnumGroup.
   */
  public EnumValue getDefaultValue()
  {
    return(defaultValue);
  }

  /**
   *  Assign the default EnumValue for this EnumGroup.
   */
  public void setDefaultValue(EnumValue defaultValue)
  {
    this.defaultValue = defaultValue;
  }

  /**
   *  Return whether this EnumGroup is sealed.
   */
  public boolean isSealed()
  {
    return(sealed);
  }

  /**
   *  Assign whether this EnumGroup is sealed.
   */
  public void setSealed(boolean sealed)
  {
    this.sealed = sealed;
  }

  /**
   *  Return a human readable String representing this EnumGroup.
   */
  @Override
public String toString()
  {
    String parentGroupPart = ((parentGroup == null) ? "" :
                              (" parentGroup=" + parentGroup.enumGroupName));
    return("<EnumGroup " + enumGroupName + parentGroupPart + ">");
  }

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  private representation
  ///
  /////////////////////////////////////////////////////////////////////////////

  private               String          enumGroupName;
  private               HashMap         instanceMap;
  private               EnumValue       defaultValue;
  private               boolean         sealed;
  private               EnumGroup       parentGroup;

  private transient     Constructor     enumValueCtor;
  private transient     boolean         constructSubclassInstance = true;

  private static        HashMap         ENUM_GROUP_EXTENT = new HashMap();
}
