/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */

package modelobjects.util;

import java.io.ObjectStreamException;
import java.io.Serializable;

/**
 *  EnumValue represents a typed enumeration value.
 *
 *  Instances of Enumvalue subclasses may be serialized as direct
 *  instance of EnumValue, and deserialized back into the corresponding
 *  values by the VM performing the deserialization, allowing a large
 *  degree of type independence between the sender and receiver.
 */
public class EnumValue implements Serializable, Comparable
{
  /**
   *  Construct an EnumValue for the specified EnumGroup, code and name.
   */
  public EnumValue(EnumGroup enumGroup, int code, String name)
  {
    this.group = enumGroup;
    groupName  = enumGroup.getEnumGroupName();

    this.code  = code;
    this.name  = name;

    // record the enum value in the EnumGroup 
    enumGroup.recordEnumValue(this);
  }

  /**
   *  Return this EnumValue's EnumGroup.
   */
  public EnumGroup getGroup()
  {
    if ((group == null) && (groupName != null))
    {
        group = EnumGroup.getEnumGroup(groupName);
    }
    return(group);
  }

  /**
   *  Return this EnumValue's int code.
   */
  public int getCode()
  {
    return(code);
  }

  /**
   *  Return this EnumValue's name.
   */
  public String getName()
  {
    return(name);
  }

  /**
   *  Hash EnumValues by their integer codes
   */
  @Override
public int hashCode()
  {
    return(code);
  }

  /**
   *  Compare EnumValues based on their codes.
   */
  @Override
public int compareTo(Object other)
  {
    EnumValue that = (EnumValue)other;

    int result = this.code - that.code;
    if (result != 0)
    {
        return(result);
    }
    else
    {
        return(this.name.compareTo(that.name));
    }
  }

  /**
   *  Treat two EnumValues as equal if they have the same EnumGroup, name,
   *  and code.
   */
  @Override
public boolean equals(Object obj)
  {
	if (obj == null || this.getClass() != obj.getClass()) {
		return false;
	}
	if (this == obj) {
      return true;
    }
	  EnumValue that = (EnumValue)obj;
	  return (this.group == that.group) &&
	         (this.code == that.code) &&
	         (this.name.equals(that.name));
  }

  /**
   *  Return a human readable String representing this EnumValue.
   */
  @Override
public String toString()
  {
    int lastDot = groupName.lastIndexOf('.');
    String group = ((lastDot == -1) ? groupName :
                    groupName.substring(lastDot+1));
    return(group + ":" + name + "=" + code);
  }

  /**
   *  Return the EnumValue with the same group name and enum name as this
   *  EnumValue, but that is actually recorded with that group in this VM.
   *  This method is used to guarantee uniqueness of EnumValue instances,
   *  and is used automatically to canonicalize EnumValue references during
   *  deserialization.
   *  If the EnumGroup is not known, it is created.  A local instance of
   *  EnumValue is obtained by calling the findOrCreateEnumValue method
   *  on the EnumGroup.  If that method returns null, the value returned
   *  is the EnumGroup's default value, if any.
   */
  public EnumValue getRecordedEnumValue()
  {
    EnumGroup enumGroup = EnumGroup.getEnumGroup(this.groupName);

    // create the EnumGroup if it doesn't already exist
    if (enumGroup == null)
    {
        enumGroup = new EnumGroup(this.groupName);
    }

    // use findOrCreateEnumValue to find or create a local value
    EnumValue localValue =
      enumGroup.findOrCreateEnumValue(this.name, this.code);

    // if there is no local value, return the default value if there is one
    return((localValue != null) ? localValue : enumGroup.getDefaultValue());
  }

  /**
   *  Return whether or not deserialization of an instance of a subclass
   *  of EnumValue should substitute an equivalent direct instance of
   *  EnumValue.  The default implementation of this methods always returns
   *  true.
   */
  protected boolean neutralizeSubclassInstancesOnDeserialization()
  {
    return(true);
  }

  /**
   *  Canonicalize all deserialized EnumValues to reference instances
   *  previously recorded in this VM.
   */
  protected Object readResolve() throws ObjectStreamException
  {
    if (this.getClass() == EnumValue.class) {
      // return the value of the right type from this VM, or null if not known
      return(getRecordedEnumValue());
    }
    else if (neutralizeSubclassInstancesOnDeserialization()) {
      // neutralize the EnumValue subclass instance for transport
      return(new EnumValue(getGroup(), code, name));
    }
    else {
      Object replacement = this.getRecordedEnumValue();
      return((replacement != null) ? replacement : this);
    }
  }

  /**
   *  Create a new EnumValue in the same group as this EnumValue, and with
   *  the same code as this EnumValue, but with the specified name.
   */
  public EnumValue makeDerivedValue(String derivedValueName)
  {
    EnumGroup enumGroup = this.getGroup();
    EnumValue derivedValue =
      enumGroup.findOrCreateEnumValue(derivedValueName, this.code);
    return(derivedValue);
  }

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  private representation
  ///
  /////////////////////////////////////////////////////////////////////////////

  static final long serialVersionUID = -7751142842063816257L;

  private               String          groupName;
  private transient     EnumGroup       group;
  private               int             code;
  private               String          name;
}
