/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */
package modelobjects.util;

import java.awt.Component;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.beans.PropertyEditorSupport;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import javax.swing.JComboBox;

/**
 * EnumeratedTypePropertyEditor is a PropertyEditor for types where more than 
 * one value is possible but only one is permissible.
 */
public class EnumeratedTypePropertyEditor extends PropertyEditorSupport
  implements ItemListener
{
  /**
   *  Construct an EnumeratedTypePropertyEditor with no property name
   *  and no entries.
   */
  public EnumeratedTypePropertyEditor()
  {
    this("", new ArrayList(), new ArrayList());
  }

  /**
   *  Construct an EnumeratedTypePropertyEditor the specified property
   *  name, and which derives its entries from the specified view-to-model
   *  HashMap.
   */
  public EnumeratedTypePropertyEditor(String propertyName, 
                                      HashMap viewToModelMap)
  {
    this(propertyName,
         new ArrayList(viewToModelMap.size()),
         new ArrayList(viewToModelMap.size()));

    Iterator iterator = viewToModelMap.keySet().iterator();
    while (iterator.hasNext()) {
      String viewValue  = (String)iterator.next();
      Object modelValue = viewToModelMap.get(viewValue);

      addViewAndModelPair(viewValue, modelValue);
    }
  }

  /**
   *  Construct an EnumeratedTypePropertyEditor the specified property
   *  name, and which derives its entries from the specified view-values
   *  array and model-values array.
   */
  public EnumeratedTypePropertyEditor(String propertyName, 
                                      String[] viewValues,
                                      Object[] modelValues)
  {
    this(propertyName, makeList(viewValues), makeList(modelValues));
  }

  /**
   *  Construct an EnumeratedTypePropertyEditor the specified property
   *  name, and which derives its entries from the specified view-values
   *  List and model-values List.
   */
  public EnumeratedTypePropertyEditor(String propertyName, 
                                      List viewValues,
                                      List modelValues)
  {
    this.propertyName = propertyName;
    this.viewValues   = viewValues;
    this.modelValues  = modelValues;

    if (viewValues.size() != modelValues.size()) {
      throw(new IllegalArgumentException
        ("view and model value lists must have the same length"));
    }

    //set up combo box with the keys of the view values.
    Object[] keyVals = viewValues.toArray();
    comboBox = new JComboBox(keyVals);
    comboBox.addItemListener(this);
  }

  /**
   *  Add a view-value and model-value association to this
   *  EnumeratedTypePropertyEditor.
   */
  public final void addViewAndModelPair(String viewValue, Object modelValue)
  {
    viewValues.add(viewValue);
    modelValues.add(modelValue);
    comboBox.addItem(viewValue);
  }

  @Override
public boolean supportsCustomEditor()
  {
    return(true);
  }

  @Override
public Component getCustomEditor()
  {
    return(comboBox);
  }

  /**
   *  Set the PropertyEditor's "model" value.
   */
  @Override
public void setValue(Object value)
  {
    if (value == null) {
      comboBox.setSelectedItem("");
    } 
    else {
      int index = modelValues.indexOf(value);
      if (index == -1)
    {
        comboBox.setSelectedItem("");
    }
    else
    {
        comboBox.setSelectedItem(viewValues.get(index));
    }
    }
  }

  /**
   *  Set the PropertyEditor's "view" value.
   */
  @Override
public void setAsText(String stringValue)
  {
    comboBox.setSelectedItem(stringValue);
  }

  @Override
public Object getValue()
  {
    String viewValue = getAsText();
    if (viewValue == null)
    {
        return(null);
    }
    int index = viewValues.indexOf(viewValue);
    return((index == -1) ? null : modelValues.get(index));
  }

  @Override
public String getAsText()
  {
    String currentViewValue = (String)comboBox.getSelectedItem();
    return(currentViewValue);
  }

  public String getPropertyName()
  {
    return(propertyName);
  }

  public void setPropertyName(String propertyName)
  {
    this.propertyName = propertyName;
  }

  ///  ItemListener interface method
  @Override
public void itemStateChanged(ItemEvent e)
  {
    firePropertyChange();
  }

  private static List makeList(Object[] values)
  {
    int num = values.length;
    ArrayList result = new ArrayList(num);
    for (int i = 0; i < num; i++)
    {
        result.add(values[i]);
    }
    return(result);
  }

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  representation
  ///
  /////////////////////////////////////////////////////////////////////////////

  private JComboBox  comboBox;
  private String     propertyName;

  private List       viewValues;
  private List       modelValues;
}
