/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */
package modelobjects.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

/**
 *  ExtensionsClassLoader can be used to load classes from a runtime-specified
 *  list of  zip files, jar files, and directories.
 */
public class ExtensionsClassLoader extends ClassLoader
{
  /**
   *  Construct a ExtensionsClassLoader to search for classes and
   *  class resources in the specified list of zip files, jar files,
   *  and directories, and using the System ClassLoader as the parent
   *  ClassLoader.
   *
   *  @param classPathFiles an array of zip file, jar files, and directories.
   */
  public ExtensionsClassLoader(File[] classPathFiles)
    throws IOException
  {
    this(classPathFiles, ClassLoader.getSystemClassLoader());
  }

  /**
   *  Construct a ExtensionsClassLoader to search for classes and
   *  class resources in the specified list of zip files, jar file,
   *  and directories, and using the specified parent ClassLoader.
   *
   *  @param classPathFiles an array of zip file, jar files, and directories.
   *  @param parentClassLoader the parent ClassLoader this one delegates to
   */
  public ExtensionsClassLoader(File[] classPathFiles,
                               ClassLoader parentClassLoader)
    throws IOException
  {
    super(parentClassLoader);
    parent = parentClassLoader;

    pathLength  = classPathFiles.length;
    pathEntries = new ZipFile[pathLength];
    absFiles    = new File[pathLength];
    for (int i = 0; i < pathLength; i++) {
      File pathEntryFile = classPathFiles[i].getAbsoluteFile();
      absFiles[i] = pathEntryFile;
      if (pathEntryFile.isDirectory())
    {
        pathEntries[i] = pathEntryFile;
    }
    else
    {
        pathEntries[i] = new ZipFile(pathEntryFile);
    }
    }
  }

  /**
   *  Return a URL for the specified ClassLoader resource.
   *  If the resource is found, the URL will be of the form
   *  jar:file:[jar-file-absolute-path]![resource-name]
   *
   *  @param resourceName resouce name which does not begin with an
   *         initial slash, but in which directories and packages
   *         are delimited by forward slashes.
   */
  @Override
protected URL findResource(String resourceName) {
    for (int i = 0; i < pathLength; i++) {
      Object pathEntry = pathEntries[i];
      if (pathEntry instanceof ZipFile) {
        ZipFile  zipFile  = (ZipFile)pathEntry;
        ZipEntry zipEntry = zipFile.getEntry(resourceName);
        if (zipEntry != null) {
          try {
            String zipFileUrlString = absFiles[i].toURL().toExternalForm();
            return(new URL("jar:" + zipFileUrlString + "!" + resourceName));
          }
          catch (MalformedURLException badURL) {
            return(null);
          }
        }
      }
      else {    // entry is a directory
        File directory = (File)pathEntry;
        File dirEntry  =
          new File(directory, resourceName.replace('/', File.separatorChar));
        if (dirEntry.exists()) {
          try {
            return(dirEntry.toURL());
          }
          catch (MalformedURLException badURL) {
            return(null);
          }
        }
      }
    }

    return(super.findResource(resourceName));
  }

  @Override
protected Class findClass(String className)
    throws ClassNotFoundException
  {
    String convertedClassName = className.replace('.', '/') + ".class";

    byte[] classBytes = findClassBytes(convertedClassName);

    if (classBytes != null)
    {
        return defineClass(className, classBytes, 0, classBytes.length);
    }
    else
    {
        return(super.findClass(className));
    }
  }

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  private internals
  ///
  /////////////////////////////////////////////////////////////////////////////

  /**
   *  Get the bytes for the specified class, or null if it cannot be found.
   */
  private byte[] findClassBytes(String className)
  {
    for (int i = 0; i < pathLength; i++) {
      Object pathEntry = pathEntries[i];
      if (pathEntry instanceof ZipFile) {
        ZipFile  zipFile  = (ZipFile)pathEntry;

        ZipEntry zipEntry = zipFile.getEntry(className);
        if (zipEntry != null) {
          try {
            return(getZipEntryBytes(zipFile, zipEntry));
          }
          catch (IOException ioe) {
            break;
          }
        }
      }
      else {    // entry is a directory
        File directory = (File)pathEntry;
        File dirEntry  = new File(directory, className.replace('/', SEPCHAR));
        if (dirEntry.exists()) {
          try {
            return(getClassFileBytes(dirEntry));
          }
          catch (IOException ioe) {
            break;
          }
        }
      }
    }

    return(null);
  }

  /**
   *  Read the data of the specified ZipEntry and return as a byte array.
   */
  private byte[] getZipEntryBytes(ZipFile zipFile, ZipEntry classFileZipEntry)
    throws IOException
  {
    long size = classFileZipEntry.getSize();
    byte[] result = new byte[(int)size];
    InputStream inStream = zipFile.getInputStream(classFileZipEntry);

    long bytesRead = 0;
    while (bytesRead < size) {
      int num = inStream.read(result, (int)bytesRead, (int)(size-bytesRead));
      bytesRead += num;
    }
    inStream.close();

    return(result);
  }

  /**
   *  Read the data from the specified File and return as a byte array.
   */
  private byte[] getClassFileBytes(File classFile)
    throws IOException
  {
    long size = classFile.length();
    byte[] result = new byte[(int)size];
    InputStream inStream = new FileInputStream(classFile);

    long bytesRead = 0;
    while (bytesRead < size) {
      int num = inStream.read(result, (int)bytesRead, (int)(size-bytesRead));
      bytesRead += num;
    }
    inStream.close();

    return(result);
  }

  // arguments are: classname archive1 archive2 ...
  public static void main(String[] args)
  {
    try {
      File[] classPathFiles = new File[args.length - 1];
      for (int i = 1; i < args.length; i++)
    {
        classPathFiles[i-1] = new File(args[i]);
    }

      ExtensionsClassLoader loader = new ExtensionsClassLoader(classPathFiles);
      Class cl = loader.loadClass(args[0]);
      System.out.println("cl = " + cl);
      Object obj = cl.newInstance();
      System.out.println("obj = " + obj);
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }

  private static final  char SEPCHAR = File.separatorChar;
  private ClassLoader   parent;
  private int           pathLength;
  private File[]        absFiles;
  private Object[]      pathEntries;
}

