/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */

package modelobjects.util;

/**
 *  FindClass is used to find Java types.  It provides functionality
 *  similar to that provided by Class.forName(String), but also can
 *  be used for primitive types and array types.
 */
public class FindClass
{
  public static Class forName(String className)
       throws ClassNotFoundException
  {
    return(forName(className, true, FindClass.class.getClassLoader()));
  }

  public static Class forName(String className, boolean initialize,
                              ClassLoader classLoader)
       throws ClassNotFoundException
  {
    if (className.endsWith("[]"))
      {
        String base = className.substring(0, className.length() - 2).trim();
        return(findArrayType(base, initialize, classLoader));
      }

    if (className.indexOf('.') == -1)
      {
        if (className.equals("boolean"))
        {
            return(boolean.class);
        }
        if (className.equals("int"))
        {
            return(int.class);
        }
        if (className.equals("char"))
        {
            return(char.class);
        }
        if (className.equals("byte"))
        {
            return(byte.class);
        }
        if (className.equals("short"))
        {
            return(short.class);
        }
        if (className.equals("long"))
        {
            return(long.class);
        }
        if (className.equals("float"))
        {
            return(float.class);
        }
        if (className.equals("double"))
        {
            return(double.class);
        }
        if (className.equals("void"))
        {
            return(void.class);
        }
      }

    return(Class.forName(className, initialize, classLoader));
  }

  public static String getName(Class cl, boolean trimPackage)
  {
    String name = cl.getName();
    int arrayCount = 0;
    while (name.startsWith("[")) {
      name = name.substring(1);
      arrayCount++;
    }
    if (arrayCount > 0) {
      switch (name.charAt(0)) {
        case 'Z': name = "boolean"; break;
        case 'C': name = "char";    break;
        case 'B': name = "byte";    break;
        case 'S': name = "short";   break;
        case 'I': name = "int";     break;
        case 'J': name = "long";    break;
        case 'F': name = "float";   break;
        case 'D': name = "double";  break;
        case 'L':
          name = name.substring(1, name.length()-1).replace('/','.');
          break;
      }

      for (int i = 0; i < arrayCount; i++)
    {
        name += "[]";
    }
    }

    if (trimPackage) {
      int lastDot = name.lastIndexOf('.');
      if (lastDot != -1)
    {
        name = name.substring(lastDot + 1);
    }
    }

    return(name);
  }

  private static Class findArrayType(String className, boolean initialize,
                                     ClassLoader classLoader)
       throws ClassNotFoundException
  {
    if (className.indexOf('.') == -1)
      {
        if (className.equals("char"))
        {
            className = "[C";
        }
        else if (className.equals("boolean"))
        {
            className = "[Z";
        }
        else if (className.equals("int"))
        {
            className = "[I";
        }
        else if (className.equals("byte"))
        {
            className = "[B";
        }
        else if (className.equals("short"))
        {
            className = "[S";
        }
        else if (className.equals("long"))
        {
            className = "[J";
        }
        else if (className.equals("float"))
        {
            className = "[F";
        }
        else if (className.equals("double"))
        {
            className = "[D";
        }
        else if (className.equals("void"))
        {
            className = "[V";
        }
        else
        {
            className = "[L" + className + ";";
        }
      }
    else
      {
        className = "[L" + className + ";";
      }

    return(Class.forName(className, initialize, classLoader));
  }

  private FindClass()
  {
    // no need to instantiate
  }
}
