/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */

package modelobjects.util;

import java.io.Serializable;
import java.util.AbstractSet;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.SortedSet;

/**
 *  IndexedSortedSet is an indexable SortedSet.  In addition to the usual
 *  SortedSet methods, it also defines methods for finding the index of
 *  an element, and for finding the element at a specified index.
 */
public class IndexedSortedSet extends AbstractSet
  implements SortedSet, IndexedSet, ResortableSortedSet, Serializable
{
  public IndexedSortedSet()
  {
    this(null, DEFAULT_INITIAL_CAPACITY);
  }

  public IndexedSortedSet(Comparator comparator)
  {
    this(comparator, DEFAULT_INITIAL_CAPACITY);
  }

  public IndexedSortedSet(int initialCapacity)
  {
    this(null, initialCapacity);
  }

  public IndexedSortedSet(Collection aColl)
  {
    this(null, Math.max(aColl.size(), DEFAULT_MINIMUM_CAPACITY));
    addColletions(aColl);
  }

  public IndexedSortedSet(Comparator comparator, Collection aColl)
  {
    this(comparator, Math.max(aColl.size(), DEFAULT_MINIMUM_CAPACITY));
    addColletions(aColl);
  }

  private void addColletions(Collection aColl) {
      this.addAll(aColl);
  }
  
  public IndexedSortedSet(Comparator comparator, int initialCapacity)
  {
    this.arrayList  = new ArrayList(initialCapacity);
    this.comparator = comparator;
  }

  @Override
public Comparator comparator()
  {
    return(comparator);
  }

  /**
   *  Re-sort this SortedSet using the specified Comparator.
   */
  @Override
public void sortBy(Comparator c)
  {
    // get the elements into an array and sort them
    Object[] elems = arrayList.toArray();
    Arrays.sort(elems, c);

    // record the new Comparator
    this.comparator = c;

    // clear the underlying ArrayList and add the sorted elements back in
    arrayList.clear();
    for (int i = 0, n = elems.length; i < n; i++)
    {
        arrayList.add(elems[i]);
    }
  }

  @Override
public int size()
  {
    return(arrayList.size());
  }

  @Override
public Iterator iterator()
  {
    return(arrayList.iterator());
  }

  @Override
public Object[] toArray()
  {
    return(arrayList.toArray());
  }

  @Override
public Object[] toArray(Object[] a)
  {
    return(arrayList.toArray(a));
  }

  @Override
public void clear()
  {
    arrayList.clear();
  }

  @Override
public boolean add(Object obj)
  {
    int index = binarySearch(obj);
    if (index < 0) {
      int insertPos = (-1 - index);
      arrayList.add(insertPos, obj);
      return(true);
    }
    else {
      return(false);
    }
  }

  @Override
public boolean remove(Object obj)
  {
    int index = binarySearch(obj);
    if (index >= 0) {
      arrayList.remove(index);
      return(true);
    }
    else {
      return(false);
    }
  }

  @Override
public int addElement(Object obj)
  {
    int index = binarySearch(obj);
    if (index < 0) {
      int insertPos = (-1 - index);
      arrayList.add(insertPos, obj);
      return(insertPos);
    }
    else {
      return(-1);
    }
  }

  @Override
public int removeElement(Object obj)
  {
    int index = binarySearch(obj);
    if (index < 0) {
      int pos = (-1 - index);
      arrayList.remove(pos);
      return(pos);
    }
    else {
      return(-1);
    }
  }

  @Override
public Object removeElementAt(int index)
  {
    Object oldVal = getElementAt(index);
    arrayList.remove(index);
    return(oldVal);
  }

  /**
   *  Returns the first member of the SortedSet.
   */
  @Override
public Object first()
  {
    return(arrayList.get(0));
  }

  /**
   *  Returns the last member of the SortedSet.
   */
  @Override
public Object last()
  {
    return(arrayList.get(arrayList.size() - 1));
  }

  /**
   *  Returns the position of the Object if is a member of the set, or -1
   *  if it is not a member.
   */
  @Override
public int indexOf(Object obj)
  {
    int index = binarySearch(obj);
    return((index < 0) ? -1 : index);
  }

  /**
   *  Returns the position of the Object if is a member of the set, or the
   *  position of the first member of the set that is greater than the object
   *  according to its sorting order.
   */
  public int indexFrom(Object obj)
  {
    int index = binarySearch(obj);
    return((index < 0) ? (-1 - index) : index);
  }

  /**
   *  Return the element at the specified index.
   */
  @Override
public Object getElementAt(int index)
  {
    return(arrayList.get(index));
  }

  @Override
public SortedSet subSet(Object fromElem, Object toElem)
  {
    if ((fromElem == null) || (toElem == null))
    {
        throw(new NullPointerException());
    }
    if (compare(fromElem, toElem) > 0)
    {
        throw(new IllegalArgumentException("fromElem > toElem"));
    }
    return(new SubSet(this, fromElem, toElem));
  }

  @Override
public SortedSet headSet(Object toElem)
  {
    if (toElem == null)
    {
        throw(new NullPointerException());
    }
    return(new SubSet(this, null, toElem));
  }

  @Override
public SortedSet tailSet(Object fromElem)
  {
    if (fromElem == null)
    {
        throw(new NullPointerException());
    }
    return(new SubSet(this, fromElem, null));
  }

  protected int binarySearch(Object obj)
  {
    int lo = 0;
    int hi = arrayList.size() - 1;

    while (hi >= lo) {
      int mid = ((lo + hi) / 2);
      int cmp = compare(obj, arrayList.get(mid));

      if (cmp > 0)
    {
        lo = mid + 1;
    }
    else if (cmp < 0)
    {
        hi = mid - 1;
    }
    else
    {
        return(mid);
    }
    }

    return(-1 - lo);
  }

  protected int compare(Object obj1, Object obj2)
  {
    if (comparator != null)
    {
        return(comparator.compare(obj1, obj2));
    }
    else
    {
        return(((Comparable)obj1).compareTo(obj2));
    }
  }

  private static final int      DEFAULT_INITIAL_CAPACITY = 8;
  private static final int      DEFAULT_MINIMUM_CAPACITY = 4;
  private ArrayList             arrayList;
  private Comparator            comparator;


  ///////////////////////////////////////////////////////////////////////////
  ///
  ///   SubSet
  ///
  ///////////////////////////////////////////////////////////////////////////

  static class SubSet extends AbstractSet implements SortedSet, Serializable
  {
    SubSet(IndexedSortedSet parent, Object loBound, Object hiBound)
    {
      this.parent  = parent;
      this.loBound = loBound;
      this.hiBound = hiBound;
    }

    @Override
    public int size()
    {
      int lo = ((loBound == null) ? 0 : parent.indexFrom(loBound));
      int hi = ((hiBound == null) ? parent.size() : parent.indexFrom(hiBound));
      return(hi - lo);
    }

    @Override
    public Iterator iterator()
    {
      return(new SubsetIterator(parent, loBound, hiBound));
    }

    @Override
    public Comparator comparator()
    {
      return(parent.comparator());
    }

    @Override
    public Object first()
    {
      int pos = ((loBound == null) ? 0 : parent.indexFrom(loBound));

      if (pos >= parent.size())
    {
        throw(new NoSuchElementException());
    }
    else
    {
        return(parent.getElementAt(pos));
    }
    }

    @Override
    public Object last()
    {
      int pos = ((hiBound == null) ? (parent.size() - 1) :
                 (parent.indexFrom(hiBound) - 1));

      if (pos >= parent.size())
    {
        throw(new NoSuchElementException());
    }
    else
    {
        return(parent.getElementAt(pos));
    }
    }

    @Override
    public SortedSet subSet(Object fromElem, Object toElem)
    {
      if ((fromElem == null) || (toElem == null))
    {
        throw(new NullPointerException());
    }

      if ((loBound != null) && (parent.compare(loBound, fromElem) < 0))
    {
        fromElem = loBound;
    }
      if ((hiBound != null) && (parent.compare(hiBound, toElem) > 0))
    {
        toElem = hiBound;
    }

      return(parent.subSet(fromElem, toElem));
    }

    @Override
    public SortedSet headSet(Object toElem)
    {
      if (toElem == null)
    {
        throw(new NullPointerException());
    }

      if ((hiBound != null) && (parent.compare(hiBound, toElem) > 0))
    {
        toElem = hiBound;
    }

      return(parent.headSet(toElem));
    }

    @Override
    public SortedSet tailSet(Object fromElem)
    {
      if (fromElem == null)
    {
        throw(new NullPointerException());
    }

      if ((loBound != null) && (parent.compare(loBound, fromElem) < 0))
    {
        fromElem = loBound;
    }

      return(parent.tailSet(fromElem));
    }

    private IndexedSortedSet    parent;
    private Object              loBound;
    private Object              hiBound;
  }

  ///////////////////////////////////////////////////////////////////////////
  ///
  ///   SubsetIterator
  ///
  ///////////////////////////////////////////////////////////////////////////

  static class SubsetIterator implements Iterator, Serializable
  {
    SubsetIterator(IndexedSortedSet parent, Object loBound, Object hiBound)
    {
      this.parent  = parent;
      this.loBound = loBound;
      this.hiBound = hiBound;
      this.iter    = parent.iterator();
      advanceToFirst();
    }

    @Override
    public boolean hasNext()
    {
      return(nextElem != NO_NEXT_VALUE);
    }

    @Override
    public Object next()
    {
      if (nextElem == NO_NEXT_VALUE)
    {
        throw(new NoSuchElementException());
    }

      Object result = nextElem;
      advanceToNext();
      return(result);
    }

    @Override
    public void remove()
    {
      throw(new UnsupportedOperationException());
    }

    private void advanceToFirst()
    {
      nextElem = NO_NEXT_VALUE;
      
      while (iter.hasNext()) {
        nextElem = iter.next();
        if ((loBound == null) || (parent.compare(nextElem, loBound) >= 0))
        {
            break;
        }
      }
    }

    private void advanceToNext()
    {
      nextElem = NO_NEXT_VALUE;
      
      if (iter.hasNext()) {
        Object elem = iter.next();
        if ((hiBound == null) || (parent.compare(elem, hiBound) < 0))
        {
            nextElem = elem;
        }
      }
    }
    
    private IndexedSortedSet    parent;
    private Object              loBound;
    private Object              hiBound;

    private final static Object NO_NEXT_VALUE = new Object();
    private Iterator            iter;
    private Object              nextElem;
  }

  ///////////////////////////////////////////////////////////////////////////
  ///
  ///    Unit Test
  ///
  ///////////////////////////////////////////////////////////////////////////

  public static void main(String[] args)
  {
    Comparator Reverse = new Comparator() {
      @Override
    public int compare(Object o1, Object o2) {
        return(- ((Comparable)o1).compareTo(o2));
      }
      @Override
    public boolean equals(Object obj) {
        return(false);
      }
      @Override
    public int hashCode() {
          return super.hashCode();
      }
    };

    IndexedSortedSet s = new IndexedSortedSet();
    s.add("this");
    s.add("is");
    s.add("a");
    s.add("zebra");
    s.add("test");
    s.add("of");
    s.add("IndexedSortedSet");
    System.out.println(s);

    System.out.println(s.subSet("a", "z"));
    System.out.println(s.subSet("a", "zebra"));

    System.out.println("indexOf(of) = " + s.indexOf("of"));
    System.out.println("indexOf(foo) = " + s.indexOf("foo"));

    System.out.println(s.headSet("test"));
    System.out.println(s.tailSet("test"));

    System.out.println("Reversing...");
    s.sortBy(Reverse);
    System.out.println(s);
    System.out.println(s.subSet("zebra", "a"));
  }
}
