/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */
package modelobjects.util;

import java.awt.Color;
import java.awt.Component;
import java.beans.PropertyEditorSupport;

import javax.swing.JTextField;
import javax.swing.UIManager;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;

/**
 *  NumberPropertyEditor is a PropertyEditor for numbers that are edited
 *  as Strings.
 */
public class NumberPropertyEditor extends PropertyEditorSupport
  implements DocumentListener
{
  public NumberPropertyEditor()
  {
    this("", Double.class, null, null);
  }

  public NumberPropertyEditor(String propertyName, Class requiredType,
                              Number min, Number max)
  {
    this.propertyName = propertyName;
    this.requiredType = requiredType;
    this.minValue = min;
    this.maxValue = max;

    if ((requiredType != Integer.class) &&
        (requiredType != Long.class) &&
        (requiredType != Float.class) &&
        (requiredType != Double.class)) {
      throw(new IllegalArgumentException
            ("type must be Integer, Long, Float, or Double"));
    }

    textField = new PropSheetJTextField(12);
    textField.getDocument().addDocumentListener(this);
  }

  @Override
public boolean supportsCustomEditor()
  {
    return(true);
  }

  @Override
public Component getCustomEditor()
  {
    return(textField);
  }

  /**
   *  Set the PropertyEditor's "model" value.
   */
  @Override
public void setValue(Object value)
  {
    if (value == null) {
      textField.setText("");
    }
    else if ( value.toString().equals(""))
    {
        textField.setText("");
    }
    else if (requiredType.isInstance(value)) {
      String stringValue = value.toString();
      checkValue(stringValue);
      textField.setText(stringValue);
    }
    else {
      throw(new IllegalArgumentException("value must be a String"));
    }
  }

  /**
   *  Set the PropertyEditor's "view" value.
   */
  @Override
public void setAsText(String stringValue)
  {
    checkValue(stringValue);
    textField.setText(stringValue);
  }

  @Override
public Object getValue()
  {
	  
    String stringValue = getAsText();
    if ( stringValue.equals(""))
    {
        return stringValue;
    }
    if (requiredType == Integer.class)
    {
        return(new Integer(stringValue));
    }
    else if (requiredType == Long.class)
    {
        return(new Long(stringValue));
    }
    else if (requiredType == Float.class)
    {
        return(new Float(stringValue));
    }
    else if (requiredType == Double.class)
    {
        return(new Double(stringValue));
    }
    else
    {
        return(stringValue);
    }
  }

  @Override
public String getAsText()
  {
    String currentViewValue = textField.getText();
    checkValue(currentViewValue);
    return(currentViewValue);
  }

  public String getPropertyName()
  {
    return(propertyName);
  }

  public void setPropertyName(String propertyName)
  {
    this.propertyName = propertyName;
  }

  public Class getRequiredType()
  {
    return(requiredType);
  }

  public void setRequiredType(Class requiredType)
  {
    this.requiredType = requiredType;
  }

  public Number getMinValue()
  {
    return(minValue);
  }

  public void setMinValue(Number minValue)
  {
    this.minValue = minValue;
  }

  public Number getMaxValue()
  {
    return(maxValue);
  }

  public void setMaxValue(Number maxValue)
  {
    this.maxValue = maxValue;
  }

  private void checkValue(String stringValue)
  {
    if (stringValue == null )
    {
        throw(new IllegalArgumentException("You must specify a value for " + propertyName));
    }

    if ( stringValue.equals(""))
    {
        return;
    }
    
    // make sure string parses correctly
    double value;
    if (Integer.class.isAssignableFrom(requiredType) ||
        Long.class.isAssignableFrom(requiredType))
    {
        value = (double)Long.parseLong(stringValue);
    }
    else
    {
        value = Double.parseDouble(stringValue);
    }

    // check range
    if ((minValue != null) && (value < minValue.doubleValue()))
    {
        throw(new IllegalArgumentException
                (propertyName + ": value is less than " + minValue));
    }
    if ((maxValue != null) && (value > maxValue.doubleValue()))
    {
        throw(new IllegalArgumentException
                (propertyName + ": value is greater than " + maxValue));
    }
  }

  ///  DocumentListener interface methods

  @Override
public void changedUpdate(DocumentEvent event) {
    checkDoc();
  }

  @Override
public void insertUpdate(DocumentEvent event) {
    checkDoc();
  }

  @Override
public void removeUpdate(DocumentEvent event) {
    checkDoc();
  }

  private void checkDoc() {
    Color background = textField.getBackground();
    String stringValue = textField.getText();
    firePropertyChange();
    try {
      checkValue(stringValue);
      if (background != GOOD)
    {
        textField.setBackground(GOOD);
    }
    }
    catch (Exception e) {
      if (background != BAD)
    {
        textField.setBackground(BAD);
    }
    }
  }

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  representation
  ///
  /////////////////////////////////////////////////////////////////////////////

  private JTextField    textField;

  private String        propertyName;
  private Class         requiredType;
  private Number        minValue;
  private Number        maxValue;

  private static Color  GOOD            = Color.white;
  private static Color  BAD             = new Color(255, 192, 192);

  private static class PropSheetJTextField extends JTextField
  {
    public PropSheetJTextField()
    {
      super();
    }

    public PropSheetJTextField(int length)
    {
      super(length);
    }

    @Override
    public boolean isFocusTraversable()
    {
      return(isEnabled() && isEditable());
    }

    @Override
    public void setEnabled(boolean enabled)
    {
      if (enabled != isEnabled()) {
        super.setEnabled(enabled);
        setBackground(enabled ?
                      UIManager.getColor("TextField.background") :
                      UIManager.getColor("Panel.background"));
      }
    }
  }
}
