/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */

package modelobjects.util;

/**
 *  This class provides higher level pattern filtering by encapsulating
 *  a set of StringMatchers for patterns to reject and a set for patterns
 *  to accept, either of which can be empty or null.
 */
public class PatternSet implements java.io.Serializable
{
  /**
   *  Construct a PatternSet.  The includePatterns and excludePatterns
   *  may be assigned using the corresponding property setter methods.
   */
  public PatternSet()
  {
    this(null, null);
  }

  /**
   *  Construct a PatternSet with the specifed includePatterns and
   *  excludePatterns.
   */
  public PatternSet(StringMatcher[] includePatterns,
                    StringMatcher[] excludePatterns)
  {
    setIncludePatterns(includePatterns);
    setExcludePatterns(excludePatterns);
  }

  /**
   *  Return whether the specified String is accepted by the PatternSet.
   *  If the String is matched by any excluded pattern, a value of false
   *  is returned.  Otherwise, if the String is matched by any included
   *  pattern, a value of true is returned.  If the String is not matched
   *  by any excluded pattern or included pattern, the method returns false.
   */
  public boolean matches(String s)
  {
    for (int i = 0; i < numExcludePatterns; i++)
    {
        if (excludePatterns[i].matches(s))
        {
            return(false);
        }
    }
    for (int i = 0; i < numIncludePatterns; i++)
    {
        if (includePatterns[i].matches(s))
        {
            return(true);
        }
    }
    return(false);
  }

  /**
   *  Return the included patterns.
   */
  public StringMatcher[] getIncludePatterns()
  {
    return(includePatterns);
  }

  /**
   *  Assign the included patterns.
   */
  public final void setIncludePatterns(StringMatcher[] includePatterns)
  {
    this.includePatterns = includePatterns;
    this.numIncludePatterns =
      ((includePatterns == null) ? 0 : includePatterns.length);
  }

  /**
   *  Return the excluded patterns.
   */
  public StringMatcher[] getExcludePatterns()
  {
    return(excludePatterns);
  }

  /**
   *  Assign the excluded patterns.
   */
  public final void setExcludePatterns(StringMatcher[] excludePatterns)
  {
    this.excludePatterns = excludePatterns;
    this.numExcludePatterns =
      ((excludePatterns == null) ? 0 : excludePatterns.length);
  }

  @Override
public String toString()
  {
    StringBuffer buf = new StringBuffer();
    buf.append("PatternSet[");

    buf.append("include:[");
    int numIncludes = ((includePatterns == null) ? 0 : includePatterns.length);
    for (int i = 0; i < numIncludes; i++) {
      if (i > 0)
    {
        buf.append(", ");
    }
      buf.append(includePatterns[i].toString());
    }
    buf.append("]");

    buf.append(", exclude:[");
    int numExcludes = ((excludePatterns == null) ? 0 : excludePatterns.length);
    for (int i = 0; i < numExcludes; i++) {
      if (i > 0)
    {
        buf.append(", ");
    }
      buf.append(excludePatterns[i].toString());
    }
    buf.append("]");

    buf.append("]");

    return(buf.toString());
  }

  private StringMatcher[] includePatterns;
  private StringMatcher[] excludePatterns;
  private int             numIncludePatterns;
  private int             numExcludePatterns;
}
