/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */

package modelobjects.util;

import java.util.ArrayList;
import java.util.Comparator;

import modelobjects.util.swing.ModelListTableModel;
import modelobjects.util.swing.TableColumnAdapter;

public class SortedArrayList extends ArrayList
{
  /**
   *  Create a SortedArrayList using a default "natural" sort comparator.
   */
  public SortedArrayList()
  {
    this(DefaultComparator.getInstance());
  }

  /**
   *  Create a SortedArrayList using the specified sort comparator.
   */
  public SortedArrayList(Comparator sortComparator)
  {
    super();

    tableColumnAdapter = new WholeObjectTableColumnAdapter(sortComparator);
    TableColumnAdapter[] columnAdapters =
      new TableColumnAdapter[] { tableColumnAdapter };
    tableModel = new ModelListTableModel(columnAdapters, this);
    tableModel.sortByColumn(0, true);
  }

  /**
   *  Override to take advantage of being sorting when possible.
   */
  @Override
public int indexOf(Object obj)
  {
    if (tableModel.getSortComparator() != null)
    {
        return(tableModel.getIndexOf(obj));
    }
    else
    {
        return(super.indexOf(obj));
    }
  }

  /**
   *  Return the current sort comparator.  Whether or not this SortedArrayList
   *  is actually sorted should be determined by the isSorted() method rather
   *  than by this method.
   */
  public Comparator getSortComparator()
  {
    return(sortComparator);
  }

  /**
   *  Set the current sort comparator, forcing a re-sort.
   */
  public void setSortComparator(Comparator sortComparator)
  {
    tableColumnAdapter.setRowComparator(sortComparator);
    tableModel.sortByColumn(0, true);
  }

  /**
   *  Add an element to this SortedArrayList, maintaining sorted order
   *  if currently sorted.
   */
  @Override
public boolean add(Object obj)
  {
    tableModel.insertRowMaintainSort(obj);
    return(true);
  }

  /**
   *  Add an element to this SortedArrayList at the specified index.
   *  This operation might cause the SortedArrayList to become unsorted.
   */
  @Override
public void add(int index, Object obj)
  {
    super.add(index, obj);
    tableModel.rowChanged(index);
  }

  /**
   *  Assign an element to this SortedArrayList at the specified index.
   *  This operation might cause the SortedArrayList to become unsorted.
   */
  @Override
public Object set(int index, Object obj)
  {
    Object prev = super.set(index, obj);
    tableModel.rowChanged(index);
    return(prev);
  }

  /**
   *  Return whether this SortedArrayList is currently sorted.
   */
  public boolean isSorted()
  {
    return(tableModel.getSortComparator() != null);
  }

  /**
   *  Force an ascending or descending sort.
   */
  public void sort(boolean ascending)
  {
    tableModel.sortByColumn(0, ascending);
  }

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  Representation
  ///
  /////////////////////////////////////////////////////////////////////////////

  TableColumnAdapter  tableColumnAdapter;
  ModelListTableModel tableModel;
  Comparator          sortComparator;

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  TableColumnAdapter subclass to use ModelListTableModel for support
  ///
  /////////////////////////////////////////////////////////////////////////////

  static class WholeObjectTableColumnAdapter extends TableColumnAdapter
  {
    public WholeObjectTableColumnAdapter(Comparator comparator)
    {
      super("", false, Object.class, 100, 100, 100);
      setProperties(comparator);
    }
    
    private void setProperties(Comparator comparator) {
        setRowComparator(comparator);
    }
    
    @Override
    public Object getColumnValue(Object rowModel, int rowNumber)
    {
      return(rowModel);
    }
    @Override
    public void setColumnValue(Object rowModel, Object columnValue)
    {
      // ignore
    }
  }

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  Unit test
  ///
  /////////////////////////////////////////////////////////////////////////////

  public static void main(String[] args)
  {
    SortedArrayList list = new SortedArrayList();
    for (int i = 0; i < args.length; i++)
    {
        list.add(args[i]);
    }
    System.out.println("isSorted = " + list.isSorted());
    System.out.println("list(1) = " + list);

    list.sort(false);
    System.out.println("list(2) = " + list);

    list.sort(true);
    System.out.println("list(3) = " + list);

    Comparator reverseComparator =
      ReverseComparator.makeReverseComparator(DefaultComparator.getInstance());
    list.setSortComparator(reverseComparator);
    System.out.println("list(4) = " + list);
  }

}
