/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */

package modelobjects.util;

/**
 *  StringMatcher implements simple string matching with wildcards.
 *
 *  Patterns are specified as Strings containing asterisks that can match
 *  zero or more characters, and other substrings that are matched exactly.
 *  Patterns can have zero or more wildcards, appearing anywhere in the
 *  pattern Strings.
 *
 *  String matching is case sensitive.  If case-insensitive matching is
 *  desired, both patterns and Strings should first be converted either to
 *  upper case or to lower case.
 *
 *  String matching is meant to be efficient, and does not allocate objects
 *  when matching.  Parsed patterns should be saved and reused rather than
 *  parsing the pattern each time a comparison is performed.
 *
 *  @author  Dan Jacobs, ModelObjects Group
 *  @version 1.0, 2000/01/16
 */
public abstract class StringMatcher implements java.io.Serializable
{
  final static char DEFAULT_WILDCARD_CHAR = '*';

  /**
   *  Test if the specified String matches the specified pattern.
   *
   *  This method is intended only for infrequent use, since it does not
   *  save the parsed pattern, but rather parses the pattern each time
   *  it is called.
   *
   *  @param s the String to test against the pattern
   *  @param pattern the pattern to use to see if the String matches
   */
  public static boolean stringMatches(String s, String pattern)
  {
    return(stringMatches(s, pattern, DEFAULT_WILDCARD_CHAR));
  }

  /**
   *  Test if the specified String matches the specified pattern.
   *
   *  This method is intended only for infrequent use, since it does not
   *  save the parsed pattern, but rather parses the pattern each time
   *  it is called.
   *
   *  @param s the String to test against the pattern
   *  @param pattern the pattern to use to see if the String matches
   *  @param wildcardChar the special wildcard character for parsing patterns
   */
  public static boolean stringMatches(String s, String pattern,
                                      char wildcardChar)
  {
    StringMatcher matcher = parsePattern(pattern, wildcardChar);
    return(matcher.matchIndex(s) != -1);
  }


  /**
   *  Return whether the specified String matches the pattern represented
   *  by the StringMatcher.
   *
   *  @param s the String to test against the StringMatcher's pattern
   */
  public final boolean matches(String s)
  {
    return(matchIndex(s) != -1);
  }


  /**
   *  Parse the specified pattern into a StringMatcher.
   *  Patterns can contain any number of asterisks which are interpretted
   *  as wildcard characters, matching zero or more characters.  All other
   *  characters in the pattern are matched exactly.
   *
   *  @param the pattern to parse into a StringMatcher
   */
  public static StringMatcher parsePattern(String pattern)
  {
    return(parsePattern(pattern, false, DEFAULT_WILDCARD_CHAR));
  }

  /**
   *  Parse the specified pattern into a StringMatcher, using the specified
   *  wildcard character.
   *  Patterns can contain any number of wildcards which match zero or more
   *  characters.  All other characters in the pattern are matched exactly.
   *
   *  @param the pattern to parse into a StringMatcher
   *  @param wildcardChar the special wildcard character for parsing patterns
   */
  public static StringMatcher parsePattern(String pattern, char wildcardChar)
  {
    return(parsePattern(pattern, false));
  }

  /**
   *  Parse the specified filename pattern into a StringMatcher.
   *  Patterns can contain any number of wildcards which match zero or more
   *  characters.  All other characters in the pattern are matched exactly.
   *  The StringMatcher returned will perform case-sensitive matching if and
   *  only if the platform's file-system uses case-sensitive file-names.
   *
   *  @param the filename pattern to parse into a StringMatcher
   */
  public static StringMatcher parseFilePattern(String pattern)
  {
    return(parseFilePattern(pattern, DEFAULT_WILDCARD_CHAR));
  }

  /**
   *  Parse the specified filename pattern into a StringMatcher, using the
   *  specified pattern and specified wildcard character.
   *  Patterns can contain any number of asterisks which are interpretted
   *  as wildcard characters, matching zero or more characters.  All other
   *  characters in the pattern are matched exactly.
   *  The StringMatcher returned will perform case-sensitive matching if and
   *  only if the platform's file-system uses case-sensitive file-names.
   *
   *  @param pattern the filename pattern to parse into a StringMatcher
   *  @param wildcardChar the special wildcard character for parsing patterns
   */
  public static StringMatcher parseFilePattern(String pattern,
                                               char wildcardChar)
  {
    return(parsePattern(pattern,
                        getIgnoreFilenameCaseOnPlatform(),
                        wildcardChar));
  }

  /**
   *  Return whether or not the current platform ignores case for filenames.
   *  Currently, this only checks to see if the operating system name starts
   *  with 'Windows'.
   */
  public static boolean getIgnoreFilenameCaseOnPlatform()
  {
    return ObjectLazyHolder.IGNORE_FILENAME_CASE_ON_PLATFORM.booleanValue();
  }

  /**
   *  Parse the specified pattern into a StringMatcher, specifying whether
   *  or not to ignore case when comparing strings for a match.
   *  Patterns can contain any number of asterisks which are interpretted
   *  as wildcard characters, matching zero or more characters.  All other
   *  characters in the pattern are matched exactly.
   *
   *  @param the pattern to parse into a StringMatcher
   *  @param ignoreCase whether or not to ignore case in string comparisons
   */
  public static StringMatcher parsePattern(String pattern, boolean ignoreCase)
  {
    return(parsePattern(pattern, ignoreCase, DEFAULT_WILDCARD_CHAR));
  }

  /**
   *  Parse the specified pattern into a StringMatcher, specifying whether
   *  or not to ignore case when comparing strings for a match, and specifying
   *  the wildcard character.
   *  Patterns can contain any number of asterisks which are interpretted
   *  as wildcard characters, matching zero or more characters.  All other
   *  characters in the pattern are matched exactly.
   *
   *  @param the pattern to parse into a StringMatcher
   *  @param ignoreCase whether or not to ignore case in string comparisons
   *  @param wildcardChar the special wildcard character for parsing patterns
   */
  public static StringMatcher parsePattern(String pattern, boolean ignoreCase,
                                           char wildcardChar)
  {
    int wildCardIndex = pattern.indexOf(wildcardChar);

    if (wildCardIndex == -1) {
      // no wildcards in pattern at all
      return(new FullMatcher(pattern, ignoreCase));
    }

    else if (wildCardIndex == (pattern.length() - 1)) {
      // only wildcard is at end of the pattern
      return(new StartsWithMatcher(pattern.substring(0, wildCardIndex),
                                   ignoreCase));
    }

    else if (wildCardIndex == 0) {
      // there's a wildcard at the beginning of the pattern
      int nextWildCard = pattern.indexOf(wildcardChar, 1);

      if (nextWildCard == -1) {
        // there's only one wildcard, and it's at the beginning
        return(new EndsWithMatcher(pattern.substring(1), ignoreCase));
      }

      else {
        // there's at least one more wildcard somewhere
        StringMatcher m1 =
          new ContainsMatcher(pattern.substring(1, nextWildCard),
                              ignoreCase);

        if (nextWildCard == (pattern.length() - 1)) {
          // the other wildcard at the end of the pattern, so we're done
          return(m1);
        }
        else {
          // the next wildcard is not at the end, so parse the rest of the
          // pattern and make a CompoundMatcher from the two pieces
          StringMatcher m2 =
            parsePattern(pattern.substring(nextWildCard),
                         ignoreCase, wildcardChar);
          return(new CompoundMatcher(m1, m2));
        }
      }
    }

    else {
      // the wildcard is not at beginning or end, so make a compound of a
      // StartsWithMatcher and whatever is parsed from the rest
      StringMatcher m1 =
        new StartsWithMatcher(pattern.substring(0, wildCardIndex),
                              ignoreCase);
      StringMatcher m2 = parsePattern(pattern.substring(wildCardIndex),
                                      ignoreCase, wildcardChar);
      return(new CompoundMatcher(m1, m2));
    }
  }


  /**
   *  Test class StringMatcher as a console application by printing out
   *  the object representation of the parsed pattern, and by testing the
   *  parsed pattern on a sample string if one is provided.
   *
   *  The arguments are <pattern> [ <sampleString> ]
   */
  public static void main(String[] args)
  {
    if (args.length == 0) {
      System.out.println("Args:  <pattern> [ <sampleString> ]");
    }
    else {
      try {
        StringMatcher matcher = parsePattern(args[0], true, '*');
        System.out.println(matcher);

        if (args.length > 1) {
          System.out.println("Match index on \"" + args[1] + "\" = " +
                             matcher.matchIndex(args[1]));
        }
      }
      catch (Exception e) {
        e.printStackTrace();
      }
    }
  }


  /**
   *  Returns the index in the String where the StringMatcher *finished*
   *  matching, starting at the beginning of the String, or -1 if the
   *  String was not matched by the StringMatcher.
   *
   *  The index of where the match finished is used in order to chain
   *  multiple substring matches together, separated by asterisks.
   *
   *  @param the String to test for a match with the StringMatcher
   */
  public abstract int matchIndex(String s);

  /**
   *  Return the index in the String where the StringMatcher *finished*
   *  matching, starting at the specified starting index, or -1 if the
   *  String starting at that index was not matched by the StringMatcher.
   *
   *  The index of where the match finished is used in order to chain
   *  multiple substring matches together, separated by asterisks.
   *
   *  @param the String to test for a match with the StringMatcher
   *  @param startIndex where in the String to start looking for a match
   */
  public abstract int matchIndex(String s, int startIndex);

  private static class ObjectLazyHolder {
      private static final Boolean IGNORE_FILENAME_CASE_ON_PLATFORM = (System.getProperty("os.name").startsWith("Windows"));
  }


  //////////////////////////////////////////////////////////////////////////
  ///
  ///   INNER CLASSES
  ///
  //////////////////////////////////////////////////////////////////////////

  /**
   *  This kind of StringMatcher is used to compose other StringMatchers
   *  into a sequence of simpler matchers.
   */
  private static final class CompoundMatcher extends StringMatcher
  {
    /**
     *  Construct a CompoundMatcher from two other StringMatchers.
     */
    CompoundMatcher(StringMatcher m1, StringMatcher m2)
    {
      this.m1 = m1;
      this.m2 = m2;
    }

    /**
     *  Returns the index in the String where the StringMatcher finished
     *  matching, starting at the beginning of the String, or -1 if the
     *  String was not matched by the StringMatcher.
     *
     *  If the first of the two sub-StringMatchers matches some of the
     *  String, the second one tries to match from where the first one
     *  left off.  Otherwise -1 is returned.
     *
     *  @param the String to test for a match with the StringMatcher
     */
    @Override
    public int matchIndex(String s)
    {
      int index1 = m1.matchIndex(s);
      return((index1 == -1) ? -1 : m2.matchIndex(s, index1));
    }

    /**
     *  Return the index in the String where the StringMatcher finished
     *  matching, starting at the specified starting index, or -1 if the
     *  String starting at that index was not matched by the StringMatcher.
     *
     *  If the first of the two sub-StringMatchers matches some of the
     *  String, the second one tries to match from where the first one
     *  left off.  Otherwise -1 is returned.
     *
     *  @param the String to test for a match with the StringMatcher
     *  @param startIndex where in the String to start looking for a match
     */
    @Override
    public int matchIndex(String s, int startIndex)
    {
      int index1 = m1.matchIndex(s, startIndex);
      return((index1 == -1) ? -1 : m2.matchIndex(s, index1));
    }

    /**
     *  Return a human-readable rendering of the StringMatcher.
     *  This method shows that it's a CompoundMatcher, and delegates the
     *  rest to the two sub-StringMatchers.
     */
    @Override
    public String toString()
    {
      return(m1 + " & " + m2);
    }

    private StringMatcher m1, m2;
  }


  /**
   *  This kind of StringMatcher is used to check that a String matches
   *  some other String exactly.  It should be used only when the pattern
   *  has no wildcard characters.
   */
  private static final class FullMatcher extends StringMatcher
  {
    /**
     *  Construct a FullMatcher from the String that must be matched.
     */
    FullMatcher(String literal, boolean ignoreCase)
    {
      this.literal = literal;
      this.literalLength = literal.length();
      this.ignoreCase = ignoreCase;
    }

    /**
     *  Returns the index in the String where the StringMatcher finished
     *  matching, starting at the beginning of the String, or -1 if the
     *  String was not matched by the StringMatcher.
     *
     *  This method matches only if the tested String is equivalent to the
     *  FullMatcher's reference literal.
     *
     *  @param the String to test for a match with the StringMatcher
     */
    @Override
    public int matchIndex(String s)
    {
      boolean matches = (ignoreCase ?
                         s.equalsIgnoreCase(literal) :
                         s.equals(literal));

      return(matches ? literalLength : -1);
    }

    /**
     *  Return the index in the String where the StringMatcher finished
     *  matching, starting at the specified starting index, or -1 if the
     *  String starting at that index was not matched by the StringMatcher.
     *
     *  @param the String to test for a match with the StringMatcher
     *  @param startIndex where in the String to start looking for a match
     */
    @Override
    public int matchIndex(String s, int startIndex)
    {
      return((startIndex == 0) ? matchIndex(s) : -1);
    }

    /**
     *  Return a human-readable rendering of the StringMatcher.
     *  This method shows that it's a FullMatcher and shows the string
     *  that must be matched.
     */
    @Override
    public String toString()
    {
      return("equals(\"" + literal + "\")");
    }

    private String  literal;
    private int     literalLength;
    private boolean ignoreCase;
  }


  /**
   *  This kind of StringMatcher is used to check for a literal fragment
   *  at the beginning of a String.
   */
  private static final class StartsWithMatcher extends StringMatcher
  {
    StartsWithMatcher(String prefix, boolean ignoreCase)
    {
      this.prefix = prefix;
      this.prefixLength = prefix.length();
      this.ignoreCase = ignoreCase;
    }

    /**
     *  Returns the index in the String where the StringMatcher finished
     *  matching, starting at the beginning of the String, or -1 if the
     *  String was not matched by the StringMatcher.
     *
     *  @param the String to test for a match with the StringMatcher
     */
    @Override
    public int matchIndex(String s)
    {
      return(matchIndex(s, 0));
    }

    /**
     *  Return the index in the String where the StringMatcher finished
     *  matching, starting at the specified starting index, or -1 if the
     *  String starting at that index was not matched by the StringMatcher.
     *
     *  @param the String to test for a match with the StringMatcher
     *  @param startIndex where in the String to start looking for a match
     */
    @Override
    public int matchIndex(String s, int startIndex)
    {
      boolean match =
        s.regionMatches(ignoreCase, startIndex, prefix, 0, prefixLength);

      return(match ? prefixLength + startIndex : -1);
    }

    /**
     *  Return a human-readable rendering of the StringMatcher.
     *  This method shows that it's a StartsWithMatcher and shows the string
     *  that must match the beginning of the tested string.
     */
    @Override
    public String toString()
    {
      return("startsWith(\"" + prefix + "\")");
    }

    private String  prefix;
    private int     prefixLength;
    private boolean ignoreCase;
  }


  /**
   *  This kind of StringMatcher is used to check for a literal fragment
   *  at the end of a String.
   */
  private static final class EndsWithMatcher extends StringMatcher
  {
    EndsWithMatcher(String suffix, boolean ignoreCase)
    {
      this.suffix = suffix;
      this.suffixLength = suffix.length();
      this.ignoreCase = ignoreCase;
    }

    /**
     *  Returns the index in the String where the StringMatcher finished
     *  matching, starting at the beginning of the String, or -1 if the
     *  String was not matched by the StringMatcher.
     *
     *  @param the String to test for a match with the StringMatcher
     */
    @Override
    public int matchIndex(String s)
    {
      boolean match = s.regionMatches(ignoreCase, s.length() - suffixLength,
                                      suffix, 0, suffixLength);
      return(match ? s.length() : -1);
    }

    /**
     *  Return the index in the String where the StringMatcher finished
     *  matching, starting at the specified starting index, or -1 if the
     *  String starting at that index was not matched by the StringMatcher.
     *
     *  @param the String to test for a match with the StringMatcher
     *  @param startIndex where in the String to start looking for a match
     */
    @Override
    public int matchIndex(String s, int startIndex)
    {
      if ((s.length() - startIndex) >= suffixLength)
    {
        return(matchIndex(s));
    }
    else
    {
        return(-1);
    }
    }

    /**
     *  Return a human-readable rendering of the StringMatcher.
     *  This method shows that it's an EndsWithMatcher and shows the string
     *  that must match the end of the tested string.
     */
    @Override
    public String toString()
    {
      return("endsWith(\"" + suffix + "\")");
    }

    private String  suffix;
    private int     suffixLength;
    private boolean ignoreCase;
  }


  /**
   *  This kind of StringMatcher is used to check for a literal fragment
   *  anywhere in a String.  It should be used when there are multiple
   *  wildcards in a pattern, such as in '*foo*' or 'foo*bar*baz', etc.
   */
  private static final class ContainsMatcher extends StringMatcher
  {
    ContainsMatcher(String fragment, boolean ignoreCase)
    {
      this.fragment = fragment;
      this.fragmentLength = fragment.length();
      this.ignoreCase = ignoreCase;
    }

    /**
     *  Returns the index in the String where the StringMatcher finished
     *  matching, starting at the beginning of the String, or -1 if the
     *  String was not matched by the StringMatcher.
     *
     *  @param the String to test for a match with the StringMatcher
     */
    @Override
    public int matchIndex(String s)
    {
      return(matchIndex(s, 0));
    }

    /**
     *  Return the index in the String where the StringMatcher finished
     *  matching, starting at the specified starting index, or -1 if the
     *  String starting at that index was not matched by the StringMatcher.
     *
     *  @param the String to test for a match with the StringMatcher
     *  @param startIndex where in the String to start looking for a match
     */
    @Override
    public int matchIndex(String s, int startIndex)
    {
      int index = -1;
      if (ignoreCase) {
        for (int i = startIndex, n = s.length() - fragmentLength; i <= n; i++) {
          if (s.regionMatches(true, i, fragment, 0, fragmentLength)) {
            index = i;
            break;
          }
        }
      }
      else {
        index = s.indexOf(fragment, startIndex);
      }
      return((index == -1) ? -1 : index + fragmentLength);
    }

    /**
     *  Return a human-readable rendering of the StringMatcher.
     *  This method shows that it's an ContainsMatcher and shows the string
     *  that must match somewhere inside of the tested string.
     */
    @Override
    public String toString()
    {
      return("contains(\"" + fragment + "\")");
    }

    private String  fragment;
    private int     fragmentLength;
    private boolean ignoreCase;
  }
}
