/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */

package modelobjects.util;

import java.lang.ref.ReferenceQueue;
import java.lang.ref.WeakReference;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 *  WeakValueMap implements a hash-based mapping that uses strong references
 *  to keys and weak references to values.  If all non-weak references to a
 *  value are dropped, the entry or entries containing that value will be
 *  removed from the mapping.
 *
 *  Note that this implementation stores WeakReferences as values in an
 *  otherwise normal HashMap, but unwraps the WeakReferences in the return
 *  values of the get, put, and remove methods.  The containsValue method
 *  also provides expected behavior, comparing the unwrapped values in the
 *  map to the provided value.  Other methods, such as values and entrySet
 *  do not attempt to conceal the WeakReferences from the caller.
 */
public class WeakValueMap extends HashMap
{
  /**
   *  Construct a WeakValueMap
   */
  public WeakValueMap()
  {
    super();
    refQueue = new ReferenceQueue();
  }

  /**
   *  Associates the specified value with the specified key in this map,
   *  and returns the previous value associated with the key, if any.
   */
  @Override
public Object put(Object key, Object value)
  {
    processRefQueue();
    WeakValue weakVal = new WeakValue(value, refQueue, key);
    WeakValue prev = (WeakValue)super.put(key, weakVal);
    return((prev == null) ? null : prev.get());
  }

  /**
   *  Returns the value associated with the key in this map, or null if
   *  no entry exists for the specified key.
   */
  @Override
public Object get(Object key)
  {
    //!! processRefQueue();
    WeakValue val = (WeakValue)super.get(key);
    return((val == null) ? null : val.get());
  }

  /**
   *  Removes the association for the specified key from this map,
   *  and returns the previous value associated with the key, if any.
   */
  @Override
public Object remove(Object key)
  {
    processRefQueue();
    WeakValue prev = (WeakValue)super.remove(key);
    return((prev == null) ? null : prev.get());
  }

  /**
   *  Returns true if this map maps one or more keys to the specified value.
   *  Note that even though WeakValueMap stores WeakReferences as values,
   *  this method compares the specified value to the unwrapped references.
   */
  @Override
public boolean containsValue(Object obj)
  {
    //!! processRefQueue();
    for (Iterator vals = values().iterator(); vals.hasNext(); ) {
      WeakValue wv = (WeakValue)vals.next();
      Object val = wv.get();
      if ((val == obj) || ((val != null) && val.equals(obj)))
    {
        return(true);
    }
    }
    return(false);
  }

  /**
   *  Return a human-readable rendering of this map.
   */
  @Override
public String toString()
  {
    //!! processRefQueue();

    StringBuffer buf = new StringBuffer();
    buf.append("{");
    boolean first = true;

    for (Iterator entries = entrySet().iterator(); entries.hasNext(); ) {
      Map.Entry entry = (Map.Entry)entries.next();
      Object key = entry.getKey();
      Object val = ((WeakValue)entry.getValue()).get();
      if (first)
    {
        first = false;
    }
    else
    {
        buf.append(", ");
    }
      buf.append(key + "=" + val);
    }
    buf.append("}");

    return(buf.toString());
  }

  /**
   *  Remove all entries for any WeakValue that has been cleared.
   */
  public void clean()
  {
    processRefQueue();
  }

  /**
   *  Process the WeakReference ReferenceQueue to remove entries for any
   *  WeakValue that has been cleared.
   */
  private void processRefQueue()
  {
    WeakValue wv;
    while ((wv = (WeakValue)refQueue.poll()) != null) {
      this.remove(wv.key);
      wv.key = null;
    }
  }

  ///////////////////////////////////////////////////////////////////////////
  ///
  ///  inner class WeakValue
  ///
  ///////////////////////////////////////////////////////////////////////////

  private static class WeakValue extends WeakReference
  {
    WeakValue(Object value, ReferenceQueue refQueue, Object key)
    {
      super(value, refQueue);
      this.key = key;
    }
    Object key;
  }    

  ///////////////////////////////////////////////////////////////////////////
  ///
  ///  private representation
  ///
  ///////////////////////////////////////////////////////////////////////////

  private ReferenceQueue refQueue;
}
