/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */

package modelobjects.util;

import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;


/**
 *  XmlWritable defines common support for graphs of objects that want to write
 *  themselves out in XML format.
 */
public abstract class XmlWritable
{
  /**
   *  Construct an XmlWritable with the specified element name.
   */
  public XmlWritable(String elementName)
  {
    this(elementName, true);
  }

  /**
   *  Construct an XmlWritable with the specified element name, and specify
   *  whether or not the denoted XML element allows indenting.
   */
  public XmlWritable(String elementName, boolean allowsIndent)
  {
    this.elementName  = elementName;
    this.allowsIndent = allowsIndent;

    this.attributes   = new HashMap();
  }

  /**
   *  Return the element name for this XmlWritable.
   */
  public String getElementName()
  {
    return(elementName);
  }

  public void setElementName(String elementName)
  {
    this.elementName = elementName;
  }

  /**
   *  Return a Map object mapping attribute-name to attribute-value.
   *  Attribute-names are expected to be non-null Strings, and attribute
   *  values must be non-null values with appropriate toString methods.
   */
  public Map getAttributes()
  {
    return(attributes);
  }

  /**
   *  Return the value of the specified attribute, or null if it has no
   *  assigned value.
   */
  public Object getAttribute(String attributeName)
  {
    return(attributes.get(attributeName));
  }

  /**
   *  Assign the Map object mapping attribute-name to attribute-value.
   *  Attribute-names are expected to be non-null Strings, and attribute
   *  values must be non-null values with appropriate toString methods.
   */
  public void setAttributes(Map attributes)
  {
    this.attributes = attributes;
  }

  /**
   *  Assign an attribute name and value pair.
   */
  public void setAttribute(String attributeName, Object attributeValue)
  {
    attributes.put(attributeName, attributeValue);
  }

  /**
   *  Return this XmlWritable's list of namespace attributes, or null
   *  if there aren't any.
   */
  public List getNamespaceAttributes()
  {
    return(namespaceAttributes);
  }

  /**
   *  Assign a namespace attribute name and value pair.
   */
  public void addNamespaceAttribute(String attributeName, String attributeValue)
  {
    if (namespaceAttributes == null)
    {
        namespaceAttributes = new ArrayList();
    }

    if (attributeName.equals("xmlns") || attributeName.startsWith("xmlns:")) {
      NamespaceAttribute namespaceAttr =
        new NamespaceAttribute(attributeName, attributeValue);
      namespaceAttributes.add(namespaceAttr);
    }
    else {
      throw(new IllegalArgumentException("namespace attribute names must " +
                                         "start with 'xmlns'"));
    }
  }

  /**
   *  Return the PCDATA content for this XmlWritable's element, or null
   *  if it does not have any.
   */
  public String getPCDataContent()
  {
    return(pcdataContent);
  }

  /**
   *  Assign the PCDATA content for this XmlWritable's element, or null
   *  if it does not have any.
   */
  public void setPCDataContent(String pcdataContent)
  {
    this.pcdataContent = pcdataContent;
  }

  /**
   *  Return a list of children XmlWritable objects that represent child
   *  elements of this XmlWritable.
   */
  public abstract List getChildren();

  /**
   *  Return whether or not the element for this XmlWritable should be
   *  indented.
   */
  public boolean getAllowsIndent()
  {
    return(allowsIndent);
  }

  /**
   *  Return whether or not the element for this XmlWritable should be
   *  indented.
   */
  public void setAllowsIndent(boolean allowsIndent)
  {
    this.allowsIndent = allowsIndent;
  }

  /**
   *  Make a CDATA tag to escape the entire specified string.
   */
  public static String makeCDataTag(String s)
  {
    return("<![CDATA[" + s + "]]>");
  }

  /**
   *  Write out this XmlWritable as XML.
   */
  public void writeAsXML(PrintWriter out)
  {
    writeAsXML(out, "  ", 0);
    out.println();
    out.flush();
  }

  /**
   *  Write out this XmlWritable as XML with the specified indent string.
   */
  public void writeAsXML(PrintWriter out, String indentString)
  {
    writeAsXML(out, indentString, 0);
  }

  /**
   *  Write out this XmlWritable as XML at the specified indent string
   *  and indent level.
   */
  private void writeAsXML(PrintWriter out, String indentString, int indentLevel)
  {
    List nsAttrs    = getNamespaceAttributes();
    Map attributes  = getAttributes();
    List children   = getChildren();
    String elemName = getElementName();

    boolean hasChildren =
      ((children != null) && (!children.isEmpty()));
    boolean hasPCData =
      ((pcdataContent != null) && (pcdataContent.length() > 0));

    if (hasChildren)
    {
        out.println();
    }

    int lineLength = startNewLine(out, indentString, indentLevel);
    lineLength += elemName.length() + 1;

    out.print(TAG_OPEN);
    out.print(elemName);

    if (nsAttrs != null) {
      for (int i = 0, n = nsAttrs.size(); i < n; i++) {
        NamespaceAttribute nsa = (NamespaceAttribute)nsAttrs.get(i);
        String attrName  = nsa.getName();
        String attrValue = nsa.getValue();
        lineLength = writeAttr(out, indentString, indentLevel,
                               attrName, attrValue, lineLength);
      }
    }

    if (attributes != null) {
      Iterator attrs = attributes.entrySet().iterator();
      while (attrs.hasNext()) {
        Map.Entry attr = (Map.Entry)attrs.next();
        String attrName  = attr.getKey().toString();
        String attrValue = attr.getValue().toString();
        lineLength = writeAttr(out, indentString, indentLevel,
                               attrName, attrValue, lineLength);
      }
    }

    if (!hasChildren && !hasPCData) {
      // empty element, close with a forward slash and closing angle bracket
      out.print(FORWARD_SLASH);
      out.print(TAG_CLOSE);
    }
    else {
      // print the closing angle bracket for the open tag
      out.print(TAG_CLOSE);

      // print out the children or PCDATA
      if (hasChildren) {
        Iterator childIterator = children.iterator();
        while (childIterator.hasNext()) {
          XmlWritable child = (XmlWritable)childIterator.next();
          child.writeAsXML(out, indentString, indentLevel + 1);
        }
      }
      else if (hasPCData) {
        out.print(pcdataContent);
      }

      // print the close tag
      if (allowsIndent && hasChildren)
    {
        startNewLine(out, indentString, indentLevel);
    }
      out.print(TAG_OPEN);
      out.print(FORWARD_SLASH);
      out.print(elemName);
      out.print(TAG_CLOSE);
    }
  }

  private int writeAttr(PrintWriter out, String indentString, int indentLevel,
                        String attrName, String attrValue, int lineLength)
  {
    if ((lineLength + attrName.length() + attrValue.length()) >= 75) {
      startNewLine(out, indentString, indentLevel + 3);
      lineLength = (indentString.length() * (indentLevel + 3));
    }
    else {
      out.print(SPACE);
    }

    lineLength += (attrName.length() + attrValue.length() + 4);

    out.print(attrName);
    out.print(EQUALS);
    out.print(DOUBLE_QUOTE);
    out.print(attrValue);
    out.print(DOUBLE_QUOTE);

    return(lineLength);
  }

  protected int startNewLine(PrintWriter out, String indentString,
                              int indentLevel)
  {
    out.println();
    if (allowsIndent) {
      for (int i = 0; i < indentLevel; i++)
    {
        out.print(indentString);
    }
      return(indentString.length() * indentLevel);
    }
    else {
      return(0);
    }
  }

  public static XmlWritable makePCDataElement(String xmlTag, String pcData,
                                              boolean useCDataWrapper)
  {
    XmlWritable result = new XmlWritable(xmlTag) {
      @Override
    public List getChildren() { return(Collections.EMPTY_LIST); }
    };

    String content = (useCDataWrapper ? makeCDataTag(pcData) : pcData);
    result.setPCDataContent(content);

    return(result);
  }

  public static XmlWritable makeEmptyContentElement(String xmlTag)
  {
    XmlWritable result = new XmlWritable(xmlTag) {
      @Override
    public List getChildren() { return(Collections.EMPTY_LIST); }
    };
    return(result);
  }

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  private representation
  ///
  /////////////////////////////////////////////////////////////////////////////

  private final static char     TAG_OPEN        = '<';
  private final static char     TAG_CLOSE       = '>';
  private final static char     FORWARD_SLASH   = '/';
  private final static char     DOUBLE_QUOTE    = '"';
  private final static char     SPACE           = ' ';
  private final static char     EQUALS          = '=';

  private String                elementName;
  private String                pcdataContent;
  private Map                   attributes;
  private List                  namespaceAttributes;
  private boolean               allowsIndent;


  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  Inner class NamespaceAttribute
  ///
  /////////////////////////////////////////////////////////////////////////////

  public static class NamespaceAttribute
  {
    public NamespaceAttribute(String name, String value)
    {
      this.name  = name;
      this.value = value;
    }

    public String getName()
    {
      return(name);
    }

    public String getValue()
    {
      return(value);
    }

    private String name;
    private String value;
  }
}
