/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */
package modelobjects.util.swing;

import java.awt.Color;

import javax.swing.DefaultCellEditor;
import javax.swing.JComboBox;
import javax.swing.JTextField;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;

/**
 *  BasicTableCellEditor extends DefaultCellEditor with the ability to give
 *  live feedback about input validation.
 */
public class BasicTableCellEditor extends DefaultCellEditor
{
  /**
   *  Construct a BasicTableCellEditor to edit values of the specified type.
   *  Primitive types cannot be used here.  Instead, the corresponding
   *  wrapper classes should be used.  The supported types are Boolean,
   *  String, Byte, Short, Integer, Long, Float, and Double.
   *
   *  @param cellType the type of cell values to be edited
   */
  public static BasicTableCellEditor makeCellEditor(Class cellType)
  {
    if (cellType == Boolean.class)
    {
        return(makeBooleanEditor());
    }
    else if (cellType == String.class)
    {
        return(makeStringEditor());
    }
    else if (cellType == Integer.class)
    {
        return(makeNumericEditor(new IntegerConverter()));
    }
    else if (cellType == Long.class)
    {
        return(makeNumericEditor(new LongConverter()));
    }
    else if (cellType == Float.class)
    {
        return(makeNumericEditor(new FloatConverter()));
    }
    else if (cellType == Double.class)
    {
        return(makeNumericEditor(new DoubleConverter()));
    }
    else if (cellType == Byte.class)
    {
        return(makeNumericEditor(new ByteConverter()));
    }
    else if (cellType == Short.class)
    {
        return(makeNumericEditor(new ShortConverter()));
    }
    else
    {
        return(makeStringEditor());
    }
  }

  public static BasicTableCellEditor makeBooleanEditor()
  {
    Object[] values = { Boolean.FALSE, Boolean.TRUE };
    JComboBox comboBox = new JComboBox(values);
    comboBox.setEditable(false);
    StringConverter converter = new NoopConverter();
    BasicTableCellEditor editor =
      new BasicTableCellEditor(comboBox, converter);
    return(editor);
  }

  public static BasicTableCellEditor makeStringEditor()
  {
    StringConverter converter = new NoopConverter();
    JTextField textField = new JTextField();
    BasicTableCellEditor editor =
      new BasicTableCellEditor(textField, converter);
    return(editor);
  }

  public static BasicTableCellEditor makeJavaIdentifierEditor()
  {
    StringConverter converter = new JavaIdentifierConverter();
    JTextField textField = new JTextField();
    BasicTableCellEditor editor =
      new BasicTableCellEditor(textField, converter);
    return(editor);
  }

  public static
    BasicTableCellEditor makeNumericEditor(StringConverter converter)
  {
    JTextField textField = new JTextField();
    BasicTableCellEditor editor =
      new BasicTableCellEditor(textField, converter);
    return(editor);
  }

  /**
   *  Returns the value contained in the editor.
   *
   *  @throws RuntimeException if the view value in the editor can be
   *  converted to an appropriate model value.
   */
  @Override
public Object getCellEditorValue() 
  {
    if (textField != null) {
      return(converter.convert(textField.getText()));
    }
    else if (comboBox != null) {
      return(comboBox.getSelectedItem());
    }
    else {
      return(null);
    }
  }

  public void setAlignment(int alignment)
  {
    if (textField != null)
    {
        textField.setHorizontalAlignment(alignment);
    }
  }

  /**
   *  Check to see if it's ok to stop editing, and do so if it's ok.
   *  If the editor cannot convert the view-value to a model-value using
   *  its converter, this method returns false.
   */
  @Override
public boolean stopCellEditing()
  {
    try {
      getCellEditorValue();
      return(super.stopCellEditing());
    }
    catch (RuntimeException e) {
      //!! System.out.println("stopCellEditing: " + e);
      if (textField != null)
    {
        textField.setBackground(BAD);
    }
      return(false);
    }
  }

  private BasicTableCellEditor(JComboBox comboBox, StringConverter converter)
  {
    super(comboBox);
    this.comboBox  = comboBox;
    this.converter = converter;
  }

  private BasicTableCellEditor(JTextField textField, StringConverter converter)
  {
    super(textField);
    this.textField = textField;
    this.converter = converter;
    FieldValidator validator = new FieldValidator(textField, converter);
    textField.getDocument().addDocumentListener(validator);
  }

  /**
   *  Return the color used to highlight invalid input in text-field editors.
   */
  public static Color getInvalidInputBackgroundColor()
  {
    return(BAD);
  }

  /**
   *  Assign the color used to highlight invalid input in text-field editors.
   */
  public static void setInvalidInputBackgroundColor(Color color)
  {
    BAD = color;
  }

  private JComboBox             comboBox;
  private JTextField            textField;
  private StringConverter       converter;

  private static final Color          GOOD = Color.white;
  private static volatile Color       BAD  = new Color(255, 192, 192);


  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  Inner classes
  ///
  /////////////////////////////////////////////////////////////////////////////

  public static abstract class StringConverter {
    public abstract Object convert(String value);
  }

  public static class NoopConverter extends StringConverter { 
    @Override
    public Object convert(String value) { return(value); }
  }

  public static class BooleanConverter extends StringConverter {
    @Override
    public Object convert(String value) { return(Boolean.valueOf(value)); }
  }

  public static class IntegerConverter extends StringConverter {
    @Override
    public Object convert(String value) { return(Integer.decode(value)); }
  }

  public static class LongConverter extends StringConverter {
    @Override
    public Object convert(String value) { return(Long.decode(value)); }
  }

  public static class ShortConverter extends StringConverter {
    @Override
    public Object convert(String value) { return(Short.decode(value)); }
  }

  public static class ByteConverter extends StringConverter {
    @Override
    public Object convert(String value) { return(Byte.decode(value)); }
  }

  public static class FloatConverter extends StringConverter {
    @Override
    public Object convert(String value) { return(Float.valueOf(value)); }
  }

  public static class DoubleConverter extends StringConverter {
    @Override
    public Object convert(String value) { return(Double.valueOf(value)); }
  }

  public static class JavaIdentifierConverter extends StringConverter { 
    @Override
    public Object convert(String value) {
      int length = ((value == null) ? 0 : value.length());
      if (length == 0)
    {
        throw(new IllegalArgumentException("null name"));
    }
      for (int i = 0; i < length; i++) {
        char c = value.charAt(i);
        if (i == 0) {
          if (!Character.isJavaIdentifierStart(c))
        {
            throw(new IllegalArgumentException("bad first character in name"));
        }
        }
        else {
          if (!Character.isJavaIdentifierPart(c))
        {
            throw(new IllegalArgumentException("bad character in name"));
        }
        }
      }
      return(value);
    }
  }

  static class FieldValidator implements DocumentListener
  {
    FieldValidator(JTextField textField, StringConverter converter)
    {
      this.textField = textField;
      this.converter = converter;
    }

    void checkDoc() {
      Color background = textField.getBackground();
      String text = textField.getText();
      try {
        converter.convert(text);
        if (background != GOOD)
        {
            textField.setBackground(GOOD);
        }
      }
      catch (Exception e) {
        if (background != BAD)
        {
            textField.setBackground(BAD);
        }
      }
    }

    @Override
    public void changedUpdate(DocumentEvent event) {
      checkDoc();
    }
    @Override
    public void insertUpdate(DocumentEvent event) {
      checkDoc();
    }
    @Override
    public void removeUpdate(DocumentEvent event) {
      checkDoc();
    }

    private JTextField      textField;
    private StringConverter converter;
  }
}
