/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */
package modelobjects.util.swing;

import java.awt.BorderLayout;
import java.awt.Component;
import java.io.File;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Date;

import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.UIManager;
import javax.swing.table.DefaultTableCellRenderer;

/**
 *  DirectoryFileTable displays the contents of a directory.
 */
public class DirectoryFileTable extends ModelTable
{
  public DirectoryFileTable()
  {
    super(makeTableColumns());
  }

  static TableColumnAdapter[] makeTableColumns()
  {
    return(new TableColumnAdapter[] {
      new FileNameColumnAdapter(),
      new FileSizeColumnAdapter(),
      new FileDateColumnAdapter(),
    });
  }

  public void setDirectory(File directory)
  {
    ArrayList contents = new ArrayList();

    if (directory != null) {
      File[] files = directory.listFiles();
      int numFiles = ((files == null) ? 0 : files.length);
      for (int i = 0; i < numFiles; i++) {
        contents.add(files[i]);
      }
    }

    tableModel.setContents(contents);
    tableModel.sortByColumn(0);
    getTableHeader().repaint(); // must repaint header after programmatic sort
  }

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  unit test
  ///
  /////////////////////////////////////////////////////////////////////////////

  public static void main(String[] args)
  {
    String dirName = ((args.length == 0) ? "." : args[0]);
    DirectoryFileTable table = new DirectoryFileTable();
    table.setDirectory(new File(dirName));

    JFrame window = new JFrame("DirectoryFileTable Test");
    window.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    window.setBounds(300, 200, 600, 400);

    window.getContentPane().add(new JScrollPane(table), BorderLayout.CENTER);
    window.setVisible(true);
  }


  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  representation
  ///
  /////////////////////////////////////////////////////////////////////////////

  ModelListTableModel           tableModel;


  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  inner classes
  ///
  /////////////////////////////////////////////////////////////////////////////

  static class FileNameColumnAdapter extends TableColumnAdapter
  {
    FileNameColumnAdapter() {
      super("Name", false, String.class, 40, 150, 1000);
      setProperties();
    }
    
    private void setProperties() {
        setCellRenderer(new FileNameCellRenderer());
        setRowComparator(new FileNameComparator());
    }
    
    @Override
    public Object getColumnValue(Object rowModel, int rowNumber) {
      return(((File)rowModel).getName());
    }
    @Override
    public void setColumnValue(Object rowModel, Object columnValue) {
      // ignore
    }
  }

  static class FileSizeColumnAdapter extends TableColumnAdapter
  {
    FileSizeColumnAdapter() {
      super("Size", false, Long.class, 20, 60, 400,
            new DefaultTableCellRenderer(), null);
      DefaultTableCellRenderer renderer =
        (DefaultTableCellRenderer)getCellRenderer();
      renderer.setHorizontalAlignment(JLabel.RIGHT);
    }
    @Override
    public Object getColumnValue(Object rowModel, int rowNumber) {
      File file = (File)rowModel;
      return(new Long(file.length()));
    }
    @Override
    public void setColumnValue(Object rowModel, Object columnValue) {
      // ignore
    }
  }

  static class FileDateColumnAdapter extends TableColumnAdapter
  {
    FileDateColumnAdapter() {
      super("Date", false, Date.class, 40, 120, 600);
      setCellRenderer(new DateCellRenderer());
    }
    @Override
    public Object getColumnValue(Object rowModel, int rowNumber) {
      File file = (File)rowModel;
      long time = file.lastModified();
      return((time == 0L) ? null : new Date(time));
    }
    @Override
    public void setColumnValue(Object rowModel, Object columnValue) {
      // ignore
    }
  }

  static class DateCellRenderer extends DefaultTableCellRenderer
  {
    DateCellRenderer() {
      super();
      setHorizontalAlignment();
    }

    private void setHorizontalAlignment() {
        setHorizontalAlignment(JLabel.RIGHT);
    }
    
    @Override
    public Component getTableCellRendererComponent
      (JTable table, Object value, boolean isSelected,
       boolean hasFocus, int row, int column)
    {
      String stringValue =
        (value instanceof Date) ? dateFormat.format((Date)value) : "";

      return(super.getTableCellRendererComponent(table, stringValue,
                                                 isSelected, hasFocus,
                                                 row, column));
    }

    SimpleDateFormat dateFormat =
      new SimpleDateFormat("MM/dd/yyyy hh:mm:ss a");
  }

  static class FileNameCellRenderer extends DefaultTableCellRenderer
  {
    @Override
    public Component getTableCellRendererComponent
      (JTable table, Object value, boolean isSelected,
       boolean hasFocus, int row, int column)
    {
      DefaultTableCellRenderer result = (DefaultTableCellRenderer)
        super.getTableCellRendererComponent(table, value, isSelected,
                                            hasFocus, row, column);

      if (row >= 0) {
        ModelListTableModel tableModel = (ModelListTableModel)table.getModel();
        File file = (File)tableModel.getRowModel(row);
        if (file.isDirectory())
        {
            result.setIcon(UIManager.getIcon("FileView.directoryIcon"));
        }
        else
        {
            result.setIcon(UIManager.getIcon("FileView.fileIcon"));
        }
      }

      return(result);
    }
  }

  static class FileNameComparator implements Comparator
  {
    @Override
    public boolean equals(Object obj)
    {
      return(obj == this);
    }
    
    @Override
    public int hashCode() {
        return super.hashCode();
    }

    @Override
    public int compare(Object obj1, Object obj2)
    {
      File file1 = (File)obj1;
      File file2 = (File)obj2;

      if ((file1 == file2) || file1.equals(file2))
    {
        return(0);
    }
      if (file1.isDirectory() == file2.isDirectory())
    {
        return(file1.getPath().compareTo(file2.getPath()));
    }
    else
    {
        return(file1.isDirectory() ? -1 : 1);
    }
    }
  }

}
