/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */
package modelobjects.util.swing;

import java.io.File;
import java.util.Comparator;
import java.util.List;

import modelobjects.util.SortedArrayList;

/**
 *  DirectoryFileTreeNode is used to show directory trees
 */
public class DirectoryFileTreeNode extends DynamicTreeNode
{
  public DirectoryFileTreeNode(File file, boolean dirsOnly)
  {
    super(file, file.isDirectory());
    this.file = file;
    this.dirsOnly = dirsOnly;
  }

  @Override
protected List computeChildren()
  {
    SortedArrayList childrenNodes = new SortedArrayList(fileComparator);
    File[] files = file.listFiles();
    int numFiles = ((files == null) ? 0 : files.length);
    for (int i = 0; i < numFiles; i++) {
      if (!dirsOnly || files[i].isDirectory())
    {
        childrenNodes.add(new DirectoryFileTreeNode(files[i], dirsOnly));
    }
    }
    return(childrenNodes);
  }

  public File getFile()
  {
    return(file);
  }

  @Override
public String toString()
  {
    String name = file.getName();
    if (name.equals("."))
    {
        return(file.getAbsolutePath());
    }
    else
    {
        return(name);
    }
  }

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  Representation
  ///
  /////////////////////////////////////////////////////////////////////////////

  private File                  file;
  private boolean               dirsOnly;

  private static Comparator     fileComparator = new FileComparator();

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  FileComparator
  ///
  /////////////////////////////////////////////////////////////////////////////

  private static class FileComparator implements Comparator
  {
    @Override
    public boolean equals(Object obj)
    {
      return(obj == this);
    }
    
    @Override
    public int hashCode() {
        return super.hashCode();
    }

    @Override
    public int compare(Object obj1, Object obj2)
    {
      DirectoryFileTreeNode node1 = (DirectoryFileTreeNode)obj1;
      DirectoryFileTreeNode node2 = (DirectoryFileTreeNode)obj2;
      File file1 = node1.getFile();
      File file2 = node2.getFile();

      if ((file1 == file2) || file1.equals(file2))
    {
        return(0);
    }
      if (file1.isDirectory() == file2.isDirectory())
    {
        return(file1.getPath().compareTo(file2.getPath()));
    }
    else
    {
        return(file1.isDirectory() ? -1 : 1);
    }
    }
  }
}
