/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */

package modelobjects.util.swing;

import java.text.Collator;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.Objects;

import javax.swing.Icon;
import javax.swing.JTree;
import javax.swing.UIManager;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.MutableTreeNode;


/**
 *  This subclass of DefaultMutableTreeNode supports more flexible
 *  rendering, expansion, and refresh behavior.
 */
public abstract class DynamicTreeNode extends DefaultMutableTreeNode
{
  /**
   *  Constructs an DynamicTreeNode with the specified user object and
   *  indicate whether the node allows children.
   */
  public DynamicTreeNode(Object userObject, boolean allowsChildren)
  {
    super(userObject, allowsChildren);

    sortChildren = true;
  }

  /**
   *  Returns a List of DynamicTreeNodes representing the children of this
   *  DynamicTreeNode. This method is called if the node indicates that
   *  it allows children but the current children List is null, or during
   *  a refresh operation.  This method must not return null.  If there are
   *  no children, the method should return a new empty List.
   */
  protected abstract List computeChildren();

  /**
   *  Override equals behavior to say that two DynamicTreeNodes are
   *  equivalent if their respective user-objects are equivalent.
   */
  @Override
public boolean equals(Object other)
  {
    if (other != null && this.getClass() == other.getClass())
    {
      DynamicTreeNode that = (DynamicTreeNode)other;
      return this.getUserObject().equals(that.getUserObject());
    }
    return false;
  }
  
  @Override
public int hashCode() {
      return Objects.hash(getUserObject());
  }

  public boolean isSortChildren()
  {
      return sortChildren;
  }

  public void setSortChildren(boolean sortChildren)
  {
      this.sortChildren = sortChildren;
  }

  /**
   *  Returns the Icon to draw for the node according to its current state.
   *  This method is not used by the default TreeCellRenderer, but rather
   *  is intended to be used with a more versatile TreeCellRenderer that
   *  knows about DynamicTreeNode.
   */
  public Icon getNodeIcon(boolean isExpanded, boolean isSelected,
                          boolean isLeaf, boolean hasFocus)
  {
    return(getAllowsChildren() ?
           (isSelected ? getOpenIcon() : getClosedIcon()) :
           getLeafIcon());
  }

  /**
   *  Returns the String to draw for the node according to its current state.
   *  This method is not used by the default TreeCellRenderer, but rather
   *  is intended to be used with a more versatile TreeCellRenderer that
   *  knows about DynamicTreeNode.
   */
  public String getNodeString(boolean isExpanded, boolean isSelected,
                              boolean isLeaf, boolean hasFocus)
  {
    return(this.toString());
  }

  /**
   *  Returns the icon to use for leaf nodes.
   */
  public Icon getLeafIcon()
  {
    if (leafIcon == null)
    {
        leafIcon = UIManager.getIcon("Tree.leafIcon");
    }
    return(leafIcon);
  }

  /**
   *  Returns the icon for expanded non-leaf nodes.
   */
  public Icon getOpenIcon()
  {
    if (openIcon == null)
    {
        openIcon = UIManager.getIcon("Tree.openIcon");
    }
    return(openIcon);
  }

  /**
   *  Returns the icon for unexpanded non-leaf nodes.
   */
  public Icon getClosedIcon()
  {
    if (closedIcon == null)
    {
        closedIcon = UIManager.getIcon("Tree.closedIcon");
    }
    return(closedIcon);
  }

  /**
   *  Assigns the icon for leaf nodes.
   */
  public void setLeafIcon(Icon icon)
  {
    leafIcon = icon;
  }

  /**
   *  Assigns the icon for expanded non-leaf nodes.
   */
  public void setOpenIcon(Icon icon)
  {
    openIcon = icon;
  }

  /**
   *  Assigns the icon for unexpanded non-leaf nodes.
   */
  public void setClosedIcon(Icon icon)
  {
    closedIcon = icon;
  }

  public void releaseChildren()
  {
    if (hasComputedChildren)
    {
      removeAllChildren();
      hasComputedChildren = false;
    }
  }

  /**
   *  Brings this subtree up to date.
   *  The default implementation of this method does nothing.
   */
  public void refresh(JTree tree, DefaultTreeModel treeModel)
  {
    // if we haven't done any work yet, don't do it now either
    if (!hasComputedChildren)
    {
        return;
    }

    // recompute the current list of children
    List currentChildren = computeChildren();

    // check for any children that aren't in the current list
    for (int i = getChildCount() - 1; i >= 0; i--)
    {
      DynamicTreeNode node = (DynamicTreeNode)getChildAt(i);
      if (!currentChildren.contains(node))
    {
        treeModel.removeNodeFromParent(node);
    }
    }

    // check to see if anything needs to be added
    for (int i = currentChildren.size() - 1; i >= 0; i--)
    {
      DynamicTreeNode node = (DynamicTreeNode)currentChildren.get(i);
      if(children == null || !children.contains(node))
    {
        treeModel.insertNodeInto(node, this, -1);
    }
    }

    // recursively refresh the children
    this.refreshChildren(tree, treeModel);
  }

  /**
   *  Refreshes the children of this DynamicTreeNode.
   *  This method is normally called by the node's own refresh() method.
   *  If the node's children have not yet been computed, this method
   *  will do nothing.  Otherwise it will call refresh() on each of the
   *  node's children.
   */
  protected void refreshChildren(JTree tree, DefaultTreeModel treeModel)
  {
    for (int i = 0; i < getChildCount(); i++)
    {
      DynamicTreeNode child = (DynamicTreeNode)getChildAt(i);
      child.refresh(tree, treeModel);
    }
  }

  /**
   *  Override to say that a leaf is a node that does not allow children,
   *  rather than a node that doesn't [currently] have any.  This is
   *  important for supporting on-demand (i.e. lazy) computation of children.
   */
  @Override
public boolean isLeaf()
  {
    if(hasComputedChildren && getChildCount() > 0)
    {
        return false;
    }

    if(!getAllowsChildren())
    {
        return true;
    }

    return false;
  }

  /**
   *  Returns the current List of child nodes, or null if there isn't one.
   *  This method does not cause computeChildren to be called.  Modification
   *  of this List does not effect the tree, so it really should never be
   *  changed except by this class.
   */
  final public List getchildren()
  {
    int count = getChildCount();

    if(count == 0)
    {
        return null;
    }

    ArrayList list = new ArrayList();

    for (int i = 0; i < count; i++)
    {
        list.add(getChildAt(i));
    }

    return list;
  }

  public boolean getHasComputedChildren()
  {
    return hasComputedChildren;
  }

  /**
   * Computes the children List if the node allows children and
   * the children List is null.
   */
  final protected void computeChildrenIfNecessary()
  {
    if (getAllowsChildren() && !hasComputedChildren)
    {
      List children = computeChildren();

      if (children == null)
    {
        throw(new IllegalStateException("computeChildren returned null"));
    }

      hasComputedChildren = true;

      // add all of the nodes to the tree
      for (int i = 0, n = children.size(); i < n; i++)
    {
        add((DynamicTreeNode)children.get(i));
    }
    }
  }

  /**
   * Overrides as final in order to enforce policy about null children.
   */
  @Override
final public int getChildCount()
  {
    if (getAllowsChildren())
    {
        computeChildrenIfNecessary();
    }

    return super.getChildCount();
  }

  /**
   *  Overrides as final in order to enforce policy about null children.
   */
  @Override
final public Enumeration children()
  {
    if (getAllowsChildren())
    {
        computeChildrenIfNecessary();
    }

    return super.children();
  }

  /**
   * Overrides in order to enforce policy about null children and to make
   * sure child is an DynamicTreeNode.
   */
  @Override
final public void add(MutableTreeNode child)
  {
    computeChildrenIfNecessary();

    if (!isSortChildren())
    {
        super.add((DynamicTreeNode)child);
    }
    else
    {
        super.insert((DynamicTreeNode)child, getInsertIndex(child));
    }
  }

  /**
   * Overrides in order to enforce policy about null children and to make
   * sure child is an DynamicTreeNode.
   */
  @Override
final public void insert(MutableTreeNode child, int index)
  {
    computeChildrenIfNecessary();

    if (isSortChildren() || (index == -1))
    {
        super.insert((DynamicTreeNode)child, getInsertIndex(child));
    }
    else
    {
        super.insert((DynamicTreeNode)child, index);
    }
  }

  protected int getInsertIndex(MutableTreeNode child)
  {
      int index = getChildCount();

      for (int i = 0; i < getChildCount(); i++)
      {
          MutableTreeNode node = (MutableTreeNode)getChildAt(i);

          if (collator.compare(child.toString(), node.toString()) <= 0)
          {
              index = i;
              break;
          }
      }
      return index;
  }

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  Representation
  ///
  /////////////////////////////////////////////////////////////////////////////

  private static final Collator collator = Collator.getInstance();

  protected boolean hasComputedChildren;
  protected boolean sortChildren;

  private transient Icon leafIcon;
  private transient Icon openIcon;
  private transient Icon closedIcon;

}
