/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */
package modelobjects.util.swing;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Graphics;

import javax.swing.Icon;
import javax.swing.JLabel;
import javax.swing.JTable;
import javax.swing.UIManager;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.JTableHeader;

/**
 *  HeaderCellRenderer works with ModelListTableModel, adding graphical
 *  annotations to header cells to show which columns are sortable, and
 *  which column the table is sorted by (if any), and whether it's sorted
 *  in ascending or descending order.
 */
public class HeaderCellRenderer extends DefaultTableCellRenderer
{
    private Icon                m_iconAscend;
    private Icon                m_iconDescend;
    private IModelTableModel    m_tableModel;
    private TableColumnAdapter  m_columnAdapter;


    public HeaderCellRenderer(IModelTableModel   tableModel,
                              TableColumnAdapter columnAdapter)
    {
      m_tableModel    = tableModel;
      m_columnAdapter = columnAdapter;
      m_iconAscend    = new TableArrowIcon(true);
      m_iconDescend   = new TableArrowIcon(false);

      setProperties();
    }

    private void setProperties() {
        setHorizontalAlignment(JLabel.CENTER);
        setHorizontalTextPosition(JLabel.LEADING);
    }
    
    @Override
    public Component getTableCellRendererComponent(JTable  table,
                                                   Object  value,
                                                   boolean isSelected,
                                                   boolean hasFocus,
                                                   int     row,
                                                   int     column)
    {
        if (table != null)
        {
            JTableHeader header = table.getTableHeader();

            if (header != null)
            {
                setForeground(header.getForeground());
                setBackground(header.getBackground());
                setFont(header.getFont());
            }
        }

        setText((value == null) ? "" : value.toString());
        setBorder(UIManager.getBorder("TableHeader.cellBorder"));

        return this;
    }

    @Override
    public void paint(Graphics g)
    {
        super.paint(g);

        g.setColor(getForeground());

        boolean ascending  = m_tableModel.isSortedInAscendingOrder();
        int     sortColumn = m_tableModel.getSortColumn();
        boolean sorted = ((sortColumn >= 0) &&
                         (m_tableModel.getColumnAdapter(sortColumn) == m_columnAdapter));

        if (sorted)
        {
            Dimension size = this.getSize();
            Icon      icon = ascending ? m_iconAscend : m_iconDescend;

            icon.paintIcon(this, g, size.width - icon.getIconWidth() - 4,
                           (size.height - icon.getIconHeight() - 2) / 2);
        }
        else
        {
            setIcon(null);
        }
    }

    class TableArrowIcon implements Icon
    {
        private boolean isAscending;

        public TableArrowIcon(boolean isAscendingDirection)
        {
            isAscending = isAscendingDirection;
        }

        @Override
        public int getIconHeight()
        {
            return 5;
        }

        @Override
        public int getIconWidth()
        {
            return 9;
        }

        @Override
        public void paintIcon(Component c, Graphics g, int x, int y)
        {
            // colors
            // 1 = controlDkShadow
            // 2 = infoText
            // 3 = controlLtHighlight
            if ( isAscending )
            {
                //{0, 0, 0, 2, 2, 0, 0, 0, 0},
                //{0, 0, 2, 1, 1, 1, 0, 0, 0},
                //{0, 2, 1, 1, 1, 1, 1, 0, 0},
                //{2, 1, 1, 1, 1, 1, 1, 1, 0},
                //{0, 3, 3, 3, 3, 3, 3, 3, 3}
                g.setColor((Color)UIManager.getDefaults().get("infoText"));
                g.drawLine(x, y + 3, x+3, y );
                g.drawLine(x+4, y, x+4, y );
                g.setColor((Color)UIManager.getDefaults().get("controlDkShadow"));
                g.drawLine(x+3, y+1, x+5, y+1 );
                g.drawLine(x+2, y+2, x+6, y+2 );
                g.drawLine(x+1, y+3, x+7, y+3 );
                g.setColor((Color)UIManager.getDefaults().get("controlLtHighlight"));
                g.drawLine(x+1, y+4, x+8, y+4 );
            }
            else
            {
                //{2, 2, 2, 2, 2, 2, 2, 2},
                //{0, 1, 1, 1, 1, 1, 1, 3},
                //{0, 0, 1, 1, 1, 1, 3, 0},
                //{0, 0, 0, 1, 1, 3, 0, 0},
                //{0, 0, 0, 0, 3, 0, 0, 0}
                g.setColor((Color)UIManager.getDefaults().get("infoText"));
                g.drawLine(x, y, x+7, y );
                g.setColor((Color)UIManager.getDefaults().get("controlDkShadow"));
                g.drawLine(x+1, y+1, x+6, y+1 );
                g.drawLine(x+2, y+2, x+5, y+2 );
                g.drawLine(x+3, y+3, x+4, y+3 );
                g.setColor((Color)UIManager.getDefaults().get("controlLtHighlight"));
                g.drawLine(x+7, y+1, x+4, y+4 );
            }
        }

    }

}
