/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product include software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */
package modelobjects.util.swing;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;

import javax.swing.event.ChangeEvent;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.TableColumnModelEvent;
import javax.swing.event.TableColumnModelListener;
import javax.swing.event.TableModelEvent;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableColumnModel;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;

/**
 *  ModelListTableModel is the central class of a framework for building
 *  highly flexible, model-object-oriented TableModels for JTables.
 *  The contents of a ModelListTableModel is a List of Model Objects mapped
 *  one to each row of the table.
 */
public class ModelListTableModel extends    AbstractTableModel
                                 implements IModelTableModel, TableColumnModelListener
{
    private int                  m_columnCount;
    private TableColumnAdapter[] m_columnAdapters;
    private List                 contents;
    private TableColumnModel     m_columnModel;

    // sorting support
    private Comparator           sortComparator;
    private int                  sortColumn;
    private boolean              sortedAscending;

    // provide minimal caching of cell values
    private int                  lastRow = -1;
    private int                  lastCol = -1;
    private Object               lastVal = null;


  public ModelListTableModel(TableColumnAdapter[] columnAdapters)
  {
    this(columnAdapters, new ArrayList());
  }

  public ModelListTableModel(TableColumnAdapter[] columnAdapters, List contents)
  {
    m_columnAdapters = columnAdapters;
    m_columnCount    = columnAdapters.length;
    m_columnModel    = makeColumnModel();

    setContents(contents);
  }

  /**
   *  Construct a ModelListTableModel for the specified row-model class,
   *  and with the specified array of TableColumnAdapters and with the
   *  specified model List contents.Selection mode is MULTIPLE_INTERVAL_SELECTION by default
   *
   * @deprecated  As of MA v2.1, replaced by {@link #ModelListTableModel(TableColumnAdapter[])}
   *              and {@link #ModelListTableModel(TableColumnAdapter[], List)}
   */
  public ModelListTableModel(Class                rowModelClass,
                             TableColumnAdapter[] columnAdapters,
                             List                 contents)
  {
    this(columnAdapters, contents);
  }

  /**
   * @deprecated  As of MA v2.1 - please use other constructors.
   */
  public ModelListTableModel(Class                rowModelClass,
                             TableColumnAdapter[] columnAdapters,
                             List                 contents,
                             int                  selectionMode)
  {
    this(columnAdapters, contents);
  }

  /**
   *  Return the model List contents of this ModelListTableModel.
   */
  @Override
public List getContents()
  {
    return contents;
  }

  /**
   *  Assign the model List contents of this ModelListTableModel.
   */
  @Override
public final void setContents(List contents)
  {
    List oldContents = this.contents;

    this.contents = contents;
    invalidateCellCache();

    sortByColumn(getSortColumn());

    fireTableDataChanged();
    fireTableStructureChanged();
    fireTableRowsInserted(0, contents.size());
  }

  /**
   *  Return this ModelListTableModel's array of TableColumnAdapters.
   */
  @Override
public TableColumnAdapter[] getColumnAdapters()
  {
    return m_columnAdapters;
  }

  /**
   *  Return the TableColumnAdapter for the specified model column index.
   */
  @Override
public TableColumnAdapter getColumnAdapter(int modelColumnIndex)
  {
    return m_columnAdapters[modelColumnIndex];
  }

  /**
   *  Override to invalidate cell cache on every change.
   */
  @Override
public void fireTableChanged(TableModelEvent event)
  {
    invalidateCellCache();

    super.fireTableChanged(event);
  }

  /**
   *  Return the number of row model objects in this ModelListTableModel.
   */
  @Override
public int getRowCount()
  {
    return contents.size();
  }

  /**
   *  Return the number of columns in this ModelListTableModel.
   */
  @Override
public int getColumnCount()
  {
    return m_columnCount;
  }

  @Override
public String getColumnName(int columnIndex)
  {
      return getColumnAdapter(columnIndex).getHeaderLabel();
  }

  /**
   *  Return the row model object at the specified index of this
   *  ModelListTableModel.
   */
  @Override
public Object getRowModel(int row)
  {
    return ((row >= 0) && (row < contents.size())) ? contents.get(row) : null;
  }

  /**
   *  Return whether the cell at the specified row and column is editable.
   */
  @Override
public boolean isCellEditable(int row, int column)
  {
    return m_columnAdapters[column].isColumnEditable(getRowModel(row));
  }

  /**
   * Returns the row index of the first match in the contents List if
   * the value is present.
   *
   * @param  startRow     the row to start the search
   * @param  columnIndex  the column inwhich to search
   * @param  value        the value to find
   * @return the row index of the first match or -1 if no match is found
   */
  @Override
public int findCell(int startRow, int columnIndex, Object value)
  {
    for (int rowIndex = startRow; rowIndex < getRowCount(); rowIndex++)
    {
      Object cellValue = getValueAt(rowIndex, columnIndex);

      if ((cellValue == null) && (value == null))
    {
        return rowIndex;
    }

      if ((cellValue != null) && (value != null) && value.equals(cellValue))
    {
        return rowIndex;
    }
    }

    return -1;
  }

  /**
   * Hacked version of the java.util.Arrays binarySearch method that tries
   * to find an exact match on the keyed object.
   */
  private int binarySearch(Object[] a, Object key, Comparator c)
  {
    int low = 0;
    int high = a.length-1;
    int mid = -1;
    int cmp = -1;

    while (low <= high)
    {
      mid = (low + high) >> 1;
      Object midVal = a[mid];
      cmp = c.compare(midVal, key);

      if (cmp < 0)
    {
        low = mid + 1;
    }
    else if (cmp > 0)
    {
        high = mid - 1;
    }
    else
    {
        break;
    }
    }

    if (cmp == 0)
    {
      // Double check to make sure we really found the right object,
      // and not just another that equivalent by the comparator, since
      // some comparators might treat many objects as equivalent.\

      if (a[mid].equals(key))
    {
        return(mid);
    }
    else
      {
        // scan backward
        for (int i = mid-1; i >= 0; i--)
        {
          if (c.compare(key, a[i]) != 0)
        {
            break;
        }

          if (key.equals(a[i]))
        {
            return(i);
        }
        }

        // scan forward
        for (int i = mid+1; i < a.length; i++)
        {
          if (c.compare(key, a[i]) != 0)
        {
            break;
        }

          if (key.equals(a[i]))
        {
            return(i);
        }
        }

        // Ok, so we couldn't find a direct match but we did get a match
        // on the comparator so the mid-point should be the insertion point.
        return -(mid + 1);
      }
    }

    return -(low + 1);  // key not found.
  }

  /**
   *  Return the index of the specified object in the contents List if it
   *  is present, or the index at which to insert the object otherwise,
   *  encoded as (-1 - insertionIndex).
   *  If the contents are sorted, a binary search is used.  Otherwise a
   *  linear search is used and if the object is not present the insertion
   *  index is at the end of the list.
   */
  @Override
public int getIndexOf(Object obj)
  {
    if (sortComparator == null)
    {
        return contents.indexOf(obj);
    }

    return binarySearch(contents.toArray(), obj, sortComparator);
  }

  /**
   *  Insert the specified row model object into this ModelListTableModel
   *  at the specified row index.
   */
  @Override
public void insertRow(Object rowModel, int index)
  {
    contents.add(index, rowModel);
    fireTableRowsInserted(index, index);
    recheckSortedOrder(index);
  }

  /**
   *  Insert the specified row model object into this ModelListTableModel
   *  maintaining sorted order if the model is currently sorted, or at the
   *  end otherwise.  It is illegal to insert a row model object that is
   *  already present.
   */
  @Override
public void insertRowMaintainSort(Object rowModel)
  {
    int index = getIndexOf(rowModel);
    boolean refresh = false;

    if (index >= 0) {
      // check for duplicate insert
      if ((index < getRowCount()) && contents.get(index).equals(rowModel))
      {
        // Lets not throw a runtime exception here...instead we will simply
        // ask the table to repaint the row relating to the rowModel.
        //
        refresh = true;
        //throw(new IllegalStateException("attempt to insert object already in the table: " + rowModel));
      }
    }
    else {
      // convert "not-present" index into insertion index
      index = -1 - index;
    }

    if (refresh)
    {
        fireTableRowsUpdated(index, index);
    }
    else
    {
        contents.add(index, rowModel);
        fireTableRowsInserted(index, index);
    }
  }

  /**
   *  Delete the row at the specified index.
   */
  @Override
public void deleteRow(int index)
  {
    contents.remove(index);
    fireTableRowsDeleted(index, index);
  }

  /**
   *  Delete the specified object from this ModelListTableModel if it is
   *  present.
   */
  @Override
public void delete(Object obj)
  {
    int row = getIndexOf(obj);
    if (row >= 0)
    {
        deleteRow(row);
    }
  }

  /**
   *  Clear the contents of this ModelListTableModel.
   */
  @Override
public void clear()
  {
    int numRows = contents.size();
    if (numRows > 0) {
      contents.clear();
      fireTableRowsDeleted(0, numRows-1);
    }
  }

  /**
   *  Notify TableModelListeners that the specified row has changed.
   */
  @Override
public void rowChanged(int index)
  {
    fireTableRowsUpdated(index, index);
    recheckSortedOrder(index);
  }

  /**
   *  Return the value of the cell at the specified row and column.
   */
  @Override
public Object getValueAt(int row, int column)
  {
    if ((row == lastRow) && (column == lastCol))
    {
        return lastVal;
    }

    Object value = m_columnAdapters[column].getColumnValue(getRowModel(row), row);

    // save last computed value since this method gets call a lot
    lastRow = row;
    lastCol = column;
    lastVal = value;

    return value;
  }

  /**
   *  Assign a new value to the cell at the specified row and column.
   */
  @Override
public void setValueAt(Object value, int row, int column)
  {
    invalidateCellCache();
    m_columnAdapters[column].setColumnValue(getRowModel(row), value);

    recheckSortedOrder(row);
  }

  /**
   *  Return the TableCellRenderer for the specified column.
   */
  public TableCellRenderer getColumnCellRenderer(int columnIndex)
  {
    return m_columnAdapters[columnIndex].getCellRenderer();
  }

  /**
   *  Assign the TableCellRenderer for the specified column.
   */
  public void setColumnCellRenderer(int columnIndex,
                                    TableCellRenderer cellRenderer)
  {
    m_columnAdapters[columnIndex].setCellRenderer(cellRenderer);
  }

  /**
   *  Return the TableCellEditor for the specified column.
   */
  public TableCellEditor getColumnCellEditor(int columnIndex)
  {
    return m_columnAdapters[columnIndex].getCellEditor();
  }

  /**
   *  Assign the TableCellEditor for the specified column.
   */
  public void setColumnCellEditor(int columnIndex, TableCellEditor cellEditor)
  {
    m_columnAdapters[columnIndex].setCellEditor(cellEditor);
  }

  /**
   *  Return the model Class of the the specified column in this
   *  ModelListTableModel.
   */
  @Override
public Class getColumnClass(int columnIndex)
  {
    return m_columnAdapters[columnIndex].getColumnClass();
  }

  /**
   *  Create the TableColumnModel for this ModelListTableModel based on the
   *  array of TableColumnAdapters.
   */
  private DefaultTableColumnModel makeColumnModel()
  {
    DefaultTableColumnModel tcm = new DefaultTableColumnModel();

    for (int i = 0; i < m_columnAdapters.length; i++)
    {
      TableColumnAdapter columnAdapter = m_columnAdapters[i];
      TableColumn        column        = columnAdapter.makeModelPropertyTableColumn(i);

      column.setHeaderRenderer(new HeaderCellRenderer(this, columnAdapter));

      tcm.addColumn(column);
    }

    return tcm;
  }

  public TableColumnModel getTableColumnModel()
  {
      return m_columnModel;
  }

  /**
   *  Invalidate the unit cell cache used for improving the performance of
   *  the getValueAt(int,int) method.
   */
  private void invalidateCellCache()
  {
    lastRow = -1;
    lastCol = -1;
    lastVal = null;
  }

  /////////////////////////////////////////////////////////////////////////////
  ///
  /// sorting support
  ///
  /////////////////////////////////////////////////////////////////////////////

  /**
   *  Sort this ModelListTableModel by the specified model column.  If the
   *  table is not currently sorted by that column, the column's normal
   *  Comparator is used to perform the sort.  Otherwise, the column's
   *  reverse comparator is used.
   */
  @Override
public void sortByColumn(int modelColumnIndex)
  {
    sortByColumn(modelColumnIndex, ((sortColumn != modelColumnIndex) || !sortedAscending));
  }

  /**
   *  Sort this ModelListTableModel by the specified model column, specifying
   *  whether to sort in ascending or descending order.
   *  <p>
   *  The ModelListTableModel is sorted by extracting the complete contents
   *  List into an object array, sorting the array using the Comparator,
   *  and then replacing the contents of the list with the sorted array.
   */
  @Override
public void sortByColumn(int modelColumnIndex, boolean ascending)
  {
    //IL(can cause array out of bound exception if column index is -1)
    if (modelColumnIndex == -1)
    {
        return;
    }
    TableColumnAdapter columnAdapter = m_columnAdapters[modelColumnIndex];

    if (!columnAdapter.isColumnSortable())
    {
        return;
    }

/*
      System.out.println("sorting by model column " + modelColumnIndex);
      System.out.println("  column name = " + columnAdapter.getHeaderLabel());
      System.out.println("  ascending = " + ascending);
*/

    Comparator comparator = (ascending ? columnAdapter.getRowComparator() :
                             columnAdapter.getReverseRowComparator());

    Object[] modelArray = contents.toArray();
    Arrays.sort(modelArray, comparator);

    // clear the contents and re-load it from the sorted array
    contents.clear();
    for (int i = 0, n = modelArray.length; i < n; i++)
    {
        contents.add(i, modelArray[i]);
    }

    setSortedByColumn(modelColumnIndex, ascending);

    fireTableDataChanged();
  }

  /**
   *  Assert that the table is sorted by the specified column's Comparator,
   *  and indicate whether it's sorted ascending or not.  If the sortColumn
   *  and/or sortComparator are changed by this method, PropertyChangeEvents
   *  are fired to notify listeners about the changes.
   */
  @Override
public void setSortedByColumn(int modelColumnIndex, boolean ascending)
  {
    Comparator oldComparator = sortComparator;
    int oldSortColumn = sortColumn;

    this.sortColumn = modelColumnIndex;
    this.sortedAscending = ascending;

    if (modelColumnIndex == -1)
    {
        this.sortComparator = null;
    }
    else
    {
      TableColumnAdapter columnAdapter = m_columnAdapters[modelColumnIndex];

      if (ascending)
    {
        this.sortComparator = columnAdapter.getRowComparator();
    }
    else
    {
        this.sortComparator = columnAdapter.getReverseRowComparator();
    }
    }

    // The table needs to re-paint its header because the sort column has changed
    // or the direction of sort has changed.
    //
    if ((oldSortColumn != sortColumn) || (oldComparator != sortComparator))
    {
        fireTableStructureChanged();
    }
  }

  /**
   *  Return the comparator by which the table is sorted, or null if it's
   *  not currently sorted.
   */
  @Override
public Comparator getSortComparator()
  {
    return sortComparator;
  }

  /**
   *  Return whether this ModelListTableModel is current sorted.
   */
  @Override
public boolean isSorted()
  {
    return (sortComparator != null);
  }

  /**
   *  Check to make sure the specified row is still in sorted order with
   *  respect to its previous and next rows.  If it's not, mark the table
   *  as being unsorted.
   */
  void recheckSortedOrder(int rowNumber)
  {
    if ((sortComparator != null) && (rowNumber >= 0)) {
      Object rowModel = getRowModel(rowNumber);

      //!! System.out.println("rechecking sortedness of row " + rowNumber);

      boolean sorted = true;

      if (rowNumber > 0) {
        //!! System.out.println("rechecking sortedness against previous row");
        Object prevRowModel = getRowModel(rowNumber - 1);
        if (sortComparator.compare(prevRowModel, rowModel) > 0)
        {
            sorted = false;
        }
      }
      if (sorted && ((rowNumber + 1) < contents.size())) {
        //!! System.out.println("rechecking sortedness against next row");
        Object nextRowModel = getRowModel(rowNumber + 1);
        if (sortComparator.compare(rowModel, nextRowModel) > 0)
        {
            sorted = false;
        }
      }

      if (!sorted) {
        //!! System.out.println("Not sorted any more!");
        setSortedByColumn(-1, false);
      }
    }
  }

  /**
   *  Return the model column by which the table is sorted, or -1 if it's
   *  not currently sorted.
   */
  @Override
public int getSortColumn()
  {
    return sortColumn;
  }

  /**
   *  Return whether this ModelListTableModel is sorted in ascending order.
   *  The results of this method are meaningless if it is not sorted at all.
   */
  @Override
public boolean isSortedInAscendingOrder()
  {
    return sortedAscending;
  }

    //-------------------------------------------------------------------------
    //
    // TableColumnModelListener implementation
    //
    //-------------------------------------------------------------------------

    @Override
    public void columnAdded(TableColumnModelEvent evt)
    {
        m_columnCount++;
    }

    @Override
    public void columnRemoved(TableColumnModelEvent evt)
    {
        m_columnCount--;

        if (sortColumn >= m_columnCount)
        {
            sortColumn = -1;
        }
    }

    @Override
    public void columnMoved(TableColumnModelEvent evt) {}
    @Override
    public void columnSelectionChanged(ListSelectionEvent evt) {}
    @Override
    public void columnMarginChanged(ChangeEvent evt) {}

    //-------------------------------------------------------------------------

}
