/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */
package modelobjects.util.swing;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.util.ArrayList;
import java.util.List;
import java.util.Vector;

import javax.swing.AbstractAction;
import javax.swing.AbstractButton;
import javax.swing.Action;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.event.PopupMenuEvent;
import javax.swing.event.PopupMenuListener;
import javax.swing.event.TableColumnModelListener;
import javax.swing.event.TableModelEvent;
import javax.swing.table.DefaultTableColumnModel;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableModel;

/**
 *  ModelTable is a subclass of JTable that works with IModelTableModel
 *  and supports automatic column sorting through TableSorter.
 */
public class ModelTable extends JTable
{
  public ModelTable(TableColumnAdapter[] columnAdapters)
  {
    this(new ModelListTableModel(columnAdapters, new ArrayList()));
  }

  public ModelTable(IModelTableModel dm)
  {
    this(dm, null, null);
  }

  public ModelTable(IModelTableModel   dm,
                    TableColumnModel   cm,
                    ListSelectionModel sm)
  {
    super(dm, cm, sm);
    prepareModelTable();
  }

  private void prepareModelTable() {
      setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
      setAutoCreateColumnsFromModel(false);
      getTableHeader().addMouseListener(new TableSorter(this));
      getTableHeader().addMouseListener(new PopupMenuShower(this, createHeaderPopupMenu()));
      getColumnModel().addColumnModelListener((TableColumnModelListener)getModel());
  }
  
  /**
   * Override this method to try and set the containing JScrollPane's
   * background to the same color as the tables.
   */
  @Override
public void addNotify()
  {
      super.addNotify();

      JScrollPane sp = (JScrollPane)SwingUtilities.getAncestorOfClass(JScrollPane.class, this);

      if (sp != null)
      {
          sp.getViewport().setBackground(getBackground());
          sp.getColumnHeader().setBackground(getBackground());
      }
  }

    @Override
    protected TableColumnModel createDefaultColumnModel()
    {
        return new ExtendedTableColumnModel();
    }

  @Override
public void createDefaultColumnsFromModel()
  {
    IModelTableModel tableModel = (IModelTableModel)getModel();

    if (tableModel != null)
    {
      // Remove any current columns
      ExtendedTableColumnModel cm = (ExtendedTableColumnModel)getColumnModel();
      cm.dropAllColumns();

      // Create new columns from the data model info
      for (int i = 0; i < tableModel.getColumnCount(); i++)
      {
        TableColumnAdapter adapter = tableModel.getColumnAdapter(i);
        TableColumn        column  = adapter.makeModelPropertyTableColumn(i);

        column.setHeaderRenderer(new HeaderCellRenderer(tableModel, adapter));

        addColumn(column);
      }
    }
  }

  @Override
public void setModel(TableModel dm)
  {
    if ((dm != null) && !(dm instanceof IModelTableModel))
    {
        throw new IllegalArgumentException("TableModel must extend IModelTableModel");
    }

    super.setModel(dm);
  }

    protected JPopupMenu createHeaderPopupMenu()
    {
        TableColumnModel model = getColumnModel();
        final JPopupMenu       pm    = new JPopupMenu("header-popupmenu");

        pm.add(new ColumnResetAction());
        pm.addSeparator();
        pm.addPopupMenuListener(new PopupMenuListener()
        {
            @Override
            public void popupMenuCanceled(PopupMenuEvent evt) {}
            @Override
            public void popupMenuWillBecomeInvisible(PopupMenuEvent evt) {}
            @Override
            public void popupMenuWillBecomeVisible(PopupMenuEvent evt)
            {
                for (int i = 0; i < pm.getComponentCount(); i++)
                {
                    Component item = pm.getComponent(i);

                    if (item instanceof AbstractButton)
                    {
                        TableAction action = (TableAction)((AbstractButton)item).getAction();

                        if (action != null)
                        {
                            item.setEnabled(action.isEnabled());

                            if (item instanceof JCheckBoxMenuItem)
                            {
                                ((JCheckBoxMenuItem)item).setSelected(action.isSelected());
                            }
                        }
                    }
                }
            }
        });

        for (int k = 0; k < getModel().getColumnCount(); k++)
        {
            TableColumn        columnModel   = model.getColumn(k);
            TableColumnAdapter columnAdapter = ((IModelTableModel)getModel()).getColumnAdapter(k);
            Action             action        = new ColumnKeeperAction(columnModel.getIdentifier(), columnModel.getHeaderValue());
            JCheckBoxMenuItem  item          = new JCheckBoxMenuItem(action);

            item.setSelected(true);

            pm.add(item);
        }

        return pm;
    }

    class TableAction extends AbstractAction
    {
        public TableAction(String name)
        {
            super(name);
        }

        @Override
        public void actionPerformed(ActionEvent evt) {}
        public boolean isSelected() { return false; }
    }

    class ColumnKeeperAction extends TableAction
    {
        protected Object m_identifier;

        public ColumnKeeperAction(Object identifier, Object headerValue)
        {
            super(headerValue.toString());

            m_identifier = identifier;
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            JCheckBoxMenuItem        item   = (JCheckBoxMenuItem)evt.getSource();
            ExtendedTableColumnModel model  = (ExtendedTableColumnModel)ModelTable.this.getColumnModel();
            TableColumn              column = model.getTableColumnExt(m_identifier);

            if (item.isSelected())
            {
                model.addColumn(column);
            }
            else
            {
                model.removeColumn(column);
            }

            ModelTable.this.tableChanged(new TableModelEvent(getModel(), TableModelEvent.HEADER_ROW));
            ModelTable.this.repaint();
        }

        @Override
        public boolean isSelected()
        {
            ExtendedTableColumnModel model  = (ExtendedTableColumnModel)ModelTable.this.getColumnModel();
            TableColumn              column = model.getTableColumnExt(m_identifier);

            return model.isColumnVisible(column);
        }

        @Override
        public boolean isEnabled()
        {
            ExtendedTableColumnModel model  = (ExtendedTableColumnModel)ModelTable.this.getColumnModel();
            TableColumn              column = model.getTableColumnExt(m_identifier);

            return !((model.getColumnCount() == 1) && model.isColumnVisible(column));
        }

    }

    class ColumnResetAction extends TableAction
    {
        public ColumnResetAction()
        {
            super("Reset to Defaults");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            ExtendedTableColumnModel model = (ExtendedTableColumnModel)ModelTable.this.getColumnModel();

            model.resetToDefaults();
        }

        @Override
        public boolean isEnabled()
        {
            ExtendedTableColumnModel model  = (ExtendedTableColumnModel)ModelTable.this.getColumnModel();

            return !model.isDefaulted();
        }
    }

    class ExtendedTableColumnModel extends DefaultTableColumnModel
    {
        private Vector m_all = new Vector();

        public ExtendedTableColumnModel()
        {
            super();
        }

        public void dropAllColumns()
        {
            while (getColumnCount() > 0)
            {
                removeColumn(getColumn(0));
            }

            m_all.clear();
        }

        @Override
        public void addColumn(TableColumn column)
        {
            int           index = findWrappedColumn(column);
            WrappedColumn wc    = (index >= 0) ? (WrappedColumn)m_all.get(index) : null;
            boolean       add   = false;

            if (wc == null)
            {
                wc = new WrappedColumn(column);

                m_all.add(wc);

                wc.setInitialPosition(m_all.indexOf(wc));

                add = true;
            }
            else
            if (!wc.isVisible())
            {
                add = true;
                wc.setVisible(true);
            }

            if (add)
            {
                // The following code finds the correct insertion point for the
                // "now" visible column...
                //
                int insertIndex = 0;

                for (int i = 0; i < m_all.size(); i++)
                {
                    WrappedColumn tmpWC = (WrappedColumn)m_all.get(i);

                    if (tmpWC.getTableColumn() == column)
                    {
                        break;
                    }
                    else
                    if (tmpWC.isVisible())
                    {
                        insertIndex++;
                    }
                }

                // There is no way of inserting a column at a particular index
                // with the DefaultTableColumnModel so we simply add it and then
                // move it to the correct location (using the superclass methods
                // so as to avoid the m_all vector house keeping.
                //
                super.addColumn(column);
                super.moveColumn(getColumnCount()-1, insertIndex);
            }
        }

        @Override
        public void removeColumn(TableColumn column)
        {
            int           index = findWrappedColumn(column);
            WrappedColumn wc    = (index >= 0) ? (WrappedColumn)m_all.get(index) : null;

            if (wc != null)
            {
                if (wc.isVisible())
                {
                    wc.setVisible(false);

                    super.removeColumn(column);
                }
            }
        }

        @Override
        public void moveColumn(int columnIndex, int newIndex)
        {
            if (columnIndex != newIndex)
            {
                WrappedColumn wc = (WrappedColumn)m_all.remove(columnIndex);

                if (wc != null)
                {
                    m_all.add(newIndex, wc);
                }
            }

            super.moveColumn(columnIndex, newIndex);
        }

        public void resetToDefaults()
        {
            for (int i = 0; i < m_all.size(); i++)
            {
                WrappedColumn wc = (WrappedColumn)m_all.get(i);

                addColumn(wc.getTableColumn());
            }

            WrappedColumn[] tmp = (WrappedColumn[])m_all.toArray(new WrappedColumn[m_all.size()]);

            for (int j = 0; j < tmp.length; j++)
            {
                WrappedColumn wc = tmp[j];

                int initialPosition = wc.getInitialPosition();
                int currentPosition = getColumnIndex(wc.getTableColumn().getIdentifier());

                if (currentPosition != initialPosition)
                {
                    moveColumn(currentPosition, initialPosition);
                }
            }
        }

        private int findWrappedColumn(TableColumn column)
        {
            for (int i = 0; i < m_all.size(); i++)
            {
                WrappedColumn wc = (WrappedColumn)m_all.get(i);

                if (wc.getTableColumn() == column)
                {
                    return i;
                }
            }

            return -1;
        }

        public boolean isColumnVisible(TableColumn column)
        {
            int           i  = findWrappedColumn(column);
            WrappedColumn wc = (i >= 0) ? (WrappedColumn)m_all.get(i) : null;

            return (wc != null) ? wc.isVisible() : false;
        }

        public TableColumn getTableColumnExt(Object identifier)
        {
            for (int i = 0; i < m_all.size(); i++)
            {
                WrappedColumn wc = (WrappedColumn)m_all.get(i);

                if (wc.getTableColumn().getIdentifier() == identifier)
                {
                    return wc.getTableColumn();
                }
            }

            return null;
        }

        public boolean isDefaulted()
        {
            for (int i = 0; i < m_all.size(); i++)
            {
                WrappedColumn wc = (WrappedColumn)m_all.get(i);

                if (!wc.isVisible() || (wc.getInitialPosition() != i))
                {
                    return false;
                }
            }

            return true;
        }

        private String debug()
        {
            StringBuffer sb = new StringBuffer();

            for (int i = 0; i < m_all.size(); i++)
            {
                WrappedColumn wc = (WrappedColumn)m_all.get(i);

                if (i > 0)
                {
                    sb.append(", ");
                }

                if (!wc.isVisible())
                {
                    sb.append("*");
                }

                sb.append(wc.getTableColumn().getHeaderValue());
                sb.append("(").append(wc.getInitialPosition()).append(")");
            }

            return sb.toString();
        }

    }

    class WrappedColumn
    {
        private TableColumn m_column;
        private boolean     m_visible;
        private int         m_initialPosition = -1;

        public WrappedColumn(TableColumn column)
        {
            this(column, true);
        }

        public WrappedColumn(TableColumn column, boolean visible)
        {
            m_column  = column;
            m_visible = visible;
        }

        public TableColumn getTableColumn()
        {
            return m_column;
        }

        public void setVisible(boolean visible)
        {
            m_visible = visible;
        }

        public boolean isVisible()
        {
            return m_visible;
        }

        public int getInitialPosition()
        {
            return m_initialPosition;
        }

        public void setInitialPosition(int initialPosition)
        {
            if (m_initialPosition == -1)
            {
                m_initialPosition = initialPosition;
            }
        }
    }

    //-------------------------------------------------------------------------
    //
    // Deprecated behavior - do not use!
    //
    //-------------------------------------------------------------------------

    /**
     * Construct a ModelTable to hold objects of the specified row-class, and
     * with the specified TableColumnAdapters, and with no initial contents.
     *
     * @deprecated  As of MA v2.1 - please use other constructors.
     */
    public ModelTable(Class rowClass, TableColumnAdapter[] columnAdapters)
    {
        this(columnAdapters);
    }

    /**
     *  Construct a ModelTable with no ModelListTableModel.  It is necessary
     *  to call setModelListTableModel(ModelListTableModel) later to set the
     *  model.
     *
     * @deprecated  As of MA v2.1 - please use other constructors.
     */
    public ModelTable(TableColumnAdapter[] columnAdapters,
                      List                 initialContents)
    {
        this(new ModelListTableModel(columnAdapters, initialContents));
    }

    /**
     * Construct a ModelTable to hold objects of the specified row-class, and
     * with the specified TableColumnAdapters, and with the specified initial
     * contents.
     *
     * @deprecated  As of MA v2.1 - please use other constructors.
     */
    public ModelTable(Class                rowClass,
                      TableColumnAdapter[] columnAdapters,
                      List                 initialContents)
    {
        this(new ModelListTableModel(columnAdapters, initialContents));
    }

    /**
     * @deprecated  As of MA v2.1 - please use the regular getModel().
     */
    public ModelListTableModel getModelListTableModel()
    {
          return (ModelListTableModel)getModel();
    }

}
