/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */

package modelobjects.util.swing;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.util.EventObject;

import javax.swing.AbstractAction;
import javax.swing.JTable;
import javax.swing.JTextPane;
import javax.swing.SwingUtilities;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.table.TableCellEditor;
import javax.swing.text.Document;
import javax.swing.text.StyledEditorKit;


public class MultilineTextCellEditor extends AbstractCellEditor
  implements TableCellEditor
{
  public MultilineTextCellEditor()
  {
    textPane = new CellEditorJTextPane();
    textPane.setEditorKit(new StyledEditorKit());
    docSizeWatcher = new DocSizeWatcher(textPane);
  }

  public JTextPane getJTextPane()
  {
    return(textPane);
  }

  @Override
public Object getCellEditorValue()
  {
    return(textPane.getText());
  }

  @Override
public Component getTableCellEditorComponent
    (JTable table, Object value, boolean isSelected, int row, int column)
  {
    this.table = table;
    textPane.setText((value == null) ? "" : (String)value);
    Document doc = textPane.getDocument();
    textPane.setDocument(doc);
    docSizeWatcher.setDocument(doc);

    // request keyboard focus to the text pane when we start editing
    SwingUtilities.invokeLater(new Runnable() {
      @Override
    public void run() {
        textPane.requestFocus();
        textPane.repaint();
      }
    });

    return(textPane);
  }

  public int getClickCountToStart()
  {
    return(clickCountToStart);
  }

  public void setClickCountToStart(int clickCount)
  {
    clickCountToStart = clickCount;
  }

  @Override
public boolean isCellEditable(EventObject anEvent)
  {
    if (anEvent instanceof MouseEvent) { 
      return ((MouseEvent)anEvent).getClickCount() >= clickCountToStart;
    }
    return true;
  }

  void adjustRowHeight()
  {
    if ((table != null) && table.isEditing()) {
      int row = table.getEditingRow();
      int currentRowHeight = table.getRowHeight(row);
      int preferredHeight  = Math.max(textPane.getPreferredSize().height,
                                      table.getRowHeight());

      if (Math.abs(currentRowHeight - preferredHeight) > 2) {
        table.setRowHeight(row, preferredHeight);
      }
    }
  }

  /**
   *  DocSizeWatcher watches for changes in edited documents, and calls
   *  a routine (after the event has been processed!) that will adjust the
   *  current row height to match the contents.
   */
  class DocSizeWatcher implements DocumentListener, Runnable
  {
    private JTextPane textPane;
    private Document  doc;

    DocSizeWatcher(JTextPane textPane) {
      this.textPane = textPane;
    }

    public void setDocument(Document doc) {
      if (this.doc != null)
    {
        this.doc.removeDocumentListener(this);
    }
      this.doc = doc;
      if (this.doc != null)
    {
        this.doc.addDocumentListener(this);
    }
    }

    @Override
    public void run() {
      adjustRowHeight();
    }
    @Override
    public void changedUpdate(DocumentEvent e) {
      SwingUtilities.invokeLater(this);
    }
    @Override
    public void removeUpdate(DocumentEvent e) {
      SwingUtilities.invokeLater(this);
    }
    @Override
    public void insertUpdate(DocumentEvent e) {
      SwingUtilities.invokeLater(this);
    }
  }

  /**
   *  This edit action actually causes the cell editor to finish editing
   *  and apply its changes.  It is bound to Shift-Enter.
   */
  class FinishEditingAction extends AbstractAction
  {
    @Override
    public void actionPerformed(ActionEvent event)
    {
      TableCellEditor editor = table.getCellEditor();
      if (editor != null) {
        editor.stopCellEditing();
      }
    }
  }                                 

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///   private representation
  ///
  /////////////////////////////////////////////////////////////////////////////

  protected JTextPane                   textPane;
  protected JTable                      table;
  protected int                         clickCountToStart = 2;
  protected DocSizeWatcher              docSizeWatcher;


  /////////////////////////////////////////////////////////////////////////////
  ///
  ///   inner classes
  ///
  /////////////////////////////////////////////////////////////////////////////

  /**
   *  CellEditorJTextPane allows normal tabbing around in the table.
   */
  protected class CellEditorJTextPane extends JTextPane
  {
    @Override
    public boolean isManagingFocus() {
      return(true);
    }

    @Override
    protected void processKeyEvent(KeyEvent e) {
      if (e.getKeyCode() == KeyEvent.VK_TAB) {
        if (!stopCellEditing())
        {
            cancelCellEditing();
        }
        table.dispatchEvent(e);
        e.consume();
      }
      if (!e.isConsumed())
    {
        super.processKeyEvent(e);
    }
    }

    @Override
    protected void processMouseMotionEvent(MouseEvent event) {
      if (isShowing())
    {
        super.processMouseMotionEvent(event);
    }
    }
  }
  
}
